<?php
/**
 * Real-time Log Viewer for MonstaFTP
 * This script displays the most recent log entries
 */

require_once(dirname(__FILE__) . '/../application/api/constants.php');
includeMonstaConfig();

$logFile = dirname(__FILE__) . '/../application/logs/mftp_debug.log';

// Handle AJAX requests for real-time updates
if (isset($_GET['ajax']) && $_GET['ajax'] == '1') {
    header('Content-Type: application/json');
    
    if (!file_exists($logFile)) {
        echo json_encode(['exists' => false, 'content' => '']);
        exit;
    }
    
    $lines = isset($_GET['lines']) ? intval($_GET['lines']) : 100;
    $content = shell_exec("tail -n $lines " . escapeshellarg($logFile));
    
    echo json_encode([
        'exists' => true,
        'content' => $content,
        'size' => filesize($logFile),
        'modified' => filemtime($logFile)
    ]);
    exit;
}

// Handle log clearing
if (isset($_POST['action']) && $_POST['action'] == 'clear') {
    if (file_exists($logFile)) {
        file_put_contents($logFile, '');
    }
    header('Location: view-logs.php');
    exit;
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>MonstaFTP Log Viewer</title>
    <style>
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'Consolas', 'Monaco', 'Courier New', monospace;
            background: #1e1e1e;
            color: #d4d4d4;
            padding: 20px;
        }
        
        .header {
            background: #2d2d30;
            padding: 15px 20px;
            border-radius: 5px 5px 0 0;
            border-bottom: 2px solid #007acc;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .header h1 {
            font-size: 18px;
            color: #007acc;
            margin: 0;
        }
        
        .controls {
            display: flex;
            gap: 10px;
        }
        
        .btn {
            background: #007acc;
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 3px;
            cursor: pointer;
            font-size: 12px;
            font-family: inherit;
        }
        
        .btn:hover {
            background: #005a9e;
        }
        
        .btn-danger {
            background: #e51400;
        }
        
        .btn-danger:hover {
            background: #b01000;
        }
        
        .info-bar {
            background: #252526;
            padding: 10px 20px;
            font-size: 12px;
            color: #858585;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .status {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .status-indicator {
            width: 10px;
            height: 10px;
            border-radius: 50%;
            background: #4ec9b0;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        
        .log-container {
            background: #1e1e1e;
            border: 1px solid #3e3e42;
            border-top: none;
            border-radius: 0 0 5px 5px;
            height: calc(100vh - 180px);
            overflow-y: auto;
            padding: 15px;
        }
        
        .log-line {
            padding: 4px 0;
            font-size: 13px;
            line-height: 1.4;
            white-space: pre-wrap;
            word-wrap: break-word;
        }
        
        .log-line.debug {
            color: #4ec9b0;
        }
        
        .log-line.info {
            color: #4fc1ff;
        }
        
        .log-line.warning {
            color: #dcdcaa;
        }
        
        .log-line.error {
            color: #f48771;
            font-weight: bold;
        }
        
        .log-line.chunkedupload {
            background: rgba(78, 201, 176, 0.1);
        }
        
        .empty-state {
            text-align: center;
            padding: 50px 20px;
            color: #858585;
        }
        
        .empty-state h2 {
            font-size: 16px;
            margin-bottom: 10px;
        }
        
        select {
            background: #3c3c3c;
            color: #d4d4d4;
            border: 1px solid #3e3e42;
            padding: 6px 12px;
            border-radius: 3px;
            font-family: inherit;
            font-size: 12px;
        }
        
        label {
            font-size: 12px;
            color: #858585;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>📋 MonstaFTP Debug Log Viewer</h1>
        <div class="controls">
            <label for="lines">Show:</label>
            <select id="lines" onchange="updateLines()">
                <option value="50">50 lines</option>
                <option value="100" selected>100 lines</option>
                <option value="200">200 lines</option>
                <option value="500">500 lines</option>
                <option value="1000">1000 lines</option>
            </select>
            <button class="btn" onclick="refreshLogs()">🔄 Refresh</button>
            <form method="POST" style="display: inline;">
                <input type="hidden" name="action" value="clear">
                <button type="submit" class="btn btn-danger" onclick="return confirm('Clear all logs?')">🗑️ Clear</button>
            </form>
        </div>
    </div>
    
    <div class="info-bar">
        <div class="status">
            <span class="status-indicator"></span>
            <span>Auto-refresh every 2 seconds</span>
        </div>
        <div id="file-info">Loading...</div>
    </div>
    
    <div class="log-container" id="log-container">
        <div class="empty-state">
            <h2>Loading logs...</h2>
            <p>Please wait while we fetch the latest log entries.</p>
        </div>
    </div>
    
    <script>
        let currentLines = 100;
        let autoRefresh = true;
        let refreshInterval;
        
        function updateLines() {
            currentLines = parseInt(document.getElementById('lines').value);
            refreshLogs();
        }
        
        function formatLogLine(line) {
            // Detect log level
            let className = 'log-line';
            
            if (line.match(/ERROR/i)) {
                className += ' error';
            } else if (line.match(/WARNING/i)) {
                className += ' warning';
            } else if (line.match(/INFO/i)) {
                className += ' info';
            } else if (line.match(/DEBUG/i)) {
                className += ' debug';
            }
            
            // Highlight ChunkedUpload messages
            if (line.match(/ChunkedUpload/i)) {
                className += ' chunkedupload';
            }
            
            return '<div class="' + className + '">' + escapeHtml(line) + '</div>';
        }
        
        function escapeHtml(text) {
            const map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return text.replace(/[&<>"']/g, function(m) { return map[m]; });
        }
        
        function refreshLogs() {
            fetch('view-logs.php?ajax=1&lines=' + currentLines)
                .then(response => response.json())
                .then(data => {
                    const container = document.getElementById('log-container');
                    const fileInfo = document.getElementById('file-info');
                    
                    if (!data.exists) {
                        container.innerHTML = '<div class="empty-state"><h2>Log file not found</h2><p>The log file doesn\'t exist yet. Try uploading a file to generate logs.</p></div>';
                        fileInfo.textContent = 'Log file not found';
                        return;
                    }
                    
                    if (!data.content || data.content.trim() === '') {
                        container.innerHTML = '<div class="empty-state"><h2>No logs yet</h2><p>The log file is empty. Try uploading a file to see logs.</p></div>';
                        fileInfo.textContent = 'Size: 0 bytes';
                        return;
                    }
                    
                    // Format logs
                    const lines = data.content.split('\n').filter(line => line.trim() !== '');
                    const formatted = lines.map(line => formatLogLine(line)).join('');
                    container.innerHTML = formatted;
                    
                    // Update file info
                    const size = data.size < 1024 ? data.size + ' bytes' : 
                                 data.size < 1048576 ? (data.size / 1024).toFixed(2) + ' KB' :
                                 (data.size / 1048576).toFixed(2) + ' MB';
                    const modified = new Date(data.modified * 1000).toLocaleString();
                    fileInfo.textContent = 'Size: ' + size + ' | Modified: ' + modified;
                    
                    // Auto-scroll to bottom
                    container.scrollTop = container.scrollHeight;
                })
                .catch(error => {
                    console.error('Error fetching logs:', error);
                });
        }
        
        // Initial load
        refreshLogs();
        
        // Auto-refresh every 2 seconds
        refreshInterval = setInterval(refreshLogs, 2000);
    </script>
</body>
</html>


