<?php
/**
 * Upload Diagnostics Tool for MonstaFTP
 * This script helps diagnose upload issues
 */

require_once(dirname(__FILE__) . '/../application/api/constants.php');
includeMonstaConfig();
require_once(dirname(__FILE__) . '/../application/api/system/SecureSessionManager.php');

// Initialize session
$sessionManager = SecureSessionManager::getInstance();
$sessionManager->initializeSession();

echo "<!DOCTYPE html>\n";
echo "<html><head><title>Upload Diagnostics</title>";
echo "<style>
    body { font-family: monospace; padding: 20px; background: #f5f5f5; }
    .section { background: white; padding: 15px; margin: 10px 0; border-radius: 5px; border-left: 4px solid #007bff; }
    .error { border-left-color: #dc3545; }
    .success { border-left-color: #28a745; }
    .warning { border-left-color: #ffc107; }
    h2 { margin-top: 0; color: #333; }
    pre { background: #f8f9fa; padding: 10px; overflow-x: auto; }
    table { width: 100%; border-collapse: collapse; }
    td, th { padding: 8px; text-align: left; border-bottom: 1px solid #ddd; }
    th { background: #007bff; color: white; }
</style>
</head><body>";

echo "<h1>MonstaFTP Upload Diagnostics</h1>";

// Check PHP Configuration
echo "<div class='section'>";
echo "<h2>PHP Configuration</h2>";
echo "<table>";
echo "<tr><th>Setting</th><th>Value</th><th>Status</th></tr>";

$checks = [
    'upload_max_filesize' => ['value' => ini_get('upload_max_filesize'), 'min' => '8M'],
    'post_max_size' => ['value' => ini_get('post_max_size'), 'min' => '8M'],
    'max_execution_time' => ['value' => ini_get('max_execution_time'), 'min' => '60'],
    'memory_limit' => ['value' => ini_get('memory_limit'), 'min' => '128M'],
    'max_input_time' => ['value' => ini_get('max_input_time'), 'min' => '60'],
    'session.gc_maxlifetime' => ['value' => ini_get('session.gc_maxlifetime'), 'min' => '1440'],
];

foreach ($checks as $setting => $info) {
    $status = '✓ OK';
    $class = 'success';
    echo "<tr><td>$setting</td><td>{$info['value']}</td><td>$status</td></tr>";
}
echo "</table>";
echo "</div>";

// Check Temp Directory
echo "<div class='section'>";
echo "<h2>Temporary Directory</h2>";
$tempDir = sys_get_temp_dir();
echo "<table>";
echo "<tr><th>Property</th><th>Value</th></tr>";
echo "<tr><td>Path</td><td>$tempDir</td></tr>";
echo "<tr><td>Exists</td><td>" . (is_dir($tempDir) ? '✓ Yes' : '✗ No') . "</td></tr>";
echo "<tr><td>Writable</td><td>" . (is_writable($tempDir) ? '✓ Yes' : '✗ No') . "</td></tr>";
echo "<tr><td>Free Space</td><td>" . round(disk_free_space($tempDir) / 1024 / 1024 / 1024, 2) . " GB</td></tr>";
echo "</table>";

// Check for temp upload files
$tempFiles = glob($tempDir . '/mftp_upload_*');
$uploadFiles = glob($tempDir . '/mftp_chunked_*');
echo "<p><strong>Active Temp Files:</strong></p>";
echo "<ul>";
echo "<li>mftp_upload_* files: " . count($tempFiles) . "</li>";
echo "<li>mftp_chunked_* files: " . count($uploadFiles) . "</li>";
echo "</ul>";
echo "</div>";

// Check Session
echo "<div class='section'>";
echo "<h2>Session Information</h2>";
echo "<table>";
echo "<tr><th>Property</th><th>Value</th></tr>";
echo "<tr><td>Session ID</td><td>" . session_id() . "</td></tr>";
echo "<tr><td>Session Started</td><td>" . (session_status() === PHP_SESSION_ACTIVE ? '✓ Yes' : '✗ No') . "</td></tr>";

// Count upload keys in session
$uploadKeys = 0;
$uploadKeysList = [];
foreach ($_SESSION as $key => $value) {
    if (strpos($key, 'UPLOAD_KEY_') === 0 && !strpos($key, '_meta')) {
        $uploadKeys++;
        $uploadKeysList[] = $key;
    }
}
echo "<tr><td>Active Upload Keys</td><td>$uploadKeys</td></tr>";
echo "</table>";

if ($uploadKeys > 0) {
    echo "<p><strong>Upload Keys:</strong></p>";
    echo "<ul>";
    foreach ($uploadKeysList as $key) {
        $meta = isset($_SESSION[$key . '_meta']) ? $_SESSION[$key . '_meta'] : null;
        $status = $meta ? ($meta['status'] ?? 'unknown') : 'no metadata';
        echo "<li>$key - Status: $status</li>";
    }
    echo "</ul>";
}
echo "</div>";

// Check FTP Extensions
echo "<div class='section'>";
echo "<h2>PHP Extensions</h2>";
echo "<table>";
echo "<tr><th>Extension</th><th>Status</th></tr>";

$extensions = ['ftp', 'openssl', 'curl', 'mbstring', 'zip'];
foreach ($extensions as $ext) {
    $loaded = extension_loaded($ext);
    $status = $loaded ? '✓ Loaded' : '✗ Not Loaded';
    $class = $loaded ? 'success' : 'error';
    echo "<tr><td>$ext</td><td>$status</td></tr>";
}
echo "</table>";
echo "</div>";

// Check Constants
echo "<div class='section'>";
echo "<h2>Upload Constants</h2>";
echo "<table>";
echo "<tr><th>Constant</th><th>Value</th></tr>";

$constants = [
    'MFTP_MAX_CONCURRENT_UPLOADS',
    'MFTP_UPLOAD_SESSION_ACTIVITY_UPDATE',
    'MFTP_UPLOAD_RETRY_ATTEMPTS',
    'MFTP_UPLOAD_RETRY_DELAY'
];

foreach ($constants as $const) {
    $value = defined($const) ? constant($const) : 'Not defined';
    echo "<tr><td>$const</td><td>" . htmlspecialchars(print_r($value, true)) . "</td></tr>";
}
echo "</table>";
echo "</div>";

// Recent PHP Errors
echo "<div class='section'>";
echo "<h2>Recent Errors</h2>";
$lastError = error_get_last();
if ($lastError) {
    echo "<pre>";
    print_r($lastError);
    echo "</pre>";
} else {
    echo "<p>No recent errors</p>";
}
echo "</div>";

// Cleanup Test
echo "<div class='section'>";
echo "<h2>Upload Test</h2>";
echo "<p>Testing if we can create and write to a temp file...</p>";

$testFile = tempnam(sys_get_temp_dir(), 'mftp_test_');
if ($testFile === false) {
    echo "<p class='error'>✗ Failed to create temp file</p>";
} else {
    $testData = "Test upload data at " . date('Y-m-d H:i:s');
    $written = file_put_contents($testFile, $testData);
    if ($written === false) {
        echo "<p class='error'>✗ Failed to write to temp file</p>";
    } else {
        $readBack = file_get_contents($testFile);
        if ($readBack === $testData) {
            echo "<p class='success'>✓ Successfully created, wrote, and read temp file</p>";
            echo "<p>File: $testFile</p>";
            echo "<p>Size: " . filesize($testFile) . " bytes</p>";
        } else {
            echo "<p class='error'>✗ Data mismatch when reading back</p>";
        }
        @unlink($testFile);
    }
}
echo "</div>";

echo "</body></html>";
?>


