<?php
/**
 * Test if WAF/ModSecurity is blocking .php in POST requests
 */

// Simple test endpoint
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    
    $testData = $_POST['test_data'] ?? 'none';
    
    echo json_encode([
        'success' => true,
        'message' => 'Request received successfully',
        'test_data' => $testData,
        'post_keys' => array_keys($_POST),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    exit;
}
?>
<!DOCTYPE html>
<html>
<head>
    <title>WAF Test</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <script>
        async function testRequest(data) {
            const formData = new FormData();
            formData.append('test_data', data);
            
            try {
                const response = await fetch('test-waf.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                return { success: true, status: response.status, data: result };
            } catch (error) {
                return { success: false, status: 'error', error: error.message };
            }
        }
        
        async function runTests() {
            const resultsDiv = document.getElementById('results');
            resultsDiv.innerHTML = '<p>Running tests...</p>';
            
            const tests = [
                { name: 'Normal text', data: 'hello world' },
                { name: 'File path without .php', data: '/path/to/file.txt' },
                { name: 'File path with .php', data: '/path/to/file.php' },
                { name: 'Rename with .php extension', data: '{"destination":"/test.php"}' },
                { name: 'Phar to PHP', data: '{"source":"/about.phar","destination":"/about.phar.php"}' }
            ];
            
            let html = '<table class="table table-bordered"><thead><tr><th>Test</th><th>Status</th><th>Response</th></tr></thead><tbody>';
            
            for (const test of tests) {
                const result = await testRequest(test.data);
                const statusClass = result.success ? 'table-success' : 'table-danger';
                const statusText = result.success ? '✓ OK (Status ' + result.status + ')' : '✗ BLOCKED';
                
                html += `<tr class="${statusClass}">
                    <td><strong>${test.name}</strong><br><small>${test.data}</small></td>
                    <td>${statusText}</td>
                    <td><pre>${JSON.stringify(result, null, 2)}</pre></td>
                </tr>`;
            }
            
            html += '</tbody></table>';
            resultsDiv.innerHTML = html;
        }
    </script>
</head>
<body class="bg-light">
    <div class="container mt-5">
        <h1 class="mb-4">🛡️ WAF / ModSecurity Test</h1>
        
        <div class="alert alert-info">
            <strong>What this tests:</strong> This tool checks if your web server is blocking POST requests that contain <code>.php</code> in the data.
        </div>
        
        <button onclick="runTests()" class="btn btn-primary btn-lg mb-4">Run Tests</button>
        
        <div id="results"></div>
        
        <div class="mt-4">
            <h5>Expected Results:</h5>
            <ul>
                <li><strong>If WAF is blocking:</strong> Tests with ".php" will fail with no response or 500 error</li>
                <li><strong>If WAF is not blocking:</strong> All tests should pass with status 200</li>
            </ul>
        </div>
        
        <div class="mt-4">
            <a href="upload-diagnostics.php" class="btn btn-secondary">📊 Diagnostics</a>
            <a href="view-logs.php" class="btn btn-secondary">📋 View Logs</a>
        </div>
    </div>
</body>
</html>


