<?php
/**
 * Test Upload Session Tool
 * This script helps test if session keys are working properly
 */

require_once(dirname(__FILE__) . '/../application/api/constants.php');
includeMonstaConfig();
require_once(dirname(__FILE__) . '/../application/api/system/SecureSessionManager.php');

// Initialize session
$sessionManager = SecureSessionManager::getInstance();
$sessionManager->initializeSession();

// Test upload ID (you can change this to match a failed upload ID from logs)
$testUploadId = 'TEST_' . bin2hex(random_bytes(8));

echo "<!DOCTYPE html>\n";
echo "<html><head><title>Upload Session Test</title>";
echo "<link href='https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css' rel='stylesheet'>";
echo "</head><body class='bg-light'>";
echo "<div class='container mt-5'>";
echo "<h1 class='mb-4'>Upload Session Test</h1>";

// Test 1: Session Creation
echo "<div class='card mb-3'>";
echo "<div class='card-header bg-primary text-white'><strong>Test 1: Session Creation</strong></div>";
echo "<div class='card-body'>";
echo "<p><strong>Session ID:</strong> " . session_id() . "</p>";
echo "<p><strong>Session Status:</strong> " . (session_status() === PHP_SESSION_ACTIVE ? '<span class="badge bg-success">Active</span>' : '<span class="badge bg-danger">Inactive</span>') . "</p>";
echo "</div></div>";

// Test 2: Session Key Generation
echo "<div class='card mb-3'>";
echo "<div class='card-header bg-primary text-white'><strong>Test 2: Session Key Generation</strong></div>";
echo "<div class='card-body'>";

require_once(dirname(__FILE__) . '/../application/api/upload-chunked.php');

$sessionKey = mftpGenerateChunkedSessionKey($testUploadId);
echo "<p><strong>Test Upload ID:</strong> <code>" . htmlspecialchars($testUploadId) . "</code></p>";
echo "<p><strong>Generated Session Key:</strong> <code>" . htmlspecialchars($sessionKey) . "</code></p>";

// Store test data in session
$testTempFile = sys_get_temp_dir() . '/mftp_test_' . $testUploadId;
touch($testTempFile);
$_SESSION[$sessionKey] = $testTempFile;
$_SESSION[$sessionKey . '_meta'] = [
    'created_at' => time(),
    'upload_id' => $testUploadId,
    'status' => 'test'
];

echo "<p><strong>Stored in Session:</strong> <span class='badge bg-success'>✓</span></p>";
echo "</div></div>";

// Test 3: Session Key Retrieval
echo "<div class='card mb-3'>";
echo "<div class='card-header bg-primary text-white'><strong>Test 3: Session Key Retrieval</strong></div>";
echo "<div class='card-body'>";

try {
    $retrievedPath = getChunkedTempPathFromSession($testUploadId);
    if ($retrievedPath === $testTempFile) {
        echo "<p><strong>Status:</strong> <span class='badge bg-success'>✓ SUCCESS</span></p>";
        echo "<p>Successfully retrieved the temp file path from session</p>";
        echo "<p><strong>Retrieved Path:</strong> <code>" . htmlspecialchars($retrievedPath) . "</code></p>";
    } else {
        echo "<p><strong>Status:</strong> <span class='badge bg-warning'>⚠ MISMATCH</span></p>";
        echo "<p>Retrieved path doesn't match stored path</p>";
        echo "<p><strong>Expected:</strong> <code>" . htmlspecialchars($testTempFile) . "</code></p>";
        echo "<p><strong>Got:</strong> <code>" . htmlspecialchars($retrievedPath) . "</code></p>";
    }
} catch (Exception $e) {
    echo "<p><strong>Status:</strong> <span class='badge bg-danger'>✗ ERROR</span></p>";
    echo "<p><strong>Error:</strong> " . htmlspecialchars($e->getMessage()) . "</p>";
}

echo "</div></div>";

// Test 4: Current Session Contents
echo "<div class='card mb-3'>";
echo "<div class='card-header bg-primary text-white'><strong>Test 4: Current Session Contents</strong></div>";
echo "<div class='card-body'>";
echo "<p><strong>All Session Keys:</strong></p>";
echo "<ul class='list-group'>";

$uploadCount = 0;
foreach ($_SESSION as $key => $value) {
    if (strpos($key, 'UPLOAD_KEY_') === 0) {
        $uploadCount++;
        $isMeta = strpos($key, '_meta') !== false;
        $displayValue = $isMeta ? json_encode($value, JSON_PRETTY_PRINT) : htmlspecialchars($value);
        echo "<li class='list-group-item'>";
        echo "<strong>" . htmlspecialchars($key) . "</strong><br>";
        echo "<small><code>" . $displayValue . "</code></small>";
        echo "</li>";
    }
}

if ($uploadCount === 0) {
    echo "<li class='list-group-item'>No upload keys found</li>";
}

echo "</ul>";
echo "<p class='mt-3'><strong>Total Upload Keys:</strong> " . $uploadCount . "</p>";
echo "</div></div>";

// Test 5: Temp Directory Access
echo "<div class='card mb-3'>";
echo "<div class='card-header bg-primary text-white'><strong>Test 5: Temp Directory Write Test</strong></div>";
echo "<div class='card-body'>";

$testFile = tempnam(sys_get_temp_dir(), 'mftp_write_test_');
if ($testFile === false) {
    echo "<p><strong>Status:</strong> <span class='badge bg-danger'>✗ FAILED</span></p>";
    echo "<p>Cannot create temp file in " . sys_get_temp_dir() . "</p>";
} else {
    $testData = "Test data: " . time();
    $written = file_put_contents($testFile, $testData);
    if ($written === false) {
        echo "<p><strong>Status:</strong> <span class='badge bg-danger'>✗ FAILED</span></p>";
        echo "<p>Cannot write to temp file</p>";
    } else {
        $readBack = file_get_contents($testFile);
        if ($readBack === $testData) {
            echo "<p><strong>Status:</strong> <span class='badge bg-success'>✓ SUCCESS</span></p>";
            echo "<p>Can successfully create, write, and read temp files</p>";
        } else {
            echo "<p><strong>Status:</strong> <span class='badge bg-warning'>⚠ WARNING</span></p>";
            echo "<p>Data mismatch when reading back</p>";
        }
    }
    @unlink($testFile);
}

echo "</div></div>";

// Cleanup
@unlink($testTempFile);
unset($_SESSION[$sessionKey]);
unset($_SESSION[$sessionKey . '_meta']);

echo "<div class='alert alert-info mt-4'>";
echo "<strong>Instructions:</strong><br>";
echo "1. If all tests pass, the session mechanism is working correctly<br>";
echo "2. If Test 3 fails, there's an issue with session key generation or storage<br>";
echo "3. If Test 5 fails, there's an issue with filesystem permissions<br>";
echo "4. Check the <a href='view-logs.php' target='_blank'>log viewer</a> for detailed error messages during actual uploads";
echo "</div>";

echo "</div></body></html>";
?>


