<?php
/**
 * Test Rename Operation
 * This script helps test renaming files to PHP extension
 */

require_once(dirname(__FILE__) . '/../application/api/constants.php');
includeMonstaConfig();
require_once(dirname(__FILE__) . '/../application/api/system/SecureSessionManager.php');
require_once(dirname(__FILE__) . '/../application/api/request_processor/RequestMarshaller.php');
require_once(dirname(__FILE__) . '/../application/api/lib/InputValidator.php');

// Initialize session
$sessionManager = SecureSessionManager::getInstance();
$sessionManager->initializeSession();

echo "<!DOCTYPE html>\n";
echo "<html><head><title>Test Rename</title>";
echo "<link href='https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css' rel='stylesheet'>";
echo "</head><body class='bg-light'>";
echo "<div class='container mt-5'>";
echo "<h1 class='mb-4'>Test Rename to .php</h1>";

// Handle test rename
if (isset($_POST['test_rename'])) {
    echo "<div class='card mb-4'>";
    echo "<div class='card-header bg-primary text-white'><strong>Rename Test Results</strong></div>";
    echo "<div class='card-body'>";
    
    try {
        $source = $_POST['source'] ?? '/test.txt';
        $destination = $_POST['destination'] ?? '/test.php';
        
        echo "<p><strong>Source:</strong> " . htmlspecialchars($source) . "</p>";
        echo "<p><strong>Destination:</strong> " . htmlspecialchars($destination) . "</p>";
        echo "<hr>";
        
        // Step 1: Validate paths
        echo "<h5>Step 1: Validating Paths</h5>";
        try {
            $validatedSource = InputValidator::validateFilePath($source, true);
            echo "<p class='text-success'>✓ Source path validated: " . htmlspecialchars($validatedSource) . "</p>";
        } catch (Exception $e) {
            echo "<p class='text-danger'>✗ Source validation failed: " . htmlspecialchars($e->getMessage()) . "</p>";
            throw $e;
        }
        
        try {
            $validatedDestination = InputValidator::validateFilePath($destination, true);
            echo "<p class='text-success'>✓ Destination path validated: " . htmlspecialchars($validatedDestination) . "</p>";
        } catch (Exception $e) {
            echo "<p class='text-danger'>✗ Destination validation failed: " . htmlspecialchars($e->getMessage()) . "</p>";
            throw $e;
        }
        
        echo "<hr>";
        echo "<h5>Step 2: Checking Extension</h5>";
        $destExtension = strtolower(pathinfo($destination, PATHINFO_EXTENSION));
        echo "<p>Destination extension: <code>$destExtension</code></p>";
        
        if ($destExtension === 'php') {
            echo "<p class='text-warning'>⚠ Destination is a PHP file</p>";
        }
        
        echo "<hr>";
        echo "<h5>Step 3: Connection Test</h5>";
        $request = [
            'connectionType' => 'FTP',
            'actionName' => 'rename',
            'configuration' => [
                // Add your FTP credentials here if you want to test actual rename
            ],
            'context' => [
                'source' => $validatedSource,
                'destination' => $validatedDestination
            ]
        ];
        
        echo "<p class='text-info'>✓ All validation checks passed!</p>";
        echo "<p class='text-success'><strong>No errors detected in validation layer</strong></p>";
        echo "<p>If rename still fails, the issue is at the FTP server level or in the connection layer.</p>";
        
    } catch (Exception $e) {
        echo "<div class='alert alert-danger'>";
        echo "<h5>Error Caught:</h5>";
        echo "<p><strong>Message:</strong> " . htmlspecialchars($e->getMessage()) . "</p>";
        echo "<p><strong>Type:</strong> " . get_class($e) . "</p>";
        echo "<p><strong>File:</strong> " . $e->getFile() . " (Line " . $e->getLine() . ")</p>";
        echo "<details><summary>Stack Trace</summary><pre>" . htmlspecialchars($e->getTraceAsString()) . "</pre></details>";
        echo "</div>";
    }
    
    echo "</div></div>";
}

// Test form
echo "<div class='card mb-4'>";
echo "<div class='card-header bg-info text-white'><strong>Test Rename Operation</strong></div>";
echo "<div class='card-body'>";
echo "<form method='POST'>";
echo "<div class='mb-3'>";
echo "<label class='form-label'>Source Path:</label>";
echo "<input type='text' name='source' class='form-control' value='/test.txt' placeholder='/path/to/file.txt'>";
echo "</div>";
echo "<div class='mb-3'>";
echo "<label class='form-label'>Destination Path:</label>";
echo "<input type='text' name='destination' class='form-control' value='/test.php' placeholder='/path/to/file.php'>";
echo "</div>";
echo "<button type='submit' name='test_rename' class='btn btn-primary'>Test Rename Validation</button>";
echo "</form>";
echo "</div></div>";

echo "<div class='alert alert-info'>";
echo "<strong>Note:</strong> This tests only the validation layer. To test actual FTP rename, you need to add your FTP credentials in the code.";
echo "</div>";

echo "<div class='mt-4'>";
echo "<a href='upload-diagnostics.php' class='btn btn-secondary'>📊 Diagnostics</a> ";
echo "<a href='view-logs.php' class='btn btn-secondary'>📋 View Logs</a>";
echo "</div>";

echo "</div></body></html>";
?>


