<?php
// SFTP Diagnostics Tool for Monsta FTP
// This tool helps diagnose SFTP connection issues

// =================================================================
// SECURITY CONFIGURATION - SET YOUR PASSWORD HERE
// =================================================================
// CHANGE THIS: Set your own secure password below
// Example: $diagnostic_password = "your_secure_password_here";
$diagnostic_password = ""; // LEAVE EMPTY INITIALLY - SET YOUR OWN PASSWORD

// =================================================================
// SFTP CONFIGURATION - SET THESE VALUES FOR TESTING
// =================================================================
$sftpHost = "";                   // SFTP host address
$sftpPort = 22;                   // Host port (normally 22)
$sftpUser = "";                   // SFTP username
$sftpPass = "";                   // SFTP password (leave blank if using key only)
$sftpPrivateKeyFile = "";         // Path to private key file (e.g. /home/user/.ssh/id_rsa)
$sftpPrivateKeyPassphrase = "";   // Passphrase for private key (if any)
$sftpDir = "";                    // Set a start folder (optional) i.e. /path/to/folder
$sftpTmp = "";                    // Set a temp folder (optional), i.e. /path/to/folder
$testFile = "mftp-test.html";     // File name used for creating test files


// =================================================================
// WORKFLOW STARTS HERE
// =================================================================
session_start();
$authorized = false;

// Check if password is configured
if (empty($diagnostic_password)) {
    echo "<!DOCTYPE html>\n";
    echo "<html>\n<head>\n";
    echo "<title>MonstaFTP SFTP Diagnostics - Password Required</title>\n";
    echo "<meta charset='utf-8'>\n";
    echo "<style>\n";
    echo "body { font-family: Arial, sans-serif; background: #f5f5f5; display: flex; justify-content: center; align-items: center; min-height: 100vh; margin: 0; }\n";
    echo ".message-container { background: white; padding: 40px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); max-width: 500px; width: 100%; text-align: center; }\n";
    echo ".message { color: #333; font-size: 18px; line-height: 1.5; }\n";
    echo "</style>\n";
    echo "</head>\n<body>\n";
    echo "<div class='message-container'>\n";
    echo "<div class='message'>Please edit this file to set a password to access this page.</div>\n";
    echo "</div>\n";
    echo "</body>\n</html>\n";
    exit;
}

if (isset($_POST['password']) && $_POST['password'] === $diagnostic_password) {
    $_SESSION['diagnostic_authorized'] = true;
    $authorized = true;
} elseif (isset($_SESSION['diagnostic_authorized']) && $_SESSION['diagnostic_authorized'] === true) {
    $authorized = true;
}

// Include composer autoloader for phpseclib3
require_once(dirname(__FILE__) . "/../application/api/vendor/autoload.php");

// Use the namespaced classes
use phpseclib3\Net\SFTP;
use phpseclib3\Crypt\RSA;

$sftpResults = array();
$sftpException = null;
$sftpRawList = "";
$phpInfo = array();
$serverInfo = array();
$sftpAuthMethod = "";

function collectSystemInfo() {
    global $phpInfo, $serverInfo;
    $phpInfo = array(
        'version' => PHP_VERSION,
        'os' => PHP_OS,
        'platform' => getPlatformName(),
        'memory_limit' => ini_get('memory_limit'),
        'max_execution_time' => ini_get('max_execution_time'),
        'upload_max_filesize' => ini_get('upload_max_filesize'),
        'post_max_size' => ini_get('post_max_size'),
        'max_input_time' => ini_get('max_input_time'),
        'file_uploads' => ini_get('file_uploads') ? 'Enabled' : 'Disabled',
        'extensions' => array(
            'openssl' => function_exists("openssl_get_publickey"),
            'mbstring' => extension_loaded('mbstring'),
            'json' => extension_loaded('json'),
            'suhosin' => extension_loaded('suhosin')
        )
    );
    $serverInfo = array(
        'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
        'php_sapi' => php_sapi_name(),
        'document_root' => $_SERVER['DOCUMENT_ROOT'] ?? 'Unknown',
        'script_filename' => $_SERVER['SCRIPT_FILENAME'] ?? 'Unknown'
    );
}

function getPlatformName() {
    $os = PHP_OS;
    if (strtoupper(substr($os, 0, 5)) == 'LINUX') return 'Linux';
    elseif (strtoupper(substr($os, 0, 7)) == 'FREEBSD') return 'FreeBSD';
    elseif (strtoupper(substr($os, 0, 3)) == 'WIN') return 'Windows';
    elseif (strtoupper(substr($os, 0, 6)) == 'DARWIN') return 'macOS';
    else return $os;
}

function createTempFileName($sftpTmp, $filename) {
    if ($sftpTmp == "") {
        $sftpTmp = ini_get('upload_tmp_dir') ? ini_get('upload_tmp_dir') : sys_get_temp_dir();
    }
    return tempnam($sftpTmp, $filename);
}

function runSFTPTests() {
    global $sftpHost, $sftpPort, $sftpUser, $sftpPass, $sftpPrivateKeyFile, $sftpPrivateKeyPassphrase, $sftpDir, $sftpTmp, $testFile;
    global $sftpResults, $sftpException, $sftpRawList, $sftpAuthMethod;
    
    if ($sftpHost == "" || $sftpUser == "") {
        $sftpResults["SFTP Credentials"] = false;
        return;
    }
    
    // Test local file operations
    $testName = "Can create file on client server";
    $filename = createTempFileName($sftpTmp, $testFile);
    $sftpResults[$testName] = @file_put_contents($filename, "Hello World!") !== false;

    $testName = "Can delete file on client server";
    $sftpResults[$testName] = @unlink($filename);

    try {
        $testName = "Can connect to SFTP server";
        
        // Parse the host if it's a URL format
        $parsedHost = $sftpHost;
        if (strpos($sftpHost, '://') !== false) {
            $urlParts = parse_url($sftpHost);
            $parsedHost = $urlParts['host'] ?? $sftpHost;
            $sftpResults["Host parsing"] = "✓ Original: $sftpHost → Parsed: $parsedHost";
        }
        
        // Escape IPv6 addresses (same as main app)
        function addressIsIpV6($address) {
            return filter_var($address, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6) !== false;
        }
        function escapeIpAddress($address) {
            return addressIsIpV6($address) ? "[$address]" : $address;
        }
        
        $escapedHost = escapeIpAddress($parsedHost);
        $sftpResults["Connection target"] = "Host: $escapedHost, Port: $sftpPort";
        
        // Create SFTP connection (same pattern as main Monsta app)
        $sftp = new SFTP($escapedHost, $sftpPort, 10);
        
        // Test connection by attempting authentication (same as main app)
        $testName = "Can authenticate to SFTP server";
        $authResult = false;
        $sftpAuthMethod = "Password";
        
        // Try different authentication methods
        if ($sftpPrivateKeyFile && file_exists($sftpPrivateKeyFile)) {
            // Try private key authentication
            try {
                $keyData = file_get_contents($sftpPrivateKeyFile);
                if ($sftpPrivateKeyPassphrase) {
                    $key = RSA::load($keyData, $sftpPrivateKeyPassphrase);
                } else {
                    $key = RSA::load($keyData);
                }
                $sftpAuthMethod = $sftpPrivateKeyPassphrase ? "Private Key (passphrase)" : "Private Key";
                $authResult = $sftp->login($sftpUser, $key);
            } catch (Exception $e) {
                $sftpResults["Private key auth error"] = "⚠️ " . $e->getMessage();
            }
        }
        
        // If private key failed or not available, try password
        if (!$authResult && $sftpPass !== "") {
            try {
                $authResult = $sftp->login($sftpUser, $sftpPass);
                $sftpAuthMethod = "Password";
            } catch (Exception $e) {
                $sftpResults["Password auth error"] = "⚠️ " . $e->getMessage();
            }
        }
        
        // Now check if we're connected and authenticated
        $connectionStatus = $sftp->isConnected();
        $sftpResults["Can connect to SFTP server"] = $connectionStatus;
        $sftpResults[$testName . " (" . $sftpAuthMethod . ")"] = $authResult;
        
        if (!$connectionStatus) {
            // Connection failed - get detailed error info
            $errors = $sftp->getErrors();
            $lastError = $sftp->getLastError();
            
            if (!empty($errors)) {
                $sftpResults["Connection errors"] = "⚠️ " . implode('; ', $errors);
            }
            if (!empty($lastError)) {
                $sftpResults["Last error"] = "⚠️ " . $lastError;
            }
            
            // Test basic TCP connectivity
            $socket = @fsockopen($escapedHost, $sftpPort, $errno, $errstr, 5);
            if ($socket) {
                fclose($socket);
                $sftpResults["TCP connectivity"] = "✓ Can reach $escapedHost:$sftpPort";
                $sftpResults["Diagnosis"] = "⚠️ TCP works but SFTP handshake failed - check SSH/SFTP service";
            } else {
                $sftpResults["TCP connectivity"] = "❌ Cannot reach $escapedHost:$sftpPort - $errstr ($errno)";
                $sftpResults["Diagnosis"] = "❌ Network connectivity issue - check host/port/firewall";
            }
            
            throw new Exception("Unable to connect to SFTP server");
        } elseif (!$authResult) {
            $sftpResults["Connection status"] = "✓ Connected but authentication failed";
            throw new Exception("Unable to authenticate to SFTP server using $sftpAuthMethod");
        } else {
            $sftpResults["Connection status"] = "✓ Successfully connected and authenticated";
        }

        // Authentication was successful, continue with other tests...

        if ($sftpDir != "") {
            $testName = "Can change folder on SFTP server";
            $sftpResults[$testName] = $sftp->chdir($sftpDir);
            if (!$sftpResults[$testName]) throw new Exception("Unable to change directory to $sftpDir");
        }

        $testName = "Can upload a file to SFTP server";
        $filename = createTempFileName($sftpTmp, $testFile);
        file_put_contents($filename, "Hello World!");
        $sftpResults[$testName] = $sftp->put($testFile, $filename, SFTP::SOURCE_LOCAL_FILE);
        @unlink($filename);
        if (!$sftpResults[$testName]) throw new Exception("Unable to upload file to SFTP server.");

        $testName = "Can delete a file from SFTP server";
        $sftpResults[$testName] = $sftp->delete($testFile);
        if (!$sftpResults[$testName]) throw new Exception("Unable to delete file from SFTP server.");

        $testName = "Can list files in SFTP directory";
        $list = $sftp->rawlist();
        $sftpResults[$testName] = is_array($list);
        $sftpRawList = print_r($list, true);
    } catch (Exception $e) {
        $sftpException = $e->getMessage();
        $sftpResults[$testName] = false;
    }
}

collectSystemInfo();
if ($authorized && ($sftpHost != "" || $sftpUser != "")) {
    runSFTPTests();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SFTP Diagnostics - Monsta FTP</title>
    <link rel="shortcut icon" type="image/x-icon" href="../application/frontend/images/logo-favicon.png">
    <link rel="apple-touch-icon" href="../application/frontend/images/logo-webclip.png">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://fonts.googleapis.com/css2?family=Open+Sans:wght@300;400;600;700&display=swap">
    <style>
        body { font-family: 'Open Sans', sans-serif; background-color: #f8f9fa; }
        .card { border: none; box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075); border-radius: 0.5rem; }
        .card-header { background-color: #fff; border-bottom: 1px solid #dee2e6; font-weight: 600; }
        .status-pass { color: #198754; font-weight: 600; }
        .status-fail { color: #dc3545; font-weight: 600; }
        .copy-btn { cursor: pointer; color: #6c757d; }
        .copy-btn:hover { color: #495057; }
        .diagnostic-section { margin-bottom: 2rem; }
        .test-result { padding: 0.5rem 0; border-bottom: 1px solid #f1f3f4; }
        .test-result:last-child { border-bottom: none; }
        
        /* Local Font Awesome CSS using local files */
        /* FontAwesome now loaded from CDN - local @font-face removed */
        .fa { font-family: 'FontAwesome'; }
        /* Add common Font Awesome icon classes */
        .fa-check:before { content: '\f00c'; }
        .fa-times:before { content: '\f00d'; }
        .fa-exclamation-triangle:before { content: '\f071'; }
        .fa-exclamation-circle:before { content: '\f06a'; }
        .fa-info-circle:before { content: '\f05a'; }
        .fa-copy:before { content: '\f0c5'; }
        .fa-server:before { content: '\f233'; }
        .fa-puzzle-piece:before { content: '\f12e'; }
        .fa-key:before { content: '\f084'; }
        .fa-network-wired:before { content: '\f6ff'; }
        .fa-question-circle:before { content: '\f059'; }
        .fa-tools:before { content: '\f7d9'; }
        .fa-lock:before { content: '\f023'; }
        .fa-sign-in-alt:before { content: '\f2f6'; }
    </style>
</head>
<body>
<div class="container-fluid py-4">
    <div class="row justify-content-center">
        <div class="col-lg-10 col-xl-8">
            <div class="text-center mb-4">
                <h1 class="h2 mb-2">
                    <i class="fa fa-tools text-primary"></i>
                    Monsta FTP SFTP Diagnostics Tool
                </h1>
                <p class="text-muted">Comprehensive SFTP connection and system diagnostics</p>
            </div>
            <?php if (!$authorized): ?>
            <div class="card">
                <div class="card-header">
                    <i class="fa fa-lock me-2"></i>Authentication Required
                </div>
                <div class="card-body">
                    <form method="POST" class="row g-3">
                        <div class="col-md-6">
                            <label for="password" class="form-label">Diagnostic Password</label>
                            <input type="password" class="form-control" id="password" name="password" placeholder="Enter diagnostic password" required>
                        </div>
                        <div class="col-12">
                            <button type="submit" class="btn btn-primary">
                                <i class="fa fa-sign-in-alt me-2"></i>Access Diagnostics
                            </button>
                        </div>
                    </form>
                </div>
            </div>
            <?php else: ?>
            <div class="card diagnostic-section">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <span><i class="fa fa-server me-2"></i>System Information</span>
                    <button class="btn btn-sm btn-outline-secondary copy-btn" onclick="copyToClipboard('system-info', event)">
                        <i class="fa fa-copy"></i>
                    </button>
                </div>
                <div class="card-body" id="system-info">
                    <div class="row">
                        <div class="col-md-6">
                            <h6 class="fw-bold">PHP Configuration</h6>
                            <div class="test-result"><strong>PHP Version:</strong> <?php echo $phpInfo['version']; ?></div>
                            <div class="test-result"><strong>Platform:</strong> <?php echo $phpInfo['platform']; ?></div>
                            <div class="test-result"><strong>Memory Limit:</strong> <?php echo $phpInfo['memory_limit']; ?></div>
                            <div class="test-result"><strong>Max Execution Time:</strong> <?php echo $phpInfo['max_execution_time']; ?> seconds</div>
                            <div class="test-result"><strong>Upload Max Filesize:</strong> <?php echo $phpInfo['upload_max_filesize']; ?></div>
                            <div class="test-result"><strong>File Uploads:</strong> <?php echo $phpInfo['file_uploads']; ?></div>
                        </div>
                        <div class="col-md-6">
                            <h6 class="fw-bold">Server Information</h6>
                            <div class="test-result"><strong>Server Software:</strong> <?php echo htmlspecialchars($serverInfo['server_software']); ?></div>
                            <div class="test-result"><strong>PHP SAPI:</strong> <?php echo $serverInfo['php_sapi']; ?></div>
                            <div class="test-result"><strong>Document Root:</strong> <?php echo htmlspecialchars($serverInfo['document_root']); ?></div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="card diagnostic-section">
                <div class="card-header">
                    <i class="fa fa-puzzle-piece me-2"></i>PHP Extensions
                </div>
                <div class="card-body">
                    <div class="row">
                        <?php foreach ($phpInfo['extensions'] as $ext => $loaded): ?>
                        <div class="col-md-4 col-lg-3 mb-2">
                            <span class="<?php echo $loaded ? 'status-pass' : 'status-fail'; ?>">
                                <i class="fa fa-<?php echo $loaded ? 'check' : 'times'; ?> me-1"></i>
                                <?php echo strtoupper($ext); ?>
                            </span>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            <?php if ($sftpHost != "" || $sftpUser != ""): ?>
            <div class="card diagnostic-section">
                <div class="card-header">
                    <i class="fa fa-key me-2"></i>SFTP Configuration
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <h6 class="fw-bold">Connection Details</h6>
                            <div class="test-result"><strong>Host:</strong> <?php echo htmlspecialchars($sftpHost); ?></div>
                            <div class="test-result"><strong>Port:</strong> <?php echo $sftpPort; ?></div>
                            <div class="test-result"><strong>Username:</strong> <?php echo htmlspecialchars($sftpUser); ?></div>
                            <div class="test-result"><strong>Authentication Type:</strong> 
                                <?php 
                                if ($sftpPrivateKeyFile && file_exists($sftpPrivateKeyFile)) {
                                    echo '<span class="status-pass">Private Key';
                                    if ($sftpPrivateKeyPassphrase) {
                                        echo ' (with passphrase)';
                                    }
                                    echo '</span>';
                                } elseif ($sftpPass !== "") {
                                    echo '<span class="status-pass">Password</span>';
                                } else {
                                    echo '<span class="status-fail">Not configured</span>';
                                }
                                ?>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <h6 class="fw-bold">Additional Settings</h6>
                            <div class="test-result"><strong>Start Directory:</strong> <?php echo $sftpDir ? htmlspecialchars($sftpDir) : 'Default'; ?></div>
                            <div class="test-result"><strong>Private Key File:</strong> <?php echo $sftpPrivateKeyFile ? htmlspecialchars($sftpPrivateKeyFile) : 'Not used'; ?></div>
                            <div class="test-result"><strong>Test File:</strong> <?php echo htmlspecialchars($testFile); ?></div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="card diagnostic-section">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <span><i class="fa fa-network-wired me-2"></i>SFTP Connection Tests</span>
                    <button class="btn btn-sm btn-outline-secondary copy-btn" onclick="copyToClipboard('sftp-tests', event)">
                        <i class="fa fa-copy"></i>
                    </button>
                </div>
                <div class="card-body" id="sftp-tests">
                    <?php if (empty($sftpResults)): ?>
                        <div class="alert alert-warning">
                            <i class="fa fa-exclamation-triangle me-2"></i>
                            Please configure SFTP credentials at the top of this file to run SFTP tests.
                        </div>
                    <?php else: ?>
                        <?php foreach ($sftpResults as $testName => $result): ?>
                        <div class="test-result">
                            <span class="<?php echo $result ? 'status-pass' : 'status-fail'; ?>">
                                <i class="fa fa-<?php echo $result ? 'check' : 'times'; ?> me-2"></i>
                                <?php echo htmlspecialchars($testName); ?>
                            </span>
                        </div>
                        <?php endforeach; ?>
                        <?php if (!is_null($sftpException)): ?>
                        <div class="alert alert-danger mt-3">
                            <i class="fa fa-exclamation-circle me-2"></i>
                            <strong>Last Exception:</strong> <?php echo htmlspecialchars($sftpException); ?>
                        </div>
                        <?php endif; ?>
                        <?php if (!empty($sftpRawList)): ?>
                        <div class="mt-3">
                            <strong>SFTP Raw Output:</strong>
                            <textarea class="form-control mt-2" rows="5" readonly><?php echo htmlspecialchars(trim($sftpRawList)); ?></textarea>
                        </div>
                        <?php endif; ?>
                    <?php endif; ?>
                </div>
            </div>
            <?php endif; ?>
            <div class="card diagnostic-section">
                <div class="card-header">
                    <i class="fa fa-question-circle me-2"></i>Need Help?
                </div>
                <div class="card-body">
                    <p>If you're experiencing issues with Monsta FTP, please:</p>
                    <ol>
                        <li>Copy the diagnostic results above</li>
                        <li>Include them in your support request</li>
                        <li>Contact us at <a href="https://www.monstaftp.com/contact" target="_blank">monstaftp.com/contact</a></li>
                    </ol>
                    <div class="alert alert-info">
                        <i class="fa fa-info-circle me-2"></i>
                        <strong>Security Note:</strong> Remember to remove or rename this file after completing diagnostics.
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </div>
</div>
        <script>
    // Simple copy-to-clipboard implementation
    function copyToClipboard(text) {
        if (navigator.clipboard && window.isSecureContext) {
            navigator.clipboard.writeText(text);
        } else {
            // Fallback for older browsers
            const textArea = document.createElement("textarea");
            textArea.value = text;
            document.body.appendChild(textArea);
            textArea.select();
            document.execCommand('copy');
            document.body.removeChild(textArea);
        }
    }
    </script>
</body>
</html> 