<?php
// FTP Diagnostics Tool for Monsta FTP
// This tool helps diagnose FTP connection issues

// =================================================================
// SECURITY CONFIGURATION - SET YOUR PASSWORD HERE
// =================================================================
// CHANGE THIS: Set your own secure password below
// Example: $diagnostic_password = "your_secure_password_here";
$diagnostic_password = ""; // LEAVE EMPTY INITIALLY - SET YOUR OWN PASSWORD

// =================================================================
// FTP CONFIGURATION - SET THESE VALUES FOR TESTING
// =================================================================
$ftpHost = "";                   // FTP host address
$ftpPort = 21;                   // Host port (normally 21)
$ftpUser = "";                   // FTP username
$ftpPass = "";                   // FTP password
$ftpPasv = 1;                    // Set to 1 for passive mode, or 0 for active mode
$ftpDir = "";                    // Set a start folder (optional) i.e. /path/to/folder
$ftpTmp = "";                    // Set a temp folder (optional), i.e. /path/to/folder
$testFile = "mftp-test.html";    // File name used for creating test files

// =================================================================
// WORKFLOW STARTS HERE
// =================================================================
// Security: Check if user is authorized to access diagnostics
session_start();
$authorized = false;

// Check if password is configured
if (empty($diagnostic_password)) {
    echo "<!DOCTYPE html>\n";
    echo "<html>\n<head>\n";
    echo "<title>MonstaFTP FTP Diagnostics - Password Required</title>\n";
    echo "<meta charset='utf-8'>\n";
    echo "<style>\n";
    echo "body { font-family: Arial, sans-serif; background: #f5f5f5; display: flex; justify-content: center; align-items: center; min-height: 100vh; margin: 0; }\n";
    echo ".message-container { background: white; padding: 40px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); max-width: 500px; width: 100%; text-align: center; }\n";
    echo ".message { color: #333; font-size: 18px; line-height: 1.5; }\n";
    echo "</style>\n";
    echo "</head>\n<body>\n";
    echo "<div class='message-container'>\n";
    echo "<div class='message'>Please edit this file to set a password to access this page.</div>\n";
    echo "</div>\n";
    echo "</body>\n</html>\n";
    exit;
}

if (isset($_POST['password']) && $_POST['password'] === $diagnostic_password) {
    $_SESSION['diagnostic_authorized'] = true;
    $authorized = true;
} elseif (isset($_SESSION['diagnostic_authorized']) && $_SESSION['diagnostic_authorized'] === true) {
    $authorized = true;
}

// Include required files
require_once(dirname(__FILE__) . "/../application/api/file_sources/connection/mftp_functions.php");
require_once(dirname(__FILE__) . "/../application/api/constants.php");
require_once(dirname(__FILE__) . "/../settings/config.php");

// Initialize variables
$systype = "";
$rawlist = "";
$exceptionMessage = null;
$testArray = array();
$serverInfo = array();
$phpInfo = array();

// Collect PHP and server information
function collectSystemInfo() {
    global $phpInfo, $serverInfo;
    
    $phpInfo = array(
        'version' => PHP_VERSION,
        'os' => PHP_OS,
        'platform' => getPlatformName(),
        'memory_limit' => ini_get('memory_limit'),
        'max_execution_time' => ini_get('max_execution_time'),
        'upload_max_filesize' => ini_get('upload_max_filesize'),
        'post_max_size' => ini_get('post_max_size'),
        'max_input_time' => ini_get('max_input_time'),
        'file_uploads' => ini_get('file_uploads') ? 'Enabled' : 'Disabled',
        'extensions' => array(
            'openssl' => function_exists("openssl_get_publickey"),
            'zip' => extension_loaded('zip'),
            'curl' => extension_loaded('curl'),
            'ftp' => extension_loaded('ftp'),
            'sockets' => extension_loaded('sockets'),
            'mbstring' => extension_loaded('mbstring'),
            'json' => extension_loaded('json'),
            'suhosin' => extension_loaded('suhosin')
        )
    );
    
    $serverInfo = array(
        'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
        'php_sapi' => php_sapi_name(),
        'document_root' => $_SERVER['DOCUMENT_ROOT'] ?? 'Unknown',
        'script_filename' => $_SERVER['SCRIPT_FILENAME'] ?? 'Unknown'
    );
}

function getPlatformName() {
    $os = PHP_OS;
    if (strtoupper(substr($os, 0, 5)) == 'LINUX') return 'Linux';
    elseif (strtoupper(substr($os, 0, 7)) == 'FREEBSD') return 'FreeBSD';
    elseif (strtoupper(substr($os, 0, 3)) == 'WIN') return 'Windows';
    elseif (strtoupper(substr($os, 0, 6)) == 'DARWIN') return 'macOS';
    else return $os;
}

function createTempFileName($ftpTmp, $filename) {
    if ($ftpTmp == "") {
        $ftpTmp = ini_get('upload_tmp_dir') ? ini_get('upload_tmp_dir') : sys_get_temp_dir();
    }
    return tempnam($ftpTmp, $filename);
}

function runFTPTests() {
    global $ftpHost, $ftpPort, $ftpUser, $ftpPass, $ftpPasv, $ftpDir, $ftpTmp, $testFile;
    global $testArray, $systype, $rawlist, $exceptionMessage;
    
    if ($ftpHost == "" || $ftpUser == "" || $ftpPass == "") {
        $testArray["FTP Credentials"] = false;
        return;
    }
    
    try {
        // Test FTP connection
        $testName = "Can connect to FTP server";
        $conn = @ftp_connect($ftpHost, $ftpPort, 10);
        $testArray[$testName] = $conn !== false;

        if ($conn === false) {
            throw new Exception("Unable to connect to FTP server.");
        }

        // Test FTP login
        $testName = "Can log in to FTP server";
        $loginResult = @ftp_login($conn, $ftpUser, $ftpPass);
        $testArray[$testName] = $loginResult;
        
        if (!$loginResult) {
            throw new Exception("Unable to login to FTP server.");
        }

        // Get server system type
        $systype = @ftp_systype($conn);

        // Test passive mode
        if ($ftpPasv) {
            $testName = "Can set FTP mode to passive";
            $pasvResult = @ftp_pasv($conn, true);
            $testArray[$testName] = $pasvResult;
        }

        // Test directory change
        if ($ftpDir != "") {
            $testName = "Can change folder on FTP server";
            $chdirResult = @ftp_chdir($conn, $ftpDir);
            $testArray[$testName] = $chdirResult;
        }

        // Test file upload using the same logic as the main app
        $testName = "Can upload a file to FTP server";
        $filename = createTempFileName($ftpTmp, $testFile);
        file_put_contents($filename, "Hello World!");
        
        // Try different upload paths like the enhanced diagnostics
        $uploadPaths = array(
            $testFile,                    // Just filename
            "/" . $testFile,              // Root with filename
            "./" . $testFile,             // Current directory
        );
        
        $uploadResult = false;
        $uploadError = "";
        $successfulUploadPath = "";
        
        foreach ($uploadPaths as $uploadPath) {
            try {
                $uploadResult = @ftp_put($conn, $uploadPath, $filename, FTP_BINARY);
                if ($uploadResult) {
                    $testArray[$testName . " (to " . $uploadPath . ")"] = true;
                    $successfulUploadPath = $uploadPath;
                    break;
                }
            } catch (Exception $e) {
                $uploadError .= "Failed to upload to $uploadPath: " . $e->getMessage() . "; ";
            }
        }
        
        if (!$uploadResult) {
            $testArray[$testName] = false;
            $exceptionMessage = "Upload failed. Errors: " . $uploadError;
        }
        
        @unlink($filename);

        // Test file deletion (only if upload succeeded)
        if ($uploadResult && !empty($successfulUploadPath)) {
            $testName = "Can delete a file from FTP server";
            $deleteResult = @ftp_delete($conn, $successfulUploadPath);
            $testArray[$testName] = $deleteResult;
        }

        // Get raw directory listing
        $rawlist = @ftp_rawlist($conn, "-a");
        if (is_array($rawlist)) {
            $rawlist = join("\n", $rawlist);
        }
        
        // Test current directory and permissions
        $testName = "Can get current working directory";
        $currentDir = @ftp_pwd($conn);
        $testArray[$testName] = $currentDir !== false;
        
        if ($currentDir !== false) {
            $testArray["Current working directory"] = $currentDir;
        }
        
        // Test if we can create a test directory
        $testName = "Can create test directory";
        $testDirName = "mftp_test_dir_" . time();
        $createDirResult = @ftp_mkdir($conn, $testDirName);
        $testArray[$testName] = $createDirResult;
        
        if ($createDirResult) {
            // Clean up the test directory
            @ftp_rmdir($conn, $testDirName);
        }
        
        @ftp_close($conn);

    } catch (Exception $e) {
        $exceptionMessage = $e->getMessage();
        $testArray[$testName] = false;
    }
}

// Collect system information
collectSystemInfo();

// Run FTP tests if authorized and credentials are provided
if ($authorized && ($ftpHost != "" || $ftpUser != "" || $ftpPass != "")) {
    runFTPTests();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>FTP Diagnostics - Monsta FTP</title>
    <link rel="shortcut icon" type="image/x-icon" href="../application/frontend/images/logo-favicon.png">
    <link rel="apple-touch-icon" href="../application/frontend/images/logo-webclip.png">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://fonts.googleapis.com/css2?family=Open+Sans:wght@300;400;600;700&display=swap">
    <style>
        body { font-family: 'Open Sans', sans-serif; background-color: #f8f9fa; }
        .card { border: none; box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075); border-radius: 0.5rem; }
        .card-header { background-color: #fff; border-bottom: 1px solid #dee2e6; font-weight: 600; }
        .status-pass { color: #198754; font-weight: 600; }
        .status-fail { color: #dc3545; font-weight: 600; }
        .copy-btn { cursor: pointer; color: #6c757d; }
        .copy-btn:hover { color: #495057; }
        .diagnostic-section { margin-bottom: 2rem; }
        .test-result { padding: 0.5rem 0; border-bottom: 1px solid #f1f3f4; }
        .test-result:last-child { border-bottom: none; }
        
        /* Local Font Awesome CSS using local files */
        /* FontAwesome now loaded from CDN - local @font-face removed */
        .fa { font-family: 'FontAwesome'; }
        /* Add common Font Awesome icon classes */
        .fa-check:before { content: '\f00c'; }
        .fa-times:before { content: '\f00d'; }
        .fa-exclamation-triangle:before { content: '\f071'; }
        .fa-exclamation-circle:before { content: '\f06a'; }
        .fa-info-circle:before { content: '\f05a'; }
        .fa-copy:before { content: '\f0c5'; }
        .fa-server:before { content: '\f233'; }
        .fa-puzzle-piece:before { content: '\f12e'; }
        .fa-key:before { content: '\f084'; }
        .fa-network-wired:before { content: '\f6ff'; }
        .fa-question-circle:before { content: '\f059'; }
        .fa-tools:before { content: '\f7d9'; }
        .fa-lock:before { content: '\f023'; }
        .fa-sign-in-alt:before { content: '\f2f6'; }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="row justify-content-center">
            <div class="col-lg-10 col-xl-8">
                
                <!-- Header -->
                <div class="text-center mb-4">
                    <h1 class="h2 mb-2">
                        <i class="fa fa-tools text-primary"></i>
                        Monsta FTP Diagnostics Tool
                    </h1>
                    <p class="text-muted">Comprehensive FTP connection and system diagnostics</p>
                </div>

                <?php if (!$authorized): ?>
                <!-- Authentication Form -->
                <div class="card">
                    <div class="card-header">
                        <i class="fa fa-lock me-2"></i>Authentication Required
                    </div>
                    <div class="card-body">
                        <form method="POST" class="row g-3">
                            <div class="col-md-6">
                                <label for="password" class="form-label">Diagnostic Password</label>
                                <input type="password" class="form-control" id="password" name="password" 
                                       placeholder="Enter diagnostic password" required>
                            </div>
                            <div class="col-12">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fa fa-sign-in-alt me-2"></i>Access Diagnostics
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
                <?php else: ?>

                <!-- System Information -->
                <div class="card diagnostic-section">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <span><i class="fa fa-server me-2"></i>System Information</span>
                        <button class="btn btn-sm btn-outline-secondary copy-btn" onclick="copyToClipboard('system-info', event)">
                            <i class="fa fa-copy"></i>
                        </button>
                    </div>
                    <div class="card-body" id="system-info">
                        <div class="row">
                            <div class="col-md-6">
                                <h6 class="fw-bold">PHP Configuration</h6>
                                <div class="test-result">
                                    <strong>PHP Version:</strong> <?php echo $phpInfo['version']; ?>
                                </div>
                                <div class="test-result">
                                    <strong>Platform:</strong> <?php echo $phpInfo['platform']; ?>
                                </div>
                                <div class="test-result">
                                    <strong>Memory Limit:</strong> <?php echo $phpInfo['memory_limit']; ?>
                                </div>
                                <div class="test-result">
                                    <strong>Max Execution Time:</strong> <?php echo $phpInfo['max_execution_time']; ?> seconds
                                </div>
                                <div class="test-result">
                                    <strong>Upload Max Filesize:</strong> <?php echo $phpInfo['upload_max_filesize']; ?>
                                </div>
                                <div class="test-result">
                                    <strong>File Uploads:</strong> <?php echo $phpInfo['file_uploads']; ?>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <h6 class="fw-bold">Server Information</h6>
                                <div class="test-result">
                                    <strong>Server Software:</strong> <?php echo htmlspecialchars($serverInfo['server_software']); ?>
                                </div>
                                <div class="test-result">
                                    <strong>PHP SAPI:</strong> <?php echo $serverInfo['php_sapi']; ?>
                                </div>
                                <div class="test-result">
                                    <strong>Document Root:</strong> <?php echo htmlspecialchars($serverInfo['document_root']); ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- PHP Extensions -->
                <div class="card diagnostic-section">
                    <div class="card-header">
                        <i class="fa fa-puzzle-piece me-2"></i>PHP Extensions
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <?php foreach ($phpInfo['extensions'] as $ext => $loaded): ?>
                            <div class="col-md-4 col-lg-3 mb-2">
                                <span class="<?php echo $loaded ? 'status-pass' : 'status-fail'; ?>">
                                    <i class="fa fa-<?php echo $loaded ? 'check' : 'times'; ?> me-1"></i>
                                    <?php echo strtoupper($ext); ?>
                                </span>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>

                <?php if ($ftpHost != "" || $ftpUser != ""): ?>
                <div class="card diagnostic-section">
                    <div class="card-header">
                        <i class="fa fa-server me-2"></i>FTP Configuration
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <h6 class="fw-bold">Connection Details</h6>
                                <div class="test-result"><strong>Host:</strong> <?php echo htmlspecialchars($ftpHost); ?></div>
                                <div class="test-result"><strong>Port:</strong> <?php echo $ftpPort; ?></div>
                                <div class="test-result"><strong>Username:</strong> <?php echo htmlspecialchars($ftpUser); ?></div>
                                <div class="test-result"><strong>Password:</strong> <?php echo $ftpPass ? 'Set' : 'Not set'; ?></div>
                                <div class="test-result"><strong>Passive Mode:</strong> <?php echo $ftpPasv ? 'Enabled' : 'Disabled'; ?></div>
                            </div>
                            <div class="col-md-6">
                                <h6 class="fw-bold">Additional Settings</h6>
                                <div class="test-result"><strong>Start Directory:</strong> <?php echo $ftpDir ? htmlspecialchars($ftpDir) : 'Default'; ?></div>
                                <div class="test-result"><strong>Test File:</strong> <?php echo htmlspecialchars($testFile); ?></div>
                                <div class="test-result"><strong>Temp Directory:</strong> <?php echo $ftpTmp ? htmlspecialchars($ftpTmp) : 'System default'; ?></div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- FTP Test Results -->
                <div class="card diagnostic-section">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <span><i class="fa fa-network-wired me-2"></i>FTP Connection Tests</span>
                        <button class="btn btn-sm btn-outline-secondary copy-btn" onclick="copyToClipboard('ftp-tests', event)">
                            <i class="fa fa-copy"></i>
                        </button>
                    </div>
                    <div class="card-body" id="ftp-tests">
                        <?php if (empty($testArray)): ?>
                            <div class="alert alert-warning">
                                <i class="fa fa-exclamation-triangle me-2"></i>
                                Please configure FTP credentials at the top of this file to run FTP tests.
                            </div>
                        <?php else: ?>
                            <?php foreach ($testArray as $testName => $result): ?>
                            <div class="test-result">
                                <span class="<?php echo $result ? 'status-pass' : 'status-fail'; ?>">
                                    <i class="fa fa-<?php echo $result ? 'check' : 'times'; ?> me-2"></i>
                                    <?php echo htmlspecialchars($testName); ?>
                                </span>
                            </div>
                            <?php endforeach; ?>
                            
                            <?php if (!is_null($exceptionMessage)): ?>
                            <div class="alert alert-danger mt-3">
                                <i class="fa fa-exclamation-circle me-2"></i>
                                <strong>Last Exception:</strong> <?php echo htmlspecialchars($exceptionMessage); ?>
                            </div>
                            <?php endif; ?>
                            
                            <?php if (!empty($systype)): ?>
                            <div class="mt-3">
                                <strong>FTP Server Platform:</strong> <?php echo htmlspecialchars(ucwords(strtolower($systype))); ?>
                            </div>
                            <?php endif; ?>
                            
                            <?php if (!empty($rawlist)): ?>
                            <div class="mt-3">
                                <strong>FTP Raw Output:</strong>
                                <textarea class="form-control mt-2" rows="5" readonly><?php echo htmlspecialchars(trim($rawlist)); ?></textarea>
                            </div>
                            <?php endif; ?>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Help Section -->
                <div class="card diagnostic-section">
                    <div class="card-header">
                        <i class="fa fa-question-circle me-2"></i>Need Help?
                    </div>
                    <div class="card-body">
                        <p>If you're experiencing issues with Monsta FTP, please:</p>
                        <ol>
                            <li>Copy the diagnostic results above</li>
                            <li>Include them in your support request</li>
                            <li>Contact us at <a href="https://www.monstaftp.com/contact" target="_blank">monstaftp.com/contact</a></li>
                        </ol>
                        <div class="alert alert-info">
                            <i class="fa fa-info-circle me-2"></i>
                            <strong>Security Note:</strong> Remember to remove or rename this file after completing diagnostics.
                        </div>
                    </div>
                </div>

                <?php endif; ?>
            </div>
        </div>
    </div>

    <script>
    // Simple copy-to-clipboard implementation
    function copyToClipboard(text) {
        if (navigator.clipboard && window.isSecureContext) {
            navigator.clipboard.writeText(text);
        } else {
            // Fallback for older browsers
            const textArea = document.createElement("textarea");
            textArea.value = text;
            document.body.appendChild(textArea);
            textArea.select();
            document.execCommand('copy');
            document.body.removeChild(textArea);
        }
    }
    </script>
</body>
</html>
