<?php
/**
 * Upload Cleanup Tool for MonstaFTP
 * This script cleans up stuck uploads and temp files
 */

require_once(dirname(__FILE__) . '/../application/api/constants.php');
includeMonstaConfig();
require_once(dirname(__FILE__) . '/../application/api/system/SecureSessionManager.php');

// Initialize session
$sessionManager = SecureSessionManager::getInstance();
$sessionManager->initializeSession();

// Handle cleanup request
if (isset($_POST['action'])) {
    $result = ['success' => false, 'message' => '', 'details' => []];
    
    switch ($_POST['action']) {
        case 'cleanup_temp_files':
            $tempDir = sys_get_temp_dir();
            $patterns = ['mftp_upload_*', 'mftp_chunked_*', 'mftp_secure_*', 'mftp_temp_*', '*.lock'];
            $deletedCount = 0;
            
            foreach ($patterns as $pattern) {
                $files = glob($tempDir . '/' . $pattern);
                foreach ($files as $file) {
                    if (file_exists($file) && @unlink($file)) {
                        $deletedCount++;
                        $result['details'][] = 'Deleted: ' . basename($file);
                    }
                }
            }
            
            $result['success'] = true;
            $result['message'] = "Deleted $deletedCount temp files";
            break;
            
        case 'cleanup_session_keys':
            $deletedCount = 0;
            $keysToRemove = [];
            
            foreach ($_SESSION as $key => $value) {
                if (strpos($key, 'UPLOAD_KEY_') === 0) {
                    $keysToRemove[] = $key;
                }
            }
            
            foreach ($keysToRemove as $key) {
                unset($_SESSION[$key]);
                $deletedCount++;
                $result['details'][] = 'Removed session key: ' . $key;
            }
            
            $result['success'] = true;
            $result['message'] = "Removed $deletedCount session keys";
            break;
            
        case 'cleanup_lock_files':
            $lockDir = sys_get_temp_dir() . '/mftp_concurrent_locks';
            $deletedCount = 0;
            
            if (is_dir($lockDir)) {
                $files = glob($lockDir . '/upload_*.lock');
                foreach ($files as $file) {
                    if (file_exists($file) && @unlink($file)) {
                        $deletedCount++;
                        $result['details'][] = 'Deleted: ' . basename($file);
                    }
                }
            }
            
            $result['success'] = true;
            $result['message'] = "Deleted $deletedCount lock files";
            break;
            
        case 'cleanup_all':
            // Cleanup temp files
            $tempDir = sys_get_temp_dir();
            $patterns = ['mftp_upload_*', 'mftp_chunked_*', 'mftp_secure_*', 'mftp_temp_*', '*.lock'];
            $tempDeleted = 0;
            
            foreach ($patterns as $pattern) {
                $files = glob($tempDir . '/' . $pattern);
                foreach ($files as $file) {
                    if (file_exists($file) && @unlink($file)) {
                        $tempDeleted++;
                    }
                }
            }
            
            // Cleanup session keys
            $sessionDeleted = 0;
            $keysToRemove = [];
            
            foreach ($_SESSION as $key => $value) {
                if (strpos($key, 'UPLOAD_KEY_') === 0) {
                    $keysToRemove[] = $key;
                }
            }
            
            foreach ($keysToRemove as $key) {
                unset($_SESSION[$key]);
                $sessionDeleted++;
            }
            
            // Cleanup lock files
            $lockDir = sys_get_temp_dir() . '/mftp_concurrent_locks';
            $lockDeleted = 0;
            
            if (is_dir($lockDir)) {
                $files = glob($lockDir . '/upload_*.lock');
                foreach ($files as $file) {
                    if (file_exists($file) && @unlink($file)) {
                        $lockDeleted++;
                    }
                }
            }
            
            $result['success'] = true;
            $result['message'] = "Cleanup complete: $tempDeleted temp files, $sessionDeleted session keys, $lockDeleted lock files";
            $result['details'] = [
                "Temp files deleted: $tempDeleted",
                "Session keys removed: $sessionDeleted",
                "Lock files deleted: $lockDeleted"
            ];
            break;
    }
    
    header('Content-Type: application/json');
    echo json_encode($result);
    exit;
}

// Get current stats
$tempDir = sys_get_temp_dir();
$uploadFiles = glob($tempDir . '/mftp_upload_*');
$chunkedFiles = glob($tempDir . '/mftp_chunked_*');
$lockFiles = glob($tempDir . '/*.lock');

$lockDir = sys_get_temp_dir() . '/mftp_concurrent_locks';
$concurrentLocks = is_dir($lockDir) ? glob($lockDir . '/upload_*.lock') : [];

$sessionUploadKeys = 0;
$sessionUploadList = [];
foreach ($_SESSION as $key => $value) {
    if (strpos($key, 'UPLOAD_KEY_') === 0 && !strpos($key, '_meta')) {
        $sessionUploadKeys++;
        $meta = isset($_SESSION[$key . '_meta']) ? $_SESSION[$key . '_meta'] : [];
        $sessionUploadList[] = [
            'key' => $key,
            'path' => $value,
            'status' => $meta['status'] ?? 'unknown',
            'created' => isset($meta['created_at']) ? date('Y-m-d H:i:s', $meta['created_at']) : 'unknown'
        ];
    }
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>Upload Cleanup Tool</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body { background: #f5f5f5; }
        .stats-card { margin-bottom: 20px; }
        .btn-cleanup { margin: 5px; }
        .result-box { 
            margin-top: 20px; 
            padding: 15px; 
            border-radius: 5px;
            display: none;
        }
        .result-box.success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .result-box.error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .upload-item { padding: 10px; margin: 5px 0; background: white; border-radius: 5px; }
    </style>
</head>
<body>
    <div class="container mt-5">
        <h1 class="mb-4">🧹 Upload Cleanup Tool</h1>
        
        <div class="alert alert-info">
            <strong>⚠️ Warning:</strong> This tool will remove stuck uploads and temporary files. 
            Only use this if uploads are failing or if you have many stuck uploads.
        </div>
        
        <div class="row">
            <div class="col-md-6">
                <div class="card stats-card">
                    <div class="card-header bg-primary text-white">
                        <strong>📁 Temporary Files</strong>
                    </div>
                    <div class="card-body">
                        <ul class="list-group">
                            <li class="list-group-item d-flex justify-content-between">
                                <span>Upload temp files (mftp_upload_*)</span>
                                <span class="badge bg-warning"><?php echo count($uploadFiles); ?></span>
                            </li>
                            <li class="list-group-item d-flex justify-content-between">
                                <span>Chunked files (mftp_chunked_*)</span>
                                <span class="badge bg-warning"><?php echo count($chunkedFiles); ?></span>
                            </li>
                            <li class="list-group-item d-flex justify-content-between">
                                <span>Lock files</span>
                                <span class="badge bg-warning"><?php echo count($lockFiles) + count($concurrentLocks); ?></span>
                            </li>
                        </ul>
                        <button class="btn btn-danger btn-cleanup mt-3" onclick="cleanup('cleanup_temp_files')">
                            🗑️ Clean Temp Files
                        </button>
                    </div>
                </div>
            </div>
            
            <div class="col-md-6">
                <div class="card stats-card">
                    <div class="card-header bg-primary text-white">
                        <strong>🔑 Session Keys</strong>
                    </div>
                    <div class="card-body">
                        <ul class="list-group">
                            <li class="list-group-item d-flex justify-content-between">
                                <span>Active upload keys</span>
                                <span class="badge bg-warning"><?php echo $sessionUploadKeys; ?></span>
                            </li>
                        </ul>
                        <button class="btn btn-danger btn-cleanup mt-3" onclick="cleanup('cleanup_session_keys')">
                            🗑️ Clean Session Keys
                        </button>
                    </div>
                </div>
            </div>
        </div>
        
        <?php if (!empty($sessionUploadList)): ?>
        <div class="card mb-4">
            <div class="card-header bg-info text-white">
                <strong>📋 Active Upload Keys</strong>
            </div>
            <div class="card-body">
                <?php foreach ($sessionUploadList as $upload): ?>
                <div class="upload-item">
                    <strong><?php echo htmlspecialchars($upload['key']); ?></strong><br>
                    <small>
                        Status: <span class="badge bg-secondary"><?php echo htmlspecialchars($upload['status']); ?></span> |
                        Created: <?php echo htmlspecialchars($upload['created']); ?><br>
                        Path: <code><?php echo htmlspecialchars($upload['path']); ?></code>
                    </small>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>
        
        <div class="card mb-4">
            <div class="card-body text-center">
                <h5>Complete Cleanup</h5>
                <p>Remove all temp files, session keys, and lock files</p>
                <button class="btn btn-danger btn-lg" onclick="cleanup('cleanup_all')">
                    🗑️ Clean Everything
                </button>
            </div>
        </div>
        
        <div class="result-box" id="result-box"></div>
        
        <div class="mt-4">
            <a href="upload-diagnostics.php" class="btn btn-secondary">📊 View Diagnostics</a>
            <a href="view-logs.php" class="btn btn-secondary">📋 View Logs</a>
            <a href="javascript:location.reload()" class="btn btn-primary">🔄 Refresh</a>
        </div>
    </div>
    
    <script>
        function cleanup(action) {
            if (!confirm('Are you sure you want to perform this cleanup?')) {
                return;
            }
            
            const resultBox = document.getElementById('result-box');
            resultBox.style.display = 'block';
            resultBox.className = 'result-box';
            resultBox.innerHTML = '<strong>Processing...</strong>';
            
            fetch('cleanup-uploads.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=' + encodeURIComponent(action)
            })
            .then(response => response.json())
            .then(data => {
                resultBox.className = 'result-box ' + (data.success ? 'success' : 'error');
                let html = '<strong>' + data.message + '</strong>';
                
                if (data.details && data.details.length > 0) {
                    html += '<ul class="mt-2 mb-0">';
                    data.details.forEach(detail => {
                        html += '<li>' + detail + '</li>';
                    });
                    html += '</ul>';
                }
                
                resultBox.innerHTML = html;
                
                // Reload after 2 seconds
                setTimeout(() => {
                    location.reload();
                }, 2000);
            })
            .catch(error => {
                resultBox.className = 'result-box error';
                resultBox.innerHTML = '<strong>Error:</strong> ' + error.message;
            });
        }
    </script>
</body>
</html>


