<?php
/**
 * Upload Configuration Settings
 * 
 * This file loads upload settings from the main config.php file
 * and provides utility functions for the chunked upload system.
 */

// Upload configuration constants are now defined in config_internal.php
// This file provides utility functions for the upload system

// Additional constants not in main config
define('MFTP_UPLOAD_MAX_EXECUTION_TIME', 300); // 5 minutes per request
define('MFTP_UPLOAD_TEMP_DIR', sys_get_temp_dir());

/**
 * Get upload configuration array
 */
function getUploadConfig() {
    return [
        'session_timeout' => MFTP_UPLOAD_SESSION_TIMEOUT,
        'max_execution_time' => MFTP_UPLOAD_MAX_EXECUTION_TIME,
        'max_concurrent_uploads' => MFTP_MAX_CONCURRENT_UPLOADS,
        'retry_attempts' => MFTP_UPLOAD_RETRY_ATTEMPTS,
        'retry_delay' => MFTP_UPLOAD_RETRY_DELAY,
        'chunk_size' => MFTP_UPLOAD_CHUNK_SIZE,
        'temp_dir' => MFTP_UPLOAD_TEMP_DIR,
        'fallback_enabled' => MFTP_UPLOAD_FALLBACK_ENABLED,
        'session_extend' => MFTP_UPLOAD_SESSION_EXTEND,
        'session_activity_update' => MFTP_UPLOAD_SESSION_ACTIVITY_UPDATE,
        'enhanced_error_handling' => MFTP_UPLOAD_ENHANCED_ERROR_HANDLING,
        'debug_logging' => MFTP_UPLOAD_DEBUG_LOGGING,
        'use_file_locking' => MFTP_UPLOAD_USE_FILE_LOCKING,
        'lock_timeout' => MFTP_UPLOAD_LOCK_TIMEOUT,
        'auto_cleanup' => MFTP_UPLOAD_AUTO_CLEANUP,
        'cleanup_age' => MFTP_UPLOAD_CLEANUP_AGE
    ];
}

/**
 * Log upload debug message
 */
function mftpUploadLog($level, $message) {
    if (MFTP_UPLOAD_DEBUG_LOGGING && function_exists('mftpLog')) {
        mftpLog($level, "[UploadConfig] " . $message);
    }
}

/**
 * Clean up old upload files and session data
 */
function cleanupOldUploads() {
    if (!MFTP_UPLOAD_AUTO_CLEANUP) {
        return;
    }
    
    $cleanupAge = MFTP_UPLOAD_CLEANUP_AGE;
    $tempDir = MFTP_UPLOAD_TEMP_DIR;
    $currentTime = time();
    
    // Clean up old temp files
    $pattern = $tempDir . '/mftp_chunked_*';
    $files = glob($pattern);
    
    foreach ($files as $file) {
        if (file_exists($file) && ($currentTime - filemtime($file)) > $cleanupAge) {
            @unlink($file);
            mftpUploadLog(LOG_DEBUG, "Cleaned up old upload file: " . basename($file));
        }
    }
    
    // Clean up old lock files
    $lockPattern = $tempDir . '/*.lock';
    $lockFiles = glob($lockPattern);
    
    foreach ($lockFiles as $lockFile) {
        if (file_exists($lockFile) && ($currentTime - filemtime($lockFile)) > $cleanupAge) {
            @unlink($lockFile);
            mftpUploadLog(LOG_DEBUG, "Cleaned up old lock file: " . basename($lockFile));
        }
    }
    
    // Clean up old session data (only completed or failed uploads)
    if (isset($_SESSION)) {
        $keysToRemove = [];
        foreach ($_SESSION as $key => $value) {
            if (strpos($key, 'UPLOAD_KEY_') === 0) {
                $metaKey = $key . '_meta';
                if (isset($_SESSION[$metaKey]) && isset($_SESSION[$metaKey]['created_at'])) {
                    $meta = $_SESSION[$metaKey];
                    $age = $currentTime - $meta['created_at'];
                    
                    // Only clean up if:
                    // 1. Upload is completed or failed (not in progress)
                    // 2. AND it's older than cleanup age
                    $isCompleted = isset($meta['status']) && in_array($meta['status'], ['completed', 'failed']);
                    $isOld = $age > $cleanupAge;
                    
                    if ($isCompleted && $isOld) {
                        $keysToRemove[] = $key;
                        $keysToRemove[] = $metaKey;
                    }
                }
            }
        }
        
        foreach ($keysToRemove as $key) {
            unset($_SESSION[$key]);
        }
        
        if (!empty($keysToRemove)) {
            mftpUploadLog(LOG_DEBUG, "Cleaned up " . count($keysToRemove) . " old completed session entries");
        }
    }
}

// Run cleanup on include
cleanupOldUploads();
