<?php
require_once(dirname(__FILE__) . "/constants.php");
includeMonstaConfig();

require_once(dirname(__FILE__) . '/system/SecureSessionManager.php');

// Initialize secure session
$sessionManager = SecureSessionManager::getInstance();
$sessionManager->initializeSession();

require_once(dirname(__FILE__) . '/system/ApplicationSettings.php');
require_once(dirname(__FILE__) . '/request_processor/RequestMarshaller.php');
require_once(dirname(__FILE__) . '/lib/helpers.php');
require_once(dirname(__FILE__) . '/lib/response_helpers.php');
require_once(dirname(__FILE__) . '/lib/InputValidator.php');
require_once(dirname(__FILE__) . '/file_sources/PathOperations.php');
require_once(dirname(__FILE__) . '/file_sources/PerformanceOptimizer.php');
require_once(dirname(__FILE__) . '/file_sources/StreamingTransferManager.php');
require_once(dirname(__FILE__) . '/file_sources/OptimizedArchiveProcessor.php');

dieIfNotPOST();

require_once(dirname(__FILE__) . '/lib/access_check.php');

/**
 * Optimized upload handler with performance improvements
 */
class OptimizedUploadHandler {
    
    private $performanceOptimizer;
    private $streamingManager;
    private $marshaller;
    private $sessionKey;
    private $uploadStats;
    
    public function __construct() {
        $this->performanceOptimizer = PerformanceOptimizer::getInstance();
        $this->streamingManager = new StreamingTransferManager(
            StreamingTransferManager::createJsonProgressCallback()
        );
        $this->marshaller = new RequestMarshaller();
        $this->uploadStats = [];
    }
    
    /**
     * Handle optimized upload request
     */
    public function handleUpload() {
        try {
            clearOldTransfers();
            
            $request = $this->parseUploadRequest();
            $this->validateUploadRequest($request);
            
            // Determine upload strategy based on file size and system capabilities
            $uploadStrategy = $this->determineUploadStrategy($request);
            
            mftpLog(LOG_INFO, "OptimizedUploadHandler: Using $uploadStrategy strategy for upload");
            
            switch ($uploadStrategy) {
                case 'streaming':
                    return $this->handleStreamingUpload($request);
                    
                case 'chunked':
                    return $this->handleChunkedUpload($request);
                    
                case 'standard':
                    return $this->handleStandardUpload($request);
                    
                case 'archive_optimized':
                    return $this->handleArchiveOptimizedUpload($request);
                    
                default:
                    throw new RuntimeException("Unknown upload strategy: $uploadStrategy");
            }
            
        } catch (Exception $e) {
            $this->handleUploadError($e);
        } finally {
            $this->marshaller->disconnect();
        }
    }
    
    /**
     * Parse and validate upload request
     */
    private function parseUploadRequest() {
        $rawRequest = $_SERVER['HTTP_X_MONSTA'] ?? null;
        
        if (!$rawRequest) {
            throw new InvalidArgumentException("No upload request data provided");
        }
        
        $jsonEncodedRequest = b64DecodeUnicode($rawRequest);
        $request = json_decode($jsonEncodedRequest, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new InvalidArgumentException("Invalid JSON in upload request: " . json_last_error_msg());
        }
        
        return $request;
    }
    
    /**
     * Validate upload request structure and security
     */
    private function validateUploadRequest($request) {
        InputValidator::validateApiRequest($request);
        
        if (isset($request['context']['remotePath'])) {
            $validatedPath = InputValidator::validateFilePath($request['context']['remotePath'], true);
            $request['context']['remotePath'] = $validatedPath;
        }
        
        $this->marshaller->testConfiguration($request, false);
    }
    
    /**
     * Determine optimal upload strategy based on file characteristics
     */
    private function determineUploadStrategy($request) {
        $contentLength = $_SERVER['CONTENT_LENGTH'] ?? 0;
        $isArchive = $request['actionName'] === 'uploadArchive';
        
        $metrics = $this->performanceOptimizer->getPerformanceMetrics();
        
        // Archive files get special handling
        if ($isArchive) {
            return 'archive_optimized';
        }
        
        // Large files use streaming
        if ($contentLength > $metrics['available_memory'] * 0.1) {
            return 'streaming';
        }
        
        // Medium files use chunked upload
        if ($contentLength > 10485760) { // > 10MB
            return 'chunked';
        }
        
        // Small files use standard upload
        return 'standard';
    }
    
    /**
     * Handle streaming upload for large files
     */
    private function handleStreamingUpload($request) {
        $uploadPath = getTempTransferPath($request['context']['remotePath']);
        $contentLength = $_SERVER['CONTENT_LENGTH'] ?? null;
        
        monstaUploadDebug("STARTED STREAMING UPLOAD TO $uploadPath");
        
        // Use optimized buffer size
        $bufferSize = $this->performanceOptimizer->getOptimalBufferSize($contentLength);
        
        // Stream directly from input to file
        $inputHandle = fopen('php://input', 'rb');
        $outputHandle = fopen($uploadPath, 'wb');
        
        if (!$inputHandle || !$outputHandle) {
            throw new RuntimeException("Failed to open upload streams");
        }
        
        try {
            $this->streamingManager->streamDownload($inputHandle, $uploadPath, $contentLength);
            monstaUploadDebug("FINISHED STREAMING UPLOAD TO $uploadPath");
            
            return $this->processUploadedFile($request, $uploadPath);
            
        } finally {
            if ($inputHandle) fclose($inputHandle);
            if ($outputHandle) fclose($outputHandle);
        }
    }
    
    /**
     * Handle chunked upload for medium files
     */
    private function handleChunkedUpload($request) {
        $uploadPath = getTempTransferPath($request['context']['remotePath']);
        $chunkSize = $this->performanceOptimizer->getOptimalChunkSize();
        
        monstaUploadDebug("STARTED CHUNKED UPLOAD TO $uploadPath");
        
        // Use chunked reading with optimal buffer size
        $result = $this->streamingManager->streamChunkedUpload(
            'php://input',
            function($chunk, $index, $offset) use ($uploadPath) {
                return file_put_contents($uploadPath, $chunk, FILE_APPEND | LOCK_EX) !== false;
            },
            $chunkSize
        );
        
        monstaUploadDebug("FINISHED CHUNKED UPLOAD TO $uploadPath");
        
        return $this->processUploadedFile($request, $uploadPath);
    }
    
    /**
     * Handle standard upload for small files
     */
    private function handleStandardUpload($request) {
        $uploadPath = getTempTransferPath($request['context']['remotePath']);
        
        monstaUploadDebug("STARTED STANDARD UPLOAD TO $uploadPath");
        
        // Use the existing readUpload function but with validation
        readUpload($uploadPath);
        
        monstaUploadDebug("FINISHED STANDARD UPLOAD TO $uploadPath");
        
        return $this->processUploadedFile($request, $uploadPath);
    }
    
    /**
     * Handle archive upload with optimization
     */
    private function handleArchiveOptimizedUpload($request) {
        $uploadPath = getTempTransferPath($request['context']['remotePath']);
        $contentLength = $_SERVER['CONTENT_LENGTH'] ?? null;
        
        monstaUploadDebug("STARTED ARCHIVE OPTIMIZED UPLOAD TO $uploadPath");
        
        // For archives, use streaming to avoid memory issues
        $this->handleStreamingUpload($request);
        
        monstaUploadDebug("FINISHED ARCHIVE OPTIMIZED UPLOAD TO $uploadPath");
        
        // Process archive with optimized processor
        $archiveProcessor = new OptimizedArchiveProcessor();
        
        try {
            $applicationSettings = new ApplicationSettings(APPLICATION_SETTINGS_PATH);
            
            // Get file count efficiently
            $archive = \wapmorgan\UnifiedArchive\UnifiedArchive::open($uploadPath);
            if (!$archive) {
                throw new RuntimeException("Invalid archive file");
            }
            
            $archiveFileCount = count($archive->getFileNames());
            $archive->close();
            
            $fileKey = generateRandomString(16);
            
            $_SESSION[MFTP_SESSION_KEY_PREFIX . $fileKey] = array(
                "archivePath" => $uploadPath,
                "extractDirectory" => PathOperations::remoteDirname($request['context']['remotePath']),
                "useOptimizedProcessor" => true
            );
            
            $response = array(
                "success" => true,
                "fileKey" => $fileKey,
                "fileCount" => $archiveFileCount,
                "optimized" => true
            );
            
            return json_encode($response);
            
        } catch (Exception $e) {
            cleanupTempTransferPath($uploadPath);
            throw $e;
        }
    }
    
    /**
     * Process uploaded file after transfer
     */
    private function processUploadedFile($request, $uploadPath) {
        // Validate uploaded file
        if (file_exists($uploadPath)) {
            $fileSize = filesize($uploadPath);
            InputValidator::validateFileUpload($uploadPath, $request['context']['remotePath'], $fileSize);
        }
        
        $request['context']['localPath'] = $uploadPath;
        
        try {
            $result = $this->marshaller->marshallRequest($request);
            cleanupTempTransferPath($uploadPath);
            return $result;
            
        } catch (Exception $e) {
            cleanupTempTransferPath($uploadPath);
            throw $e;
        }
    }
    
    /**
     * Handle upload errors with detailed logging
     */
    private function handleUploadError($exception) {
        $errorDetails = [
            'error' => $exception->getMessage(),
            'file' => $exception->getFile(),
            'line' => $exception->getLine(),
            'memory_usage' => memory_get_usage(true),
            'memory_peak' => memory_get_peak_usage(true),
            'upload_stats' => $this->uploadStats
        ];
        
        mftpLog(LOG_ERROR, "OptimizedUploadHandler: Upload failed - " . json_encode($errorDetails));
        
        handleExceptionInRequest($exception);
    }
    
    /**
     * Get upload performance statistics
     */
    public function getUploadStats() {
        return array_merge($this->uploadStats, [
            'performance_metrics' => $this->performanceOptimizer->getPerformanceMetrics(),
            'streaming_stats' => $this->streamingManager->getTransferStats()
        ]);
    }
}

// Handle the upload request
try {
    $uploadHandler = new OptimizedUploadHandler();
    echo $uploadHandler->handleUpload();
    
} catch (Exception $e) {
    handleExceptionInRequest($e);
} 