<?php
    require_once(dirname(__FILE__) . "/constants.php");
    includeMonstaConfig();
    
    // Fallback mapping for environments without LOG_ERROR
    if (!defined('LOG_ERROR') && defined('LOG_ERR')) {
        define('LOG_ERROR', LOG_ERR);
    }
    
    // Begin output buffering to capture any stray output
    ob_start();
    
    // Emit JSON body on fatal errors
    register_shutdown_function(function () {
        $err = error_get_last();
        if ($err && in_array($err['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            while (ob_get_level() > 0) { @ob_end_clean(); }
            if (!headers_sent()) {
                http_response_code(500);
                header('Content-Type: application/json');
            }
            echo json_encode([
                'success' => false,
                'errors' => ['Upload failed: ' . $err['message']],
                'localizedErrors' => [['errorName' => 'UPLOAD_FATAL_ERROR', 'context' => ['action' => $_GET['action'] ?? 'unknown', 'message' => $err['message']]]]
            ]);
        }
    });
    
    require_once(dirname(__FILE__) . '/system/SecureSessionManager.php');
    
    // Extend session lifetime for upload operations BEFORE initializing session
    $currentTimeout = ini_get('session.gc_maxlifetime');
    $uploadTimeout = max($currentTimeout, 7200); // 2 hours for bulk uploads
    @ini_set('session.gc_maxlifetime', $uploadTimeout);
    
    // Initialize secure session with enhanced security
    $sessionManager = SecureSessionManager::getInstance();
    $sessionManager->initializeSession();
    
    // Apply security headers based on environment
    $sessionManager->applySecurityHeaders();
    
    if (function_exists('mftpLog')) {
        mftpLog(LOG_DEBUG, "ChunkedUpload: Session initialized, ID: " . session_id() . ", timeout: " . ini_get('session.gc_maxlifetime'));
    }
    
    // Update session security tracking to prevent timeout during uploads
    if (isset($_SESSION['_security'])) {
        $_SESSION['_security']['last_activity'] = time();
        $_SESSION['_security']['upload_mode'] = true;
    }
    
    // Force session write to prevent timeout during bulk uploads
    // But don't restart session if we're in the middle of processing
    if (!isset($_SESSION['_upload_processing'])) {
        $_SESSION['_upload_processing'] = true;
        $_SESSION['_upload_start_time'] = time();
    }
    
    // Log session state for debugging
    if (function_exists('mftpLog')) {
        mftpLog(LOG_DEBUG, "Upload session state - ID: " . session_id() . ", Processing: " . (isset($_SESSION['_upload_processing']) ? 'true' : 'false') . ", Start time: " . (isset($_SESSION['_upload_start_time']) ? $_SESSION['_upload_start_time'] : 'not set'));
    }
    
    // Clean up old upload sessions to prevent memory bloat during bulk uploads
    // Be more aggressive for large batches to prevent resource exhaustion
    $currentTime = time();
    $keysToRemove = [];
    $activeUploads = 0;
    
    // Count active uploads first
    foreach ($_SESSION as $key => $value) {
        if (strpos($key, 'UPLOAD_KEY_') === 0) {
            $activeUploads++;
        }
    }
    
    // If we have many active uploads, be more aggressive with cleanup
    $cleanupThreshold = $activeUploads > 15 ? 300 : 1800; // 5 minutes for large batches, 30 minutes for small ones
    
    foreach ($_SESSION as $key => $value) {
        if (strpos($key, 'UPLOAD_KEY_') === 0) {
            $metaKey = $key . '_meta';
            if (isset($_SESSION[$metaKey]) && isset($_SESSION[$metaKey]['created_at'])) {
                $age = $currentTime - $_SESSION[$metaKey]['created_at'];
                // Remove completed uploads immediately for large batches
                if ($activeUploads > 15 && isset($_SESSION[$metaKey]['status']) && $_SESSION[$metaKey]['status'] === 'completed') {
                    $keysToRemove[] = $key;
                    $keysToRemove[] = $metaKey;
                }
                // Remove uploads older than 2 hours and marked as completed
                elseif ($age > 7200 && isset($_SESSION[$metaKey]['status']) && $_SESSION[$metaKey]['status'] === 'completed') {
                    $keysToRemove[] = $key;
                    $keysToRemove[] = $metaKey;
                }
                // Remove stuck uploads based on threshold
                elseif ($age > $cleanupThreshold && (!isset($_SESSION[$metaKey]['status']) || $_SESSION[$metaKey]['status'] !== 'completed')) {
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_WARNING, "Removing stuck upload session: $key (age: {$age}s, threshold: {$cleanupThreshold}s)");
                    }
                    $keysToRemove[] = $key;
                    $keysToRemove[] = $metaKey;
                }
            }
        }
    }
    
    // Log cleanup activity
    if (function_exists('mftpLog') && count($keysToRemove) > 0) {
        mftpLog(LOG_DEBUG, "Cleaning up " . (count($keysToRemove)/2) . " upload sessions (active: $activeUploads, threshold: {$cleanupThreshold}s)");
    }
    
    foreach ($keysToRemove as $key) {
        unset($_SESSION[$key]);
    }
    require_once(dirname(__FILE__) . '/request_processor/RequestMarshaller.php');
    require_once(dirname(__FILE__) . '/lib/helpers.php');
    require_once(dirname(__FILE__) . '/lib/response_helpers.php');
    require_once(dirname(__FILE__) . '/lib/InputValidator.php');
    require_once(dirname(__FILE__) . '/file_sources/MultiStageUploadHelper.php');
    require_once(dirname(__FILE__) . '/upload_config.php');

    dieIfNotPOST();

    require_once(dirname(__FILE__) . '/lib/access_check.php');

    if (!isset($_GET['action']) || !isset($_GET['uploadId'])) {
        die();
    }

    $mftpUploadAction = $_GET['action'];
    $mftpUploadId = $_GET['uploadId'];
    
    // DEBUG: Log the raw upload ID to see if it's being corrupted
    if (function_exists('mftpLog')) {
        mftpLog(LOG_DEBUG, "ChunkedUpload: Raw upload ID from GET: '$mftpUploadId' (length: " . strlen($mftpUploadId) . ")");
        mftpLog(LOG_DEBUG, "ChunkedUpload: Raw upload ID bytes: " . bin2hex($mftpUploadId));
        mftpLog(LOG_DEBUG, "ChunkedUpload: Action: '$mftpUploadAction'");
    }
    
    // Backup direct error logging
    error_log("MFTP DEBUG: Raw upload ID from GET: '$mftpUploadId' (length: " . strlen($mftpUploadId) . ")");
    error_log("MFTP DEBUG: Raw upload ID bytes: " . bin2hex($mftpUploadId));
    error_log("MFTP DEBUG: Action: '$mftpUploadAction'");
    
    // Resource usage debugging
    $memoryUsage = memory_get_usage(true);
    $memoryPeak = memory_get_peak_usage(true);
    $sessionSize = strlen(serialize($_SESSION));
    $uploadCount = 0;
    foreach ($_SESSION as $key => $value) {
        if (strpos($key, 'UPLOAD_KEY_') === 0 && !strpos($key, '_meta')) {
            $uploadCount++;
        }
    }
    error_log("MFTP DEBUG: Memory usage: " . round($memoryUsage/1024/1024, 2) . "MB, Peak: " . round($memoryPeak/1024/1024, 2) . "MB, Session size: " . round($sessionSize/1024, 2) . "KB, Upload count: $uploadCount");
    
    // PHP limits debugging
    $memoryLimit = ini_get('memory_limit');
    $maxExecutionTime = ini_get('max_execution_time');
    $uploadMaxFilesize = ini_get('upload_max_filesize');
    $postMaxSize = ini_get('post_max_size');
    $maxFileUploads = ini_get('max_file_uploads');
    $sessionGcMaxlifetime = ini_get('session.gc_maxlifetime');
    error_log("MFTP DEBUG: PHP Limits - Memory: $memoryLimit, Max exec time: $maxExecutionTime, Upload max filesize: $uploadMaxFilesize, Post max size: $postMaxSize, Max file uploads: $maxFileUploads, Session GC: $sessionGcMaxlifetime");
    
    // Validate GET parameters
    if (!in_array($mftpUploadAction, ['initiate', 'progress', 'finish'], true)) {
        die('Invalid action');
    }
    
    try {
        InputValidator::validateSessionKey($mftpUploadId);
    } catch (InvalidArgumentException $e) {
        die('Invalid upload ID');
    }

    set_time_limit(600); // 10 minutes for bulk uploads
    
    // Add memory limit increase for bulk uploads
    ini_set('memory_limit', '512M');
    
    // Force garbage collection for large batches
    if (function_exists('gc_collect_cycles')) {
        gc_collect_cycles();
    }
    
    // Add small delay to prevent overwhelming the server during bulk uploads
    usleep(100000); // 100ms delay
    
    // For large batches, add additional delay to prevent session conflicts
    $activeUploads = 0;
    foreach ($_SESSION as $key => $value) {
        if (strpos($key, 'UPLOAD_KEY_') === 0) {
            $activeUploads++;
        }
    }
    
    if ($activeUploads > 15) {
        usleep(200000); // Additional 200ms delay for large batches
        if (function_exists('mftpLog')) {
            mftpLog(LOG_DEBUG, "Large batch detected ($activeUploads uploads), applying additional throttling");
        }
    }

    function mftpGenerateChunkedSessionKey($uploadId) {
        // FIXED: Use simple, consistent session key generation
        // The key should be the same for initiate, progress, and finish phases
        
        // Normalize upload ID to prevent case/encoding issues
        $uploadId = trim($uploadId);
        
        if (function_exists('mftpLog')) {
            mftpLog(LOG_DEBUG, "ChunkedUpload: Generating session key for upload ID: '$uploadId' (length: " . strlen($uploadId) . ")");
        }
        
        // Use a simple, consistent key format without hash suffix
        // This ensures the same key is used across all phases
        $sessionKey = 'UPLOAD_KEY_' . $uploadId;
        
        if (function_exists('mftpLog')) {
            mftpLog(LOG_DEBUG, "ChunkedUpload: Using session key '$sessionKey' for upload $uploadId");
        }
        
        return $sessionKey;
    }

    function getChunkedTempPathFromSession($uploadId) {
        // Normalize upload ID
        $uploadId = trim($uploadId);
        
        if (function_exists('mftpLog')) {
            mftpLog(LOG_DEBUG, "ChunkedUpload: Looking for temp path for upload ID: '$uploadId' (length: " . strlen($uploadId) . ")");
        }
        
        $sessionKey = mftpGenerateChunkedSessionKey($uploadId);

        if (!isset($_SESSION[$sessionKey])) {
            if (function_exists('mftpLog')) {
                mftpLog(LOG_WARNING, "ChunkedUpload: Primary session key '$sessionKey' not found for upload $uploadId");
            }
            
            // Try to find the session key using the old format as fallback
            $oldSessionKey = 'UPLOAD_KEY_' . $uploadId;
            if (isset($_SESSION[$oldSessionKey])) {
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_INFO, "ChunkedUpload: Using old session key format for upload $uploadId");
                }
                return $_SESSION[$oldSessionKey];
            }
            
            // Try to find any session key that starts with the upload ID
            $uploadKeyPrefix = 'UPLOAD_KEY_' . $uploadId;
            $foundKeys = [];
            foreach ($_SESSION as $key => $value) {
                if (strpos($key, $uploadKeyPrefix) === 0) {
                    $foundKeys[] = $key;
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_INFO, "ChunkedUpload: Found matching session key '$key' for upload $uploadId");
                    }
                    return $value;
                }
            }
            
        // Debug information for session issues
        if (function_exists('mftpLog')) {
            mftpLog(LOG_ERROR, "ChunkedUpload: Session key '$sessionKey' not found for upload '$uploadId'");
            mftpLog(LOG_ERROR, "ChunkedUpload: Found keys starting with '$uploadKeyPrefix': " . implode(', ', $foundKeys));
            mftpLog(LOG_ERROR, "ChunkedUpload: All session keys: " . implode(', ', array_keys($_SESSION)));
            mftpLog(LOG_ERROR, "ChunkedUpload: Session ID: " . session_id());
        }
        
        // Also log to error_log for debugging
        error_log("MFTP DEBUG: Session key '$sessionKey' not found for upload ID '$uploadId'");
        error_log("MFTP DEBUG: Found keys starting with '$uploadKeyPrefix': " . implode(', ', $foundKeys));
        error_log("MFTP DEBUG: All session keys: " . implode(', ', array_keys($_SESSION)));
        error_log("MFTP DEBUG: Session ID: " . session_id());
        
        // ENHANCED FALLBACK: Try multiple temp file patterns
        $fallbackPaths = [
            sys_get_temp_dir() . '/mftp_chunked_' . $uploadId,
            sys_get_temp_dir() . '/mftp_secure_' . $uploadId,
            sys_get_temp_dir() . '/mftp_upload_' . $uploadId,
            sys_get_temp_dir() . '/mftp_temp_' . $uploadId,
            sys_get_temp_dir() . '/mftp_' . $uploadId . '.tmp',
            sys_get_temp_dir() . '/mftp_' . $uploadId . '.upload'
        ];
        
        foreach ($fallbackPaths as $fallbackTempPath) {
            if (file_exists($fallbackTempPath)) {
                // Check if this is a fallback file that contains the actual temp path
                $content = file_get_contents($fallbackTempPath);
                if ($content && file_exists($content)) {
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_INFO, "ChunkedUpload: Using fallback file '$fallbackTempPath' which points to: $content");
                    }
                    return $content;
                } else if (function_exists('mftpLog')) {
                    mftpLog(LOG_INFO, "ChunkedUpload: Using fallback temp file: $fallbackTempPath");
                }
                return $fallbackTempPath;
            }
        }
        
        // Try to find any temp file that might contain this upload ID
        $tempDir = sys_get_temp_dir();
        $files = glob($tempDir . '/mftp*' . $uploadId . '*');
        if (!empty($files)) {
            $tempFile = $files[0];
            if (file_exists($tempFile)) {
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_INFO, "ChunkedUpload: Found temp file by pattern: $tempFile");
                }
                return $tempFile;
            }
        }
            
            // If no fallback temp file found, throw error
            throw new Exception("Session key not set and no fallback temp file found. Upload ID: $uploadId, Session Key: $sessionKey");
        }

        return $_SESSION[$sessionKey];
    }

    function mftpChunkedUploadInitiate($marshaller, $uploadId, $request) {
        // Validate request structure
        InputValidator::validateApiRequest($request);
        
        // Validate remote path
        if (isset($request['context']['remotePath'])) {
            $validatedPath = InputValidator::validateFilePath($request['context']['remotePath'], true);
            $request['context']['remotePath'] = $validatedPath;
        }
        
        $sessionKey = mftpGenerateChunkedSessionKey($uploadId);

        // SIMPLIFIED: Use file system locks to track concurrent uploads instead of session counting
        $debugFile = dirname(__FILE__) . '/../logs/mftp_debug.log';
        $lockDir = sys_get_temp_dir() . '/mftp_concurrent_locks';
        
        // Create lock directory if it doesn't exist
        if (!is_dir($lockDir)) {
            mkdir($lockDir, 0755, true);
        }
        
        // Clean up stale lock files (older than 1 hour)
        $staleLocks = glob($lockDir . '/upload_*.lock');
        $cleanedLocks = 0;
        foreach ($staleLocks as $lockFile) {
            if (file_exists($lockFile) && (time() - filemtime($lockFile)) > 3600) {
                @unlink($lockFile);
                $cleanedLocks++;
            }
        }
        
        if ($cleanedLocks > 0) {
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: Cleaned up $cleanedLocks stale lock files\n", FILE_APPEND);
        }
        
        // Count active uploads by counting lock files
        $activeLocks = glob($lockDir . '/upload_*.lock');
        $activeUploads = 0;
        foreach ($activeLocks as $lockFile) {
            if (file_exists($lockFile) && (time() - filemtime($lockFile)) < 3600) {
                $activeUploads++;
            }
        }
        
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: Action: initiate, Upload ID: " . $uploadId . ", Active uploads: " . $activeUploads . "/" . MFTP_MAX_CONCURRENT_UPLOADS . "\n", FILE_APPEND);
        
        // Limit concurrent uploads to prevent session conflicts
        if ($activeUploads >= MFTP_MAX_CONCURRENT_UPLOADS) {
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: BLOCKED - Too many concurrent uploads (active: " . $activeUploads . ", max: " . MFTP_MAX_CONCURRENT_UPLOADS . ") for upload: " . $uploadId . "\n", FILE_APPEND);
            throw new Exception("Too many concurrent uploads (max: " . MFTP_MAX_CONCURRENT_UPLOADS . "). Please wait for some uploads to complete before starting new ones.");
        }
        
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: PASSED concurrent upload check for upload: " . $uploadId . "\n", FILE_APPEND);

        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: About to test FTP configuration for upload: " . $uploadId . "\n", FILE_APPEND);
        
        try {
            $marshaller->testConfiguration($request);
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: FTP configuration test PASSED for upload: " . $uploadId . "\n", FILE_APPEND);
        } catch (Exception $e) {
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: FTP configuration test FAILED for upload: " . $uploadId . " - " . $e->getMessage() . "\n", FILE_APPEND);
            throw new Exception("Failed to test FTP configuration: " . $e->getMessage());
        }
        
        $fileName = monstaBasename($request['context']['remotePath']);
        
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: About to create temp file for upload: " . $uploadId . "\n", FILE_APPEND);
        
        // FIXED: Use a simple temp file name to avoid path conflicts
        $tempFilePath = tempnam(sys_get_temp_dir(), 'mftp_upload_');
        if ($tempFilePath === false) {
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: FAILED to create temp file for upload: " . $uploadId . "\n", FILE_APPEND);
            error_log("MFTP DEBUG: Failed to create temp file in " . sys_get_temp_dir());
            throw new Exception("Failed to create temporary file in " . sys_get_temp_dir());
        }
        
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: Created temp file: " . $tempFilePath . " for upload: " . $uploadId . "\n", FILE_APPEND);
        
        // Debug temp file creation
        $tempDir = sys_get_temp_dir();
        $tempFiles = glob($tempDir . '/mftp_upload_*');
        error_log("MFTP DEBUG: Created temp file: $tempFilePath, Total temp files: " . count($tempFiles));
        $fallbackTempPath = sys_get_temp_dir() . '/mftp_chunked_' . $uploadId;
        
        // Create fallback file as a copy/link to the main temp file
        if (!file_exists($fallbackTempPath)) {
            touch($fallbackTempPath);
        }
        
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: About to store session data for upload: " . $uploadId . "\n", FILE_APPEND);
        
        // ENHANCED: Store session data with additional metadata and locking
        $_SESSION[$sessionKey] = $tempFilePath;
        $_SESSION[$sessionKey . '_meta'] = [
            'created_at' => time(),
            'upload_id' => $uploadId,
            'upload_id_bytes' => bin2hex($uploadId), // Store raw bytes for debugging
            'remote_path' => $request['context']['remotePath'],
            'session_id' => session_id(),
            'temp_file' => $tempFilePath,
            'fallback_file' => $fallbackTempPath,
            'status' => 'initiated',
            'active_uploads' => $activeUploads + 1
        ];
        
        // Also create a fallback file with the temp path for extra reliability
        file_put_contents($fallbackTempPath, $tempFilePath);
        
        // Create additional fallback files in different locations
        $additionalFallbacks = [
            sys_get_temp_dir() . '/mftp_secure_' . $uploadId,
            sys_get_temp_dir() . '/mftp_upload_' . $uploadId,
            sys_get_temp_dir() . '/mftp_temp_' . $uploadId
        ];
        
        foreach ($additionalFallbacks as $fallbackPath) {
            file_put_contents($fallbackPath, $tempFilePath);
        }
        
        // Debug session storage
        error_log("MFTP DEBUG: Stored session key '$sessionKey' with value '$tempFilePath' for upload ID '$uploadId'");
        error_log("MFTP DEBUG: Session ID: " . session_id() . ", Session keys: " . implode(', ', array_keys($_SESSION)));
        
        // Force session write to ensure it's saved
        // Don't restart session here as it can interfere with ongoing uploads
        // session_write_close();
        // session_start();
        
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: Session data stored successfully for upload: " . $uploadId . "\n", FILE_APPEND);
        
        // Session will remain active throughout the request to maintain consistency
        
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: About to create lock file for upload: " . $uploadId . "\n", FILE_APPEND);
        
        // Create a lock file to prevent race conditions
        $lockFile = $tempFilePath . '.lock';
        file_put_contents($lockFile, json_encode([
            'upload_id' => $uploadId,
            'created_at' => time(),
            'session_id' => session_id()
        ]));
        
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: Lock file created successfully for upload: " . $uploadId . "\n", FILE_APPEND);
        
        // Create concurrent upload lock file
        $concurrentLockFile = $lockDir . '/upload_' . $uploadId . '.lock';
        file_put_contents($concurrentLockFile, json_encode([
            'upload_id' => $uploadId,
            'created_at' => time(),
            'session_id' => session_id()
        ]));
        
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: Created concurrent lock file: " . $concurrentLockFile . " for upload: " . $uploadId . "\n", FILE_APPEND);
        
        if (function_exists('mftpLog')) {
            mftpLog(LOG_DEBUG, "ChunkedUpload: Initiated upload $uploadId, session key: $sessionKey, temp file: $tempFilePath, fallback: $fallbackTempPath, session ID: " . session_id() . ", active uploads: " . ($activeUploads + 1));
        }
        
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: INITIATE COMPLETED successfully for upload: " . $uploadId . "\n", FILE_APPEND);
    }

    function mftpChunkedUploadProgress($uploadId) {
        $debugFile = dirname(__FILE__) . '/../logs/mftp_debug.log';
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: PROGRESS started for upload: " . $uploadId . "\n", FILE_APPEND);
        
        if (function_exists('mftpLog')) {
            mftpLog(LOG_DEBUG, "ChunkedUpload: Progress phase for upload $uploadId, session ID: " . session_id());
        }
        
        // ENHANCED: Update session activity to prevent timeout
        if (MFTP_UPLOAD_SESSION_ACTIVITY_UPDATE && isset($_SESSION['_security'])) {
            $_SESSION['_security']['last_activity'] = time();
        }
        
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: About to get temp path from session for upload: " . $uploadId . "\n", FILE_APPEND);
        
        $tempFilePath = getChunkedTempPathFromSession($uploadId);
        $fallbackTempPath = sys_get_temp_dir() . '/mftp_chunked_' . $uploadId;
        
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: Got temp path: " . $tempFilePath . " for upload: " . $uploadId . "\n", FILE_APPEND);
        
        // ENHANCED: Use file locking with timeout to prevent race conditions
        $lockFile = $tempFilePath . '.lock';
        $lockHandle = fopen($lockFile, 'w');
        
        if ($lockHandle && flock($lockHandle, LOCK_EX | LOCK_NB)) {
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: Got lock for upload: " . $uploadId . "\n", FILE_APPEND);
            
            try {
                $fileSizeBefore = file_exists($tempFilePath) ? filesize($tempFilePath) : 0;
                
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: File size before readUpload: " . $fileSizeBefore . " bytes for upload: " . $uploadId . "\n", FILE_APPEND);
                
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_DEBUG, "ChunkedUpload: Progress - File size before readUpload: $fileSizeBefore bytes");
                }
                error_log("MFTP DEBUG: Progress - File size before readUpload: $fileSizeBefore bytes");
                
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: About to call readUpload for upload: " . $uploadId . "\n", FILE_APPEND);
                
                // Read upload data and append to both files for redundancy
                readUpload($tempFilePath, "a");
                
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: readUpload completed for upload: " . $uploadId . "\n", FILE_APPEND);
                
                $fileSizeAfter = file_exists($tempFilePath) ? filesize($tempFilePath) : 0;
                $bytesReceived = $fileSizeAfter - $fileSizeBefore;
                
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_DEBUG, "ChunkedUpload: Progress - File size after readUpload: $fileSizeAfter bytes (received: $bytesReceived bytes)");
                }
                error_log("MFTP DEBUG: Progress - File size after readUpload: $fileSizeAfter bytes (received: $bytesReceived bytes)");
                
                // Also maintain the fallback file
                if (file_exists($fallbackTempPath) && file_exists($tempFilePath)) {
                    file_put_contents($fallbackTempPath, file_get_contents($tempFilePath));
                }
                
                // Update session metadata
                $sessionKey = mftpGenerateChunkedSessionKey($uploadId);
                if (isset($_SESSION[$sessionKey . '_meta'])) {
                    $_SESSION[$sessionKey . '_meta']['last_progress'] = time();
                    $_SESSION[$sessionKey . '_meta']['status'] = 'progress';
                }
                
                // Session will remain active throughout the request to maintain consistency
                
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: PROGRESS COMPLETED successfully for upload: " . $uploadId . "\n", FILE_APPEND);
                
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_DEBUG, "ChunkedUpload: Progress completed for upload $uploadId");
                }
            } finally {
                flock($lockHandle, LOCK_UN);
                fclose($lockHandle);
                // Don't delete lock file here - keep it until finish
            }
        } else {
            // If we can't get a lock, fall back to direct write with retry
            if (function_exists('mftpLog')) {
                mftpLog(LOG_WARNING, "ChunkedUpload: Could not acquire lock for upload $uploadId, proceeding without lock");
            }
            
            // Try multiple times with small delay
            $retryCount = 0;
            while ($retryCount < MFTP_UPLOAD_RETRY_ATTEMPTS) {
                try {
                    readUpload($tempFilePath, "a");
                    
                    if (file_exists($fallbackTempPath) && file_exists($tempFilePath)) {
                        file_put_contents($fallbackTempPath, file_get_contents($tempFilePath));
                    }
                    break; // Success
                } catch (Exception $e) {
                    $retryCount++;
                    if ($retryCount < MFTP_UPLOAD_RETRY_ATTEMPTS) {
                        usleep(MFTP_UPLOAD_RETRY_DELAY);
                    } else {
                        throw new Exception("Failed to write upload data after " . MFTP_UPLOAD_RETRY_ATTEMPTS . " attempts: " . $e->getMessage());
                    }
                }
            }
            
            if ($lockHandle) {
                fclose($lockHandle);
            }
        }
    }

    function mftpChunkedUploadFinish($marshaller, $uploadId, $request) {
        // Debug: Log finish function entry
        $debugFile = dirname(__FILE__) . '/../logs/mftp_debug.log';
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: mftpChunkedUploadFinish() called for upload ID: " . $uploadId . "\n", FILE_APPEND);
        
        // Session is already active from the main script
        
        // Validate request structure
        InputValidator::validateApiRequest($request);
        
        // DEBUG: Log the original request
        if (function_exists('mftpLog')) {
            mftpLog(LOG_DEBUG, "ChunkedUpload: Original remote path: " . ($request['context']['remotePath'] ?? 'NOT_SET'));
        }
        
        // Validate remote path
        if (isset($request['context']['remotePath'])) {
            $validatedPath = InputValidator::validateFilePath($request['context']['remotePath'], true);
            $request['context']['remotePath'] = $validatedPath;
            
            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "ChunkedUpload: Validated remote path: " . $validatedPath);
            }
        }
        
        $sessionKey = mftpGenerateChunkedSessionKey($uploadId);
        if (function_exists('mftpLog')) {
            mftpLog(LOG_DEBUG, "ChunkedUpload: Finish phase for upload '$uploadId' (bytes: " . bin2hex($uploadId) . ") with session key $sessionKey");
        }
        
        // Debug: Log session key generation and current session state
        error_log("MFTP DEBUG: Finish phase - Generated session key: '$sessionKey' for upload ID: '$uploadId'");
        error_log("MFTP DEBUG: Finish phase - Session ID: " . session_id() . ", Session keys: " . implode(', ', array_keys($_SESSION)));
        error_log("MFTP DEBUG: Finish phase - Session key exists: " . (isset($_SESSION[$sessionKey]) ? 'YES' : 'NO'));

        $tempFilePath = getChunkedTempPathFromSession($uploadId);
        
        // Debug: Log temp file path
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: Temp file path: " . $tempFilePath . " for upload ID: " . $uploadId . "\n", FILE_APPEND);
        
        // Validate uploaded file with enhanced checks
        if (!file_exists($tempFilePath)) {
            $errorMsg = "Temporary file not found: $tempFilePath for upload $uploadId";
            if (function_exists('mftpLog')) {
                mftpLog(LOG_ERROR, "ChunkedUpload: $errorMsg");
                mftpLog(LOG_ERROR, "ChunkedUpload: Session key: $sessionKey");
                mftpLog(LOG_ERROR, "ChunkedUpload: Session data: " . json_encode($_SESSION[$sessionKey] ?? 'NOT_SET'));
            }
            throw new Exception($errorMsg);
        }
        
        $fileSize = filesize($tempFilePath);
        if ($fileSize === false || $fileSize === 0) {
            // Enhanced error reporting for file size issues
            $fileExists = file_exists($tempFilePath);
            $fileReadable = is_readable($tempFilePath);
            $fileWritable = is_writable($tempFilePath);
            $filePerms = $fileExists ? substr(sprintf('%o', fileperms($tempFilePath)), -4) : 'N/A';
            $fileMtime = $fileExists ? date('Y-m-d H:i:s', filemtime($tempFilePath)) : 'N/A';
            
            $errorMsg = "Invalid file size for upload $uploadId: $tempFilePath (size: $fileSize, exists: " . ($fileExists ? 'YES' : 'NO') . ", readable: " . ($fileReadable ? 'YES' : 'NO') . ", writable: " . ($fileWritable ? 'YES' : 'NO') . ", perms: $filePerms, modified: $fileMtime)";
            
            if (function_exists('mftpLog')) {
                mftpLog(LOG_ERROR, "ChunkedUpload: $errorMsg");
                mftpLog(LOG_ERROR, "ChunkedUpload: This usually indicates the upload was interrupted or the socket connection was closed");
            }
            
            // Check if this is a socket connection issue
            if ($fileSize === 0 && $fileExists) {
                $errorMsg = "Upload interrupted - file is empty (likely due to socket connection being closed by server). Upload ID: $uploadId";
            }
            
            throw new Exception($errorMsg);
        }
        
        InputValidator::validateFileUpload($tempFilePath, $request['context']['remotePath'], $fileSize);

        try {
            $request['context']['localPath'] = $tempFilePath;
            
            // Enhanced logging for debugging
            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "ChunkedUpload: Finish phase - Upload ID: $uploadId");
                mftpLog(LOG_DEBUG, "ChunkedUpload: Action name: " . ($request['actionName'] ?? 'NOT_SET'));
                mftpLog(LOG_DEBUG, "ChunkedUpload: Local path: $tempFilePath");
                mftpLog(LOG_DEBUG, "ChunkedUpload: Remote path: " . $request['context']['remotePath']);
                mftpLog(LOG_DEBUG, "ChunkedUpload: File exists: " . (file_exists($tempFilePath) ? 'YES' : 'NO'));
                mftpLog(LOG_DEBUG, "ChunkedUpload: File size: " . (file_exists($tempFilePath) ? filesize($tempFilePath) : 'N/A'));
            }

            try {
                // FORCE: Always use uploadFileToNewDirectory to ensure directory creation
                if (isset($request['actionName']) && $request['actionName'] !== "uploadArchive") {
                    $request['actionName'] = "uploadFileToNewDirectory";
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_DEBUG, "ChunkedUpload: Forced action to uploadFileToNewDirectory");
                    }
                }
                
                if ($request['actionName'] == "uploadArchive") {
                    $remotePath = $request['context']['remotePath'];

                    $ext = pathinfo($remotePath, PATHINFO_EXTENSION);
                    $newFilePath = monstaReplaceExtension($tempFilePath, $ext);
                    rename($tempFilePath, $newFilePath);
                    $tempFilePath = $newFilePath;
                    $request['context']['localPath'] = $tempFilePath;

                    $applicationSettings = new ApplicationSettings(APPLICATION_SETTINGS_PATH);
    
                    $extractor = new ArchiveExtractor($newFilePath, null, $applicationSettings->getSkipMacOsSpecialFiles());
    
                    $archiveFileCount = $extractor->getFileCount(); // will throw exception if it's not valid
    
                    $fileKey = generateRandomString(16);
    
                    $_SESSION[MFTP_SESSION_KEY_PREFIX . $fileKey] = array(
                        "archivePath" => $newFilePath,
                        "extractDirectory" => PathOperations::remoteDirname($request['context']['remotePath'])
                    );
    
                    $response = array(
                        "success" => true,
                        "fileKey" => $fileKey,
                        "fileCount" => $archiveFileCount
                    );
    
                    // Mark upload as completed in session metadata
                    if (isset($_SESSION[$sessionKey . '_meta'])) {
                        $_SESSION[$sessionKey . '_meta']['status'] = 'completed';
                        $_SESSION[$sessionKey . '_meta']['completed_at'] = time();
                    }
                    
                    // IMMEDIATE CLEANUP: Remove completed upload from session to prevent counting issues
                    unset($_SESSION[$sessionKey]);
                    unset($_SESSION[$sessionKey . '_meta']);
                    
                    // Remove concurrent upload lock file
                    $lockDir = sys_get_temp_dir() . '/mftp_concurrent_locks';
                    $concurrentLockFile = $lockDir . '/upload_' . $uploadId . '.lock';
                    if (file_exists($concurrentLockFile)) {
                        @unlink($concurrentLockFile);
                        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: Removed concurrent lock file: " . $concurrentLockFile . " for archive upload: " . $uploadId . "\n", FILE_APPEND);
                    }
                    
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_DEBUG, "ChunkedUpload: Immediately cleaned up completed archive upload $uploadId");
                    }
                    
                    // Session will remain active throughout the request to maintain consistency
    
                    print json_encode($response);
                } else {
                    // REVERTED: Use original marshaller approach with proper error handling
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_DEBUG, "ChunkedUpload: Using marshaller for $uploadId");
                    }
                    
                    try {
                        if (function_exists('mftpLog')) {
                            mftpLog(LOG_DEBUG, "ChunkedUpload: About to call marshaller->marshallRequest for upload $uploadId");
                            mftpLog(LOG_DEBUG, "ChunkedUpload: Request action: " . ($request['actionName'] ?? 'NOT_SET'));
                            mftpLog(LOG_DEBUG, "ChunkedUpload: Request context: " . json_encode($request['context'] ?? []));
                        }
                        
                        $result = $marshaller->marshallRequest($request);
                        $marshaller->disconnect();
                        
                        if (function_exists('mftpLog')) {
                            mftpLog(LOG_DEBUG, "ChunkedUpload: Marshaller completed successfully for upload $uploadId");
                        }
                        
                        // Mark upload as completed in session metadata
                        if (isset($_SESSION[$sessionKey . '_meta'])) {
                            $_SESSION[$sessionKey . '_meta']['status'] = 'completed';
                            $_SESSION[$sessionKey . '_meta']['completed_at'] = time();
                        }
                        
                        // IMMEDIATE CLEANUP: Remove completed upload from session to prevent counting issues
                        unset($_SESSION[$sessionKey]);
                        unset($_SESSION[$sessionKey . '_meta']);
                        
                        // Remove concurrent upload lock file
                        $lockDir = sys_get_temp_dir() . '/mftp_concurrent_locks';
                        $concurrentLockFile = $lockDir . '/upload_' . $uploadId . '.lock';
                        if (file_exists($concurrentLockFile)) {
                            @unlink($concurrentLockFile);
                            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: Removed concurrent lock file: " . $concurrentLockFile . " for upload: " . $uploadId . "\n", FILE_APPEND);
                        }
                        
                        if (function_exists('mftpLog')) {
                            mftpLog(LOG_DEBUG, "ChunkedUpload: Immediately cleaned up completed upload $uploadId");
                        }
                        
                        // Session will remain active throughout the request to maintain consistency
                        
                        print $result;
                    } catch (Exception $e) {
                        if (function_exists('mftpLog')) {
                            mftpLog(LOG_ERROR, "ChunkedUpload: Marshaller failed for upload $uploadId: " . $e->getMessage());
                            mftpLog(LOG_ERROR, "ChunkedUpload: Exception type: " . get_class($e));
                            mftpLog(LOG_ERROR, "ChunkedUpload: Exception trace: " . $e->getTraceAsString());
                        }
                        throw $e;
                    }
                }
            } catch (Exception $e) {
                cleanupTempTransferPath($tempFilePath);
                throw $e;
            }
        } catch (Exception $e) {
            if(file_exists($tempFilePath)){
                unlink($tempFilePath);
            }
            
            // Mark upload as failed before cleanup
            $sessionKey = mftpGenerateChunkedSessionKey($uploadId);
            if (isset($_SESSION[$sessionKey . '_meta'])) {
                $_SESSION[$sessionKey . '_meta']['status'] = 'failed';
                $_SESSION[$sessionKey . '_meta']['failed_at'] = time();
                $_SESSION[$sessionKey . '_meta']['error'] = $e->getMessage();
            }
            
            // Session will remain active throughout the request to maintain consistency
            
            // Clean up all possible session keys for this upload
            $keysToClean = [];
            $uploadKeyPrefix = 'UPLOAD_KEY_' . $uploadId;
            foreach ($_SESSION as $key => $value) {
                if (strpos($key, $uploadKeyPrefix) === 0) {
                    $keysToClean[] = $key;
                }
            }
            
            foreach ($keysToClean as $keyToClean) {
                unset($_SESSION[$keyToClean]);
                unset($_SESSION[$keyToClean . '_meta']);
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_DEBUG, "ChunkedUpload: Cleaned up session key $keyToClean for upload $uploadId");
                }
            }
            
            // Remove concurrent upload lock file on failure
            $lockDir = sys_get_temp_dir() . '/mftp_concurrent_locks';
            $concurrentLockFile = $lockDir . '/upload_' . $uploadId . '.lock';
            if (file_exists($concurrentLockFile)) {
                @unlink($concurrentLockFile);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: Removed concurrent lock file: " . $concurrentLockFile . " for failed upload: " . $uploadId . "\n", FILE_APPEND);
            }
            
            throw $e;
        }

        // Clean up fallback files with multiple possible patterns
        $fallbackPaths = [
            sys_get_temp_dir() . '/mftp_chunked_' . $uploadId,
            sys_get_temp_dir() . '/mftp_secure_' . $uploadId,
            getMonstaSharedTransferDirectory() . '/chunked_' . $uploadId
        ];
        
        foreach ($fallbackPaths as $fallbackTempPath) {
            if (file_exists($fallbackTempPath)) {
                unlink($fallbackTempPath);
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_DEBUG, "ChunkedUpload: Cleaned up fallback file $fallbackTempPath for upload $uploadId");
                }
            }
        }
        
        // Session cleanup is now handled immediately when upload completes
        // No need for additional cleanup here since we do immediate cleanup above
        
        // Clean up lock files
        $lockFile = $tempFilePath . '.lock';
        if (file_exists($lockFile)) {
            @unlink($lockFile);
            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "ChunkedUpload: Cleaned up lock file $lockFile for upload $uploadId");
            }
        }
        
        // Update session activity
        if (isset($_SESSION['_security'])) {
            $_SESSION['_security']['last_activity'] = time();
        }
        
        // Session will remain active throughout the request to maintain consistency
    }

    try {
        $debugFile = dirname(__FILE__) . '/../logs/mftp_debug.log';
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: Processing action: " . $mftpUploadAction . " for upload ID: " . $mftpUploadId . "\n", FILE_APPEND);
        
        if ($mftpUploadAction === "progress") {
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: PROGRESS action started for upload: " . $mftpUploadId . "\n", FILE_APPEND);
            
            // Clear any unexpected output before emitting JSON
            $unexpectedOutput = ob_get_contents();
            ob_end_clean();
            
            if (!empty($unexpectedOutput)) {
                error_log("MFTP ChunkedUpload: Unexpected output during progress: " . $unexpectedOutput);
            }
            
            // Set proper JSON headers
            if (!headers_sent()) {
                header('Content-Type: application/json');
            }
            
            mftpChunkedUploadProgress($mftpUploadId);
            echo json_encode(['success' => true]);
            
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: PROGRESS action completed for upload: " . $mftpUploadId . "\n", FILE_APPEND);

        } else if ($mftpUploadAction === "initiate" || $mftpUploadAction === "finish") {
            $marshaller = new RequestMarshaller(null, true); // Pass true for chunked upload context
            if (!isset($_POST['request'])) {
                throw new InvalidArgumentException("Missing 'request' parameter in POST data");
            }
            $request = json_decode($_POST['request'], true);
            
            // Validate JSON parsing
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new InvalidArgumentException("Invalid JSON in chunked upload request: " . json_last_error_msg());
            }

            if ($mftpUploadAction === "initiate") {
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: INITIATE action started for upload: " . $mftpUploadId . "\n", FILE_APPEND);
                
                // Clear any unexpected output before emitting JSON
                $unexpectedOutput = ob_get_contents();
                ob_end_clean();
                
                if (!empty($unexpectedOutput)) {
                    error_log("MFTP ChunkedUpload: Unexpected output during initiate: " . $unexpectedOutput);
                }
                
                // Set proper JSON headers
                if (!headers_sent()) {
                    header('Content-Type: application/json');
                }
                
                mftpChunkedUploadInitiate($marshaller, $mftpUploadId, $request);
                echo json_encode(['success' => true]);
                
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: INITIATE action completed for upload: " . $mftpUploadId . "\n", FILE_APPEND);
            } else {
                // Debug: Log finish action
                $debugFile = dirname(__FILE__) . '/../logs/mftp_debug.log';
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: FINISH action for upload ID: " . $mftpUploadId . "\n", FILE_APPEND);
                
                // Clear any unexpected output before emitting JSON  
                $unexpectedOutput = ob_get_contents();
                ob_end_clean();
                
                if (!empty($unexpectedOutput)) {
                    error_log("MFTP ChunkedUpload: Unexpected output during finish: " . $unexpectedOutput);
                }
                
                // Set proper JSON headers
                if (!headers_sent()) {
                    header('Content-Type: application/json');
                }
                
                mftpChunkedUploadFinish($marshaller, $mftpUploadId, $request);
                
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: FINISH action completed for upload: " . $mftpUploadId . "\n", FILE_APPEND);
            }
        } else {
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: UNKNOWN action: " . $mftpUploadAction . " for upload: " . $mftpUploadId . "\n", FILE_APPEND);
            throw new Exception("Unknown action $mftpUploadAction.");
        }
    } catch (Exception $e) {
        $debugFile = dirname(__FILE__) . '/../logs/mftp_debug.log';
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: EXCEPTION caught for upload: " . $mftpUploadId . " - " . $e->getMessage() . "\n", FILE_APPEND);
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: Exception type: " . get_class($e) . "\n", FILE_APPEND);
        // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] UPLOAD DEBUG: Exception trace: " . $e->getTraceAsString() . "\n", FILE_APPEND);
        
        // Clean any output buffer before handling the exception
        while (ob_get_level() > 0) { @ob_end_clean(); }
        handleExceptionInRequest($e);
    }
    
    // Clear upload processing flag at the end
    if (isset($_SESSION['_upload_processing'])) {
        unset($_SESSION['_upload_processing']);
    }
