<?php

require_once(dirname(__FILE__) . "/../constants.php");

class SecureSessionManager {
    private static $instance = null;
    private $isInitialized = false;
    private $sessionConfig = []; // Will be set in constructor based on environment

    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        // Private constructor for singleton
        // Configure session settings based on environment and security requirements
        $this->initializeSecureSessionConfig();
    }

    /**
     * Initialize secure session configuration based on environment and security audit recommendations
     */
    private function initializeSecureSessionConfig() {
        $environment = $this->detectEnvironment();
        $isLocalhost = $this->isLocalhost();
        $isHttps = $this->isHttps();
        
        // Security audit recommendations implementation
        switch ($environment) {
            case 'development':
                $this->sessionConfig = [
                    'cookie_lifetime' => 0, // Session cookie only
                    'cookie_secure' => false, // Allow HTTP for development
                    'cookie_httponly' => true, // Always prevent XSS
                    'cookie_samesite' => 'Lax', // Relaxed for development
                    'use_strict_mode' => true, // Always reject uninitialized session IDs
                    'sid_length' => 48, // Longer session IDs
                    'sid_bits_per_character' => 6, // More entropy
                    'gc_maxlifetime' => defined('MFTP_SESSION_TIMEOUT') ? MFTP_SESSION_TIMEOUT : 3600, // Configurable, default 1 hour for dev
                    'gc_probability' => 1,
                    'gc_divisor' => 100,
                    'security_level' => 'development'
                ];
                break;
                
            case 'staging':
                $this->sessionConfig = [
                    'cookie_lifetime' => 0, // Session cookie only
                    'cookie_secure' => true, // SECURITY AUDIT: Always require HTTPS in staging/production
                    'cookie_httponly' => true, // Always prevent XSS
                    'cookie_samesite' => 'Strict', // SECURITY AUDIT: Always use Strict in production environments
                    'use_strict_mode' => true, // Always reject uninitialized session IDs
                    'sid_length' => 48, // Longer session IDs
                    'sid_bits_per_character' => 6, // More entropy
                    'gc_maxlifetime' => defined('MFTP_SESSION_TIMEOUT') ? MFTP_SESSION_TIMEOUT : 2700, // 45 minutes for staging
                    'gc_probability' => 2, // More frequent cleanup
                    'gc_divisor' => 100,
                    'security_level' => 'staging'
                ];
                break;
                
            case 'production':
            default:
                $this->sessionConfig = [
                    'cookie_lifetime' => 0, // Session cookie only
                    'cookie_secure' => true, // SECURITY AUDIT: Always require HTTPS in production
                    'cookie_httponly' => true, // Always prevent XSS
                    'cookie_samesite' => 'Strict', // SECURITY AUDIT: Always use Strict in production
                    'use_strict_mode' => true, // Always reject uninitialized session IDs
                    'sid_length' => 48, // Longer session IDs
                    'sid_bits_per_character' => 6, // More entropy
                    'gc_maxlifetime' => defined('MFTP_SESSION_TIMEOUT') ? MFTP_SESSION_TIMEOUT : 1800, // SECURITY AUDIT: 30 minutes (1800 seconds) for production
                    'gc_probability' => 5, // More frequent cleanup in production
                    'gc_divisor' => 100,
                    'security_level' => 'production'
                ];
                break;
        }
        
        // Allow override via configuration constants
        if (defined('MFTP_FORCE_SECURE_COOKIES') && MFTP_FORCE_SECURE_COOKIES) {
            $this->sessionConfig['cookie_secure'] = true;
        }
        
        if (defined('MFTP_FORCE_STRICT_SAMESITE') && MFTP_FORCE_STRICT_SAMESITE) {
            $this->sessionConfig['cookie_samesite'] = 'Strict';
        }
        
        // Log security configuration for audit trail
        if (function_exists('mftpLog')) {
            mftpLog(LOG_INFO, "SecureSessionManager: Initialized for {$environment} environment with " . 
                   ($this->sessionConfig['gc_maxlifetime'] / 60) . " minute timeout, " .
                   "secure=" . ($this->sessionConfig['cookie_secure'] ? 'true' : 'false') . ", " .
                   "samesite=" . $this->sessionConfig['cookie_samesite']);
        }
    }

    /**
     * Detect environment based on various indicators
     */
    private function detectEnvironment(): string {
        // Check for explicit environment configuration
        if (defined('MFTP_ENVIRONMENT')) {
            return strtolower(MFTP_ENVIRONMENT);
        }
        
        // Check environment variables
        $env = getenv('MFTP_ENV') ?: getenv('APP_ENV') ?: getenv('ENVIRONMENT');
        if ($env) {
            return strtolower($env);
        }
        
        // Auto-detect based on domain and host patterns
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        
        // Development indicators
        if ($this->isLocalhost() || 
            strpos($host, 'dev.') === 0 || 
            strpos($host, 'localhost') !== false ||
            strpos($host, '127.0.0.1') !== false ||
            strpos($host, '.local') !== false) {
            return 'development';
        }
        
        // Staging indicators
        if (strpos($host, 'staging.') === 0 || 
            strpos($host, 'test.') === 0 ||
            strpos($host, 'qa.') === 0) {
            return 'staging';
        }
        
        // Default to production for security
        return 'production';
    }

    /**
     * Configure session cookie for "remember me" BEFORE starting session
     * This must be called before initializeSession() to set cookie lifetime
     */
    public function configureRememberMeCookie($lifetimeSeconds = 2592000) {
        // Update session config to use extended lifetime
        $this->sessionConfig['cookie_lifetime'] = $lifetimeSeconds;
        
        // Configure session cookie params with extended lifetime
        $environment = $this->sessionConfig['security_level'] ?? 'production';
        $cookieParams = [
            'lifetime' => $lifetimeSeconds,
            'path' => '/',
            'domain' => $this->getSecureDomainForEnvironment($environment),
            'secure' => $this->sessionConfig['cookie_secure'],
            'httponly' => $this->sessionConfig['cookie_httponly'],
            'samesite' => $this->sessionConfig['cookie_samesite']
        ];
        
        // Apply environment-specific overrides
        if ($environment === 'development' && $this->isLocalhost()) {
            $cookieParams['domain'] = '';
            $cookieParams['secure'] = false;
        } elseif ($environment === 'development' && !$this->isHttps()) {
            $cookieParams['secure'] = false;
        }
        
        // Set cookie parameters BEFORE session_start()
        if (PHP_VERSION_ID >= 70300) {
            session_set_cookie_params([
                'lifetime' => $cookieParams['lifetime'],
                'path' => $cookieParams['path'],
                'domain' => $cookieParams['domain'],
                'secure' => $cookieParams['secure'],
                'httponly' => $cookieParams['httponly'],
                'samesite' => $cookieParams['samesite']
            ]);
        } else {
            session_set_cookie_params(
                $cookieParams['lifetime'],
                $cookieParams['path'],
                $cookieParams['domain'],
                $cookieParams['secure'],
                $cookieParams['httponly']
            );
        }
        
        if (function_exists('mftpLog')) {
            mftpLog(LOG_INFO, "SecureSessionManager: Configured remember me cookie with " . 
                   ($lifetimeSeconds / 86400) . " day lifetime");
        }
    }

    public function initializeSession($regenerateId = false) {
        if ($this->isInitialized && !$regenerateId) {
            return true;
        }

        // Don't start session in unit test mode
        if (defined("MONSTA_UNIT_TEST_MODE")) {
            return false;
        }

        // Check if session is already active
        if (session_status() === PHP_SESSION_ACTIVE) {
            if ($regenerateId) {
                $this->regenerateSessionId();
            }
            return true;
        }

        // Configure session security settings
        $this->configureSessionSecurity();

        // Start the session
        if (!session_start()) {
            throw new Exception("Failed to start secure session");
        }

        $this->isInitialized = true;

        // Regenerate session ID if requested
        if ($regenerateId) {
            $this->regenerateSessionId();
        }

        // Initialize session security tracking
        $this->initializeSessionSecurity();

        return true;
    }

    private function configureSessionSecurity() {
        // Enhanced session security configuration based on environment
        $environment = $this->sessionConfig['security_level'] ?? 'production';
        
        // Check if cookie params are already set (e.g., by configureRememberMeCookie)
        $existingParams = session_get_cookie_params();
        $lifetimeAlreadySet = $existingParams['lifetime'] > 0 && $this->sessionConfig['cookie_lifetime'] > 0;
        
        // Configure session cookie parameters based on security level
        $cookieParams = [
            'lifetime' => $lifetimeAlreadySet ? $this->sessionConfig['cookie_lifetime'] : $this->sessionConfig['cookie_lifetime'],
            'path' => '/',
            'domain' => $this->getSecureDomainForEnvironment($environment),
            'secure' => $this->sessionConfig['cookie_secure'],
            'httponly' => $this->sessionConfig['cookie_httponly'],
            'samesite' => $this->sessionConfig['cookie_samesite']
        ];
        
        // Preserve remember me cookie lifetime if already configured
        if ($lifetimeAlreadySet && $this->sessionConfig['cookie_lifetime'] > 0) {
            $cookieParams['lifetime'] = $this->sessionConfig['cookie_lifetime'];
        }
        
        // Apply environment-specific overrides
        if ($environment === 'development' && $this->isLocalhost()) {
            // Development localhost - allow some flexibility
            $cookieParams['domain'] = ''; // Empty domain for localhost
            $cookieParams['secure'] = false; // Allow HTTP for local development
        } elseif ($environment === 'development' && !$this->isHttps()) {
            // Development non-localhost but no HTTPS - log warning but continue
            if (function_exists('mftpLog')) {
                mftpLog(LOG_WARNING, "SecureSessionManager: Development environment without HTTPS detected. Consider using HTTPS even in development.");
            }
            $cookieParams['secure'] = false;
        }
        
        // Only set cookie parameters if they haven't been set already by configureRememberMeCookie
        // Check if session cookie params are at default (lifetime = 0)
        if (!$lifetimeAlreadySet || $existingParams['lifetime'] == 0) {
            if (PHP_VERSION_ID >= 70300) {
                session_set_cookie_params([
                    'lifetime' => $cookieParams['lifetime'],
                    'path' => $cookieParams['path'],
                    'domain' => $cookieParams['domain'],
                    'secure' => $cookieParams['secure'],
                    'httponly' => $cookieParams['httponly'],
                    'samesite' => $cookieParams['samesite']
                ]);
            } else {
                session_set_cookie_params(
                    $cookieParams['lifetime'],
                    $cookieParams['path'],
                    $cookieParams['domain'],
                    $cookieParams['secure'],
                    $cookieParams['httponly']
                );
            }
        }
        
        // Log cookie configuration for security audit trail
        if (function_exists('mftpLog')) {
            mftpLog(LOG_DEBUG, "SecureSessionManager: Cookie params set - " .
                   "secure=" . ($cookieParams['secure'] ? 'true' : 'false') . ", " .
                   "httponly=" . ($cookieParams['httponly'] ? 'true' : 'false') . ", " .
                   "samesite=" . $cookieParams['samesite'] . ", " .
                   "domain=" . ($cookieParams['domain'] ?: 'default'));
        }

        // Set session configuration with error checking
        if (ini_set('session.use_strict_mode', $this->sessionConfig['use_strict_mode']) === false) {
            error_log("Warning: Failed to set session.use_strict_mode. Session security may be compromised.");
        }
        // Note: session.sid_length and session.sid_bits_per_character are deprecated in PHP 8.4+
        if (ini_set('session.gc_maxlifetime', $this->sessionConfig['gc_maxlifetime']) === false) {
            error_log("Warning: Failed to set session.gc_maxlifetime. Session cleanup may not work properly.");
        }
        if (ini_set('session.gc_probability', $this->sessionConfig['gc_probability']) === false) {
            error_log("Warning: Failed to set session.gc_probability. Session cleanup frequency may be affected.");
        }
        if (ini_set('session.gc_divisor', $this->sessionConfig['gc_divisor']) === false) {
            error_log("Warning: Failed to set session.gc_divisor. Session cleanup frequency may be affected.");
        }
        
        // Use only cookies, no URL parameters
        if (ini_set('session.use_only_cookies', 1) === false) {
            error_log("Warning: Failed to set session.use_only_cookies. Session security may be compromised.");
        }
        if (ini_set('session.use_trans_sid', 0) === false) {
            error_log("Warning: Failed to set session.use_trans_sid. Session security may be compromised.");
        }
    }

    private function initializeSessionSecurity() {
        // Initialize session security tracking if not exists
        if (!isset($_SESSION['_security'])) {
            $_SESSION['_security'] = [
                'created_at' => time(),
                'last_activity' => time(),
                'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
            ];
        } else {
            // If session exists, preserve remember_me settings if they were set
            // This ensures "remember me" persists across page loads
            if (isset($_SESSION['_security']['remember_me']) && $_SESSION['_security']['remember_me'] === true) {
                // Keep the remember_me flag and lifetime
                // Don't overwrite them
            }
        }

        // Update last activity
        $_SESSION['_security']['last_activity'] = time();
        
        // Validate session security - don't throw exceptions to prevent breaking page load
        // The frontend will handle authentication if needed
        try {
            $this->validateSessionSecurity();
        } catch (Exception $e) {
            // Log the validation failure but don't break the page
            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "Session security validation: " . $e->getMessage() . " - Allowing page load to continue");
            }
            // Don't re-throw - let the page load continue
            // The frontend will check authentication state and prompt login if needed
        }
    }

    private function validateSessionSecurity() {
        if (!isset($_SESSION['_security'])) {
            $this->destroySession();
            throw new Exception("Invalid session security data");
        }

        $security = $_SESSION['_security'];
        
        // Check session timeout
        // Use extended lifetime if "remember me" is enabled
        if (isset($security['remember_me']) && $security['remember_me'] === true && isset($security['remember_me_lifetime'])) {
            $maxLifetime = $security['remember_me_lifetime'];
        } else {
            $maxLifetime = $this->sessionConfig['gc_maxlifetime'];
        }
        
        if ((time() - $security['last_activity']) > $maxLifetime) {
            $this->destroySession();
            throw new Exception("Session expired");
        }

        // Check IP address consistency (optional, can be disabled for proxy environments)
        $currentIp = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        if ($security['ip_address'] !== $currentIp && !$this->isProxyEnvironment()) {
            $this->destroySession();
            throw new Exception("Session IP mismatch detected");
        }
    }

    public function regenerateSessionId($deleteOldSession = true) {
        if (session_status() !== PHP_SESSION_ACTIVE) {
            throw new Exception("Cannot regenerate session ID: no active session");
        }

        // Check if remember_me is enabled BEFORE regeneration
        $rememberMe = isset($_SESSION['_security']['remember_me']) && $_SESSION['_security']['remember_me'] === true;
        $rememberMeLifetime = $_SESSION['_security']['remember_me_lifetime'] ?? null;

        // Regenerate session ID
        // NOTE: PHP's session_regenerate_id() will automatically send a Set-Cookie header
        // with the NEW session ID, but it uses DEFAULT cookie params (lifetime=0 = Session)
        // We need to immediately extend it if remember_me is enabled
        if (!session_regenerate_id($deleteOldSession)) {
            throw new Exception("Failed to regenerate session ID");
        }

        // Update security tracking - preserve remember_me settings
        if (isset($_SESSION['_security'])) {
            $_SESSION['_security']['created_at'] = time();
            
            // Preserve remember_me settings after regeneration
            if ($rememberMe) {
                $_SESSION['_security']['remember_me'] = true;
                if ($rememberMeLifetime !== null) {
                    $_SESSION['_security']['remember_me_lifetime'] = $rememberMeLifetime;
                }
            }
        }

        // If remember_me is enabled, immediately extend the cookie with the NEW session ID
        // This must happen right after regeneration, before any other output
        if ($rememberMe && $rememberMeLifetime && !headers_sent()) {
            try {
                // Use the remembered lifetime
                $this->extendSessionCookie($rememberMeLifetime);
            } catch (Exception $e) {
                // Log but don't fail - the cookie extension will be called again after auth
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_DEBUG, "Failed to extend cookie during session regeneration: " . $e->getMessage());
                }
            }
        }

        return session_id();
    }

    /**
     * Extend session cookie lifetime for "remember me" functionality
     * @param int $lifetimeSeconds Number of seconds for cookie to persist (default: 30 days)
     */
    public function extendSessionCookie($lifetimeSeconds = 2592000) {
        if (session_status() !== PHP_SESSION_ACTIVE) {
            throw new Exception("Cannot extend session cookie: no active session");
        }

        // Check if headers have been sent - if so, we can't modify cookies
        if (headers_sent($file, $line)) {
            if (function_exists('mftpLog')) {
                mftpLog(LOG_WARNING, "Cannot extend session cookie: headers already sent in $file:$line");
            }
            return false;
        }

        $cookieParams = session_get_cookie_params();
        
        // Set new cookie with extended lifetime
        $expireTime = time() + $lifetimeSeconds;
        
        // Update session config to reflect the extended lifetime
        $this->sessionConfig['cookie_lifetime'] = $lifetimeSeconds;
        
        // Mark session as "remember me" enabled so validation uses extended timeout
        if (!isset($_SESSION['_security'])) {
            $_SESSION['_security'] = [];
        }
        $_SESSION['_security']['remember_me'] = true;
        $_SESSION['_security']['remember_me_lifetime'] = $lifetimeSeconds;
        
        // Get current session ID and cookie name
        $cookieName = session_name();
        $cookieValue = session_id();
        
        // Get the actual cookie domain/path from session params
        // CRITICAL: Use EXACTLY what session_get_cookie_params() returns
        // The browser requires exact match of domain, path, secure, httponly, and samesite
        $actualDomain = $cookieParams['domain'];
        $actualPath = $cookieParams['path'] ?: '/';
        
        // CRITICAL: Match the exact domain that was used when the cookie was first set
        // The cookie shows "www.monstaftp.com" (no leading dot) in the browser
        // If cookie params return empty or different domain, use HTTP_HOST
        if (empty($actualDomain) || ($actualDomain && strpos($actualDomain, '.') === 0)) {
            // Get domain from HTTP_HOST to match what was likely used originally
            $host = $_SERVER['HTTP_HOST'] ?? '';
            $host = explode(':', $host)[0]; // Remove port
            // Use exact host match (no leading dot) to match existing cookie
            // This ensures browser accepts the new cookie as replacement
            $actualDomain = $host;
            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "SecureSessionManager: Using HTTP_HOST domain: '" . $actualDomain . "' to match existing cookie");
            }
        }
        
        // Log the exact domain/path we're using for debugging
        if (function_exists('mftpLog')) {
            mftpLog(LOG_DEBUG, "SecureSessionManager: Setting cookie - domain: '" . ($actualDomain ?: 'empty') . 
                   "', path: '" . $actualPath . "', secure: " . ($cookieParams['secure'] ? 'true' : 'false') . 
                   ", expires: " . date('Y-m-d H:i:s', $expireTime) . ", session_id: " . substr($cookieValue, 0, 8) . "...");
        }
        
        // Don't delete the old cookie - just set the new one with extended lifetime
        // Setting a cookie with the same name/domain/path/secure/httponly will replace it
        // Deleting first can cause race conditions in some browsers
        
        // Now set the new cookie with extended lifetime
        // This will override the session cookie
        if (PHP_VERSION_ID >= 70300) {
            $success = setcookie(
                $cookieName,
                $cookieValue,
                [
                    'expires' => $expireTime,
                    'path' => $actualPath,
                    'domain' => $actualDomain,
                    'secure' => $cookieParams['secure'],
                    'httponly' => $cookieParams['httponly'],
                    'samesite' => $cookieParams['samesite'] ?? 'Lax'
                ]
            );
        } else {
            // For PHP < 7.3, use traditional setcookie (samesite handled via header)
            $success = setcookie(
                $cookieName,
                $cookieValue,
                $expireTime,
                $actualPath . '; SameSite=' . ($cookieParams['samesite'] ?? 'Lax'),
                $actualDomain,
                $cookieParams['secure'],
                $cookieParams['httponly']
            );
        }
        
        if ($success) {
            if (function_exists('mftpLog')) {
                mftpLog(LOG_INFO, "SecureSessionManager: Extended session cookie lifetime to " . 
                       ($lifetimeSeconds / 86400) . " days for remember me (cookie expires: " . date('Y-m-d H:i:s', $expireTime) . 
                       ", domain: '" . ($actualDomain ?: 'default') . "', path: '" . $actualPath . 
                       "', secure: " . ($cookieParams['secure'] ? 'true' : 'false') . 
                       ", httponly: " . ($cookieParams['httponly'] ? 'true' : 'false') . 
                       ", samesite: " . ($cookieParams['samesite'] ?? 'Lax'));
            }
        } else {
            if (function_exists('mftpLog')) {
                $headersSent = headers_sent($file, $line);
                mftpLog(LOG_WARNING, "SecureSessionManager: Failed to set extended session cookie - headers_sent: " . 
                       ($headersSent ? "yes in $file:$line" : 'no') . 
                       ", domain: '" . ($actualDomain ?: 'default') . "', path: '" . $actualPath . "'");
            }
        }
        
        // Also update session cookie params for future session operations
        // Only update if session is not yet active (session_set_cookie_params can't be called when session is active)
        if (session_status() !== PHP_SESSION_ACTIVE) {
            if (PHP_VERSION_ID >= 70300) {
                session_set_cookie_params([
                    'lifetime' => $lifetimeSeconds,
                    'path' => $actualPath,
                    'domain' => $actualDomain,
                    'secure' => $cookieParams['secure'],
                    'httponly' => $cookieParams['httponly'],
                    'samesite' => $cookieParams['samesite'] ?? 'Lax'
                ]);
            } else {
                session_set_cookie_params(
                    $lifetimeSeconds,
                    $actualPath,
                    $actualDomain,
                    $cookieParams['secure'],
                    $cookieParams['httponly']
                );
            }
        }
        
        return $success;
    }

    public function destroySession() {
        if (session_status() === PHP_SESSION_ACTIVE) {
            // Clear session data
            $_SESSION = [];
            
            // Delete session cookie
            if (isset($_COOKIE[session_name()])) {
                $params = session_get_cookie_params();
                setcookie(
                    session_name(),
                    '',
                    time() - 42000,
                    $params['path'],
                    $params['domain'],
                    $params['secure'],
                    $params['httponly']
                );
            }
            
            // Destroy session
            session_destroy();
        }
        
        $this->isInitialized = false;
    }

    public function isActive() {
        return session_status() === PHP_SESSION_ACTIVE && $this->isInitialized;
    }

    public function getSessionInfo() {
        if (!$this->isActive()) {
            return null;
        }

        return [
            'id' => session_id(),
            'security' => $_SESSION['_security'] ?? null,
            'data_keys' => array_keys($_SESSION)
        ];
    }

    private function isHttps() {
        return isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' ||
               isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https' ||
               isset($_SERVER['HTTP_X_FORWARDED_SSL']) && $_SERVER['HTTP_X_FORWARDED_SSL'] === 'on';
    }

    private function getSecureDomain() {
        // For development/localhost, return empty string to use default
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        if ($host === 'localhost' || preg_match('/^127\.0\.0\./', $host) || preg_match('/^192\.168\./', $host)) {
            return '';
        }
        return $host;
    }

    /**
     * Get secure domain based on environment
     */
    private function getSecureDomainForEnvironment(string $environment): string {
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        
        // For development environments, be more flexible
        if ($environment === 'development') {
            if ($this->isLocalhost() || 
                strpos($host, '.local') !== false ||
                preg_match('/^192\.168\./', $host) ||
                preg_match('/^10\./', $host)) {
                return ''; // Empty domain for local development
            }
        }
        
        // For staging/production, use the actual host
        // Remove port number if present
        $domain = explode(':', $host)[0];
        
        // Validate domain format for security
        if (!preg_match('/^[a-zA-Z0-9.-]+$/', $domain)) {
            if (function_exists('mftpLog')) {
                mftpLog(LOG_WARNING, "SecureSessionManager: Invalid domain format detected: " . $domain);
            }
            return '';
        }
        
        return $domain;
    }

    private function isProxyEnvironment() {
        // Check for common proxy headers
        return isset($_SERVER['HTTP_X_FORWARDED_FOR']) ||
               isset($_SERVER['HTTP_X_REAL_IP']) ||
               isset($_SERVER['HTTP_CF_CONNECTING_IP']);
    }
    
    private function isLocalhost() {
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        return $host === 'localhost' || preg_match('/^127\.0\.0\./', $host) || preg_match('/^192\.168\./', $host);
    }

    // Generate CSRF token
    public function generateCSRFToken() {
        if (!$this->isActive()) {
            throw new Exception("Cannot generate CSRF token: no active session");
        }

        $token = bin2hex(random_bytes(32));
        $_SESSION['_csrf_token'] = $token;
        return $token;
    }

    // Validate CSRF token
    public function validateCSRFToken($token) {
        if (!$this->isActive()) {
            return false;
        }

        if (!isset($_SESSION['_csrf_token'])) {
            
            // AUTO-FIX: Generate a new token if missing (for session stability)
            if ($token && is_string($token) && strlen($token) === 64) {
                $_SESSION['_csrf_token'] = $token;
                return true;
            }
            
            return false;
        }

        // Ensure token is a string before comparison
        if ($token === null || !is_string($token)) {
            return false;
        }

        $sessionToken = $_SESSION['_csrf_token'];
        return hash_equals($sessionToken, $token);
    }

    // Get current CSRF token
    public function getCSRFToken() {
        if (!$this->isActive()) {
            return null;
        }

        if (!isset($_SESSION['_csrf_token'])) {
            return $this->generateCSRFToken();
        }

        return $_SESSION['_csrf_token'];
    }

    /**
     * Get current session security configuration
     */
    public function getSecurityConfig(): array {
        return [
            'environment' => $this->sessionConfig['security_level'] ?? 'unknown',
            'cookie_secure' => $this->sessionConfig['cookie_secure'] ?? false,
            'cookie_samesite' => $this->sessionConfig['cookie_samesite'] ?? 'None',
            'session_timeout_minutes' => ($this->sessionConfig['gc_maxlifetime'] ?? 3600) / 60,
            'session_timeout_seconds' => $this->sessionConfig['gc_maxlifetime'] ?? 3600,
            'use_strict_mode' => $this->sessionConfig['use_strict_mode'] ?? false,
            'is_https' => $this->isHttps(),
            'is_localhost' => $this->isLocalhost(),
            'current_domain' => $_SERVER['HTTP_HOST'] ?? 'unknown'
        ];
    }

    /**
     * Validate current session security settings against requirements
     */
    public function validateSecuritySettings(): array {
        $config = $this->getSecurityConfig();
        $environment = $config['environment'];
        $issues = [];
        $recommendations = [];

        // Validate production security requirements
        if ($environment === 'production') {
            if (!$config['cookie_secure']) {
                $issues[] = 'CRITICAL: cookie_secure should be true in production';
                $recommendations[] = 'Set MFTP_FORCE_SECURE_COOKIES=true or ensure HTTPS is properly configured';
            }
            
            if ($config['cookie_samesite'] !== 'Strict') {
                $issues[] = 'HIGH: cookie_samesite should be Strict in production';
                $recommendations[] = 'Set MFTP_FORCE_STRICT_SAMESITE=true';
            }
            
            if ($config['session_timeout_minutes'] > 30) {
                $issues[] = 'MEDIUM: Session timeout exceeds recommended 30 minutes for production';
                $recommendations[] = 'Set MFTP_SESSION_TIMEOUT=1800 for 30-minute sessions';
            }
            
            if (!$config['is_https']) {
                $issues[] = 'CRITICAL: HTTPS is not enabled in production environment';
                $recommendations[] = 'Configure HTTPS/SSL for production deployment';
            }
        }

        // Validate staging security requirements
        if ($environment === 'staging') {
            if (!$config['cookie_secure']) {
                $issues[] = 'HIGH: cookie_secure should be true in staging';
            }
            
            if ($config['session_timeout_minutes'] > 45) {
                $issues[] = 'LOW: Session timeout exceeds recommended 45 minutes for staging';
            }
        }

        // General security validations
        if (!$config['use_strict_mode']) {
            $issues[] = 'MEDIUM: use_strict_mode should always be enabled';
        }

        return [
            'environment' => $environment,
            'is_secure' => empty($issues),
            'security_score' => max(0, 100 - (count($issues) * 15)), // Rough security score
            'issues' => $issues,
            'recommendations' => $recommendations,
            'config' => $config
        ];
    }

    /**
     * Apply security headers for enhanced protection
     */
    public function applySecurityHeaders(): void {
        if (headers_sent()) {
            return; // Headers already sent, cannot modify
        }

        $environment = $this->sessionConfig['security_level'] ?? 'production';
        
        // Apply security headers based on environment
        if ($environment === 'production') {
            // Strict security headers for production
            header('X-Frame-Options: DENY');
            header('X-Content-Type-Options: nosniff');
            header('X-XSS-Protection: 1; mode=block');
            header('Referrer-Policy: strict-origin-when-cross-origin');
            
            if ($this->isHttps()) {
                header('Strict-Transport-Security: max-age=31536000; includeSubDomains; preload');
            }
            
        } elseif ($environment === 'staging') {
            // Moderate security headers for staging
            header('X-Frame-Options: SAMEORIGIN');
            header('X-Content-Type-Options: nosniff');
            header('X-XSS-Protection: 1; mode=block');
            header('Referrer-Policy: strict-origin-when-cross-origin');
            
        } else {
            // Minimal security headers for development
            header('X-Content-Type-Options: nosniff');
            header('X-Frame-Options: SAMEORIGIN');
        }

        // Always set cache control for session pages
        header('Cache-Control: no-cache, no-store, must-revalidate');
        header('Pragma: no-cache');
        header('Expires: 0');
    }

    /**
     * Get environment detection information for debugging
     */
    public function getEnvironmentInfo(): array {
        return [
            'detected_environment' => $this->detectEnvironment(),
            'is_localhost' => $this->isLocalhost(),
            'is_https' => $this->isHttps(),
            'is_proxy_environment' => $this->isProxyEnvironment(),
            'host' => $_SERVER['HTTP_HOST'] ?? 'unknown',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
            'remote_addr' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            'environment_variables' => [
                'MFTP_ENVIRONMENT' => defined('MFTP_ENVIRONMENT') ? MFTP_ENVIRONMENT : 'not set',
                'MFTP_ENV' => getenv('MFTP_ENV') ?: 'not set',
                'APP_ENV' => getenv('APP_ENV') ?: 'not set',
                'ENVIRONMENT' => getenv('ENVIRONMENT') ?: 'not set'
            ]
        ];
    }

    /**
     * Force session security settings (for emergency security upgrades)
     */
    public function forceSecurityMode(string $mode = 'maximum'): void {
        switch ($mode) {
            case 'maximum':
                $this->sessionConfig['cookie_secure'] = true;
                $this->sessionConfig['cookie_samesite'] = 'Strict';
                $this->sessionConfig['gc_maxlifetime'] = 900; // 15 minutes
                break;
                
            case 'strict':
                $this->sessionConfig['cookie_secure'] = true;
                $this->sessionConfig['cookie_samesite'] = 'Strict';
                $this->sessionConfig['gc_maxlifetime'] = 1800; // 30 minutes
                break;
                
            case 'moderate':
                $this->sessionConfig['cookie_secure'] = $this->isHttps();
                $this->sessionConfig['cookie_samesite'] = 'Strict';
                $this->sessionConfig['gc_maxlifetime'] = 3600; // 1 hour
                break;
        }
        
        if (function_exists('mftpLog')) {
            mftpLog(LOG_WARNING, "SecureSessionManager: Forced security mode '{$mode}' applied");
        }
    }
} 