<?php

    if (!function_exists('hex2bin')) {
        function hex2bin($str) {
            $sbin = "";
            $len = strlen($str);
            for ($i = 0; $i < $len; $i += 2) {
                $sbin .= pack("H*", substr($str, $i, 2));
            }

            return $sbin;
        }
    }

    class EncryptionSuite {
        private $cipherSuite;

        public function __construct($cipherSuite) {
            $this->cipherSuite = $cipherSuite;
        }

        public function splitMethodIVAndPayload($methodIVAndPayload) {
            $parts = explode("|", $methodIVAndPayload);
            
            // Handle both old format (method|IV|payload) and new GCM format (method|IV|tag|payload)
            if (count($parts) == 3) {
                // Old format: method|IV|payload (CBC mode)
                return array($parts[0], $parts[1], $parts[2], null);
            } elseif (count($parts) == 4) {
                // New format: method|IV|tag|payload (GCM mode)
                return array($parts[0], $parts[1], $parts[3], $parts[2]); // method, IV, payload, tag
            } else {
                throw new EncryptionException("Invalid encrypted data format",
                    LocalizableExceptionDefinition::$DECRYPT_ERROR ?? LocalizableExceptionDefinition::$GENERAL_FILE_SOURCE_ERROR);
            }
        }

        public function combineMethodIVAndPayload($method, $iv, $payload, $tag = null) {
            // Check if this is GCM mode
            $isGCM = strpos(strtolower($method), 'gcm') !== false;
            
            if ($isGCM && $tag !== null) {
                // New format: method|IV|tag|payload (GCM mode)
                return $method . "|" . $iv . "|" . $tag . "|" . $payload;
            } else {
                // Old format: method|IV|payload (CBC mode)
                return $method . "|" . $iv . "|" . $payload;
            }
        }

        public function buildLengthPayload($message) {
            return strlen($message) . "|" . $message;
        }

        public function extractMessageFromPayload($payload) {
            $lengthAndMessage = explode("|", $payload, 2);

            if (count($lengthAndMessage) != 2)
                throw new EncryptionException("Could not read configuration, the password is probably incorrect.",
                    LocalizableExceptionDefinition::$PROBABLE_INCORRECT_PASSWORD_ERROR);

            return substr($lengthAndMessage[1], 0, (int)$lengthAndMessage[0]);
        }

        public function encryptWithMethod($methodName, $message, $key) {
            $payload = $this->buildLengthPayload($message);
            $ivLength = openssl_cipher_iv_length($methodName);
            
            // Use secure random_bytes() instead of deprecated openssl_random_pseudo_bytes()
            try {
                $iv = random_bytes($ivLength);
            } catch (Exception $e) {
                throw new EncryptionException("Could not generate cryptographically secure IV: " . $e->getMessage(),
                    LocalizableExceptionDefinition::$IV_GENERATE_ERROR);
            }

            // Check if this is an authenticated encryption mode (GCM)
            $isGCM = strpos(strtolower($methodName), 'gcm') !== false;
            
            if ($isGCM) {
                // For GCM mode, we need to handle the authentication tag
                $tag = '';
                $encryptedMessage = openssl_encrypt($payload, $methodName, $key, 0, $iv, $tag);
                
                if ($encryptedMessage === false) {
                    throw new EncryptionException("GCM encryption failed for method: " . $methodName,
                        LocalizableExceptionDefinition::$ENCRYPT_ERROR ?? LocalizableExceptionDefinition::$GENERAL_FILE_SOURCE_ERROR);
                }
                
                // Return encrypted message, IV, and authentication tag
                return array($encryptedMessage, bin2hex($iv), bin2hex($tag));
            } else {
                // Traditional CBC mode
                $encryptedMessage = openssl_encrypt($payload, $methodName, $key, 0, $iv);
                
                if ($encryptedMessage === false) {
                    throw new EncryptionException("Encryption failed for method: " . $methodName,
                        LocalizableExceptionDefinition::$ENCRYPT_ERROR ?? LocalizableExceptionDefinition::$GENERAL_FILE_SOURCE_ERROR);
                }
                
                // Return encrypted message and IV (no auth tag for CBC)
                return array($encryptedMessage, bin2hex($iv));
            }
        }

        public function decryptWithMethod($methodName, $payload, $key, $iv, $tag = null) {
            $binaryIV = hex2bin($iv);
            
            // Check if this is an authenticated encryption mode (GCM)
            $isGCM = strpos(strtolower($methodName), 'gcm') !== false;
            
            if ($isGCM) {
                if ($tag === null) {
                    throw new EncryptionException("Authentication tag required for GCM mode but not provided",
                        LocalizableExceptionDefinition::$DECRYPT_ERROR ?? LocalizableExceptionDefinition::$GENERAL_FILE_SOURCE_ERROR);
                }
                
                $binaryTag = hex2bin($tag);
                $decryptedPayload = openssl_decrypt($payload, $methodName, $key, 0, $binaryIV, $binaryTag);
                
                if ($decryptedPayload === false) {
                    throw new EncryptionException("GCM decryption failed - authentication verification failed or invalid data",
                        LocalizableExceptionDefinition::$DECRYPT_ERROR ?? LocalizableExceptionDefinition::$GENERAL_FILE_SOURCE_ERROR);
                }
            } else {
                // Traditional CBC mode
                $decryptedPayload = openssl_decrypt($payload, $methodName, $key, 0, $binaryIV);
                
                if ($decryptedPayload === false) {
                    throw new EncryptionException("Decryption failed for method: " . $methodName,
                        LocalizableExceptionDefinition::$DECRYPT_ERROR ?? LocalizableExceptionDefinition::$GENERAL_FILE_SOURCE_ERROR);
                }
            }
            
            return $this->extractMessageFromPayload($decryptedPayload);
        }

        public function encryptWithBestCipherMethod($message, $key) {
            $cipherMethodName = $this->cipherSuite->getBestCipherMethod();
            $encryptionResult = $this->encryptWithMethod($cipherMethodName, $message, $key);

            // Check if this is GCM mode (has authentication tag)
            if (count($encryptionResult) == 3) {
                // GCM mode: [encrypted_message, iv, tag]
                return $this->combineMethodIVAndPayload($cipherMethodName, $encryptionResult[1],
                    $encryptionResult[0], $encryptionResult[2]);
            } else {
                // CBC mode: [encrypted_message, iv]
                return $this->combineMethodIVAndPayload($cipherMethodName, $encryptionResult[1],
                    $encryptionResult[0]);
            }
        }

        public function decryptWithInlineCipherMethod($payload, $key) {
            $methodIVPayloadTag = $this->splitMethodIVAndPayload($payload);
            // $methodIVPayloadTag = [method, IV, payload, tag (or null)]
            return $this->decryptWithMethod($methodIVPayloadTag[0], $methodIVPayloadTag[2], $key,
                $methodIVPayloadTag[1], $methodIVPayloadTag[3]);
        }
    }