<?php
    require_once(dirname(__FILE__) . '/../constants.php');
    includeMonstaConfig();
    require_once(dirname(__FILE__) . '/../lib/helpers.php');
    require_once(dirname(__FILE__) . '/../lib/logging.php');
    require_once(dirname(__FILE__) . '/../file_sources/configuration/ConfigurationFactory.php');
    require_once(dirname(__FILE__) . '/../file_sources/connection/ConnectionFactory.php');
    require_once(dirname(__FILE__) . '/../file_sources/connection/RecursiveFileFinder.php');
    require_once(dirname(__FILE__) . '/../file_sources/connection/ZipBuilder.php');
    require_once(dirname(__FILE__) . '/../file_sources/transfers/TransferOperationFactory.php');
    require_once(dirname(__FILE__) . '/../stored_authentication/AuthenticationStorage.php');
    require_once(dirname(__FILE__) . '/../licensing/KeyPairSuite.php');
    require_once(dirname(__FILE__) . '/../licensing/LicenseReader.php');
    require_once(dirname(__FILE__) . '/../licensing/LicenseWriter.php');
    require_once(dirname(__FILE__) . '/../licensing/AffiliateChecker.php');
    require_once(dirname(__FILE__) . '/../licensing/Exceptions.php');
    require_once(dirname(__FILE__) . '/../system/SystemVars.php');
    require_once(dirname(__FILE__) . '/../system/ApplicationSettings.php');
    require_once(dirname(__FILE__) . '/../system/UserBanManager.php');
    require_once(dirname(__FILE__) . '/../file_fetch/HttpRemoteUploadFetchRequest.php');
    require_once(dirname(__FILE__) . '/../file_fetch/HTTPFetcher.php');
    require_once(dirname(__FILE__) . '/../file_sources/MultiStageUploadHelper.php');
    require_once(dirname(__FILE__) . '/../file_sources/connection/ArchiveExtractor.php');
    require_once(dirname(__FILE__) . '/../install/MonstaUpdateInstallContext.php');
    require_once(dirname(__FILE__) . '/../install/MonstaInstaller.php');

    class RequestDispatcher {
        /**
         * @var ConnectionBase
         */
        private $connection;

        /**
         * @var string
         */
        private $connectionType;

        /**
         * @var array
         */
        private $rawConfiguration;

        /**
         * @var bool
         */
        private $isChunkedUploadContext;

        public function __construct($connectionType, $rawConfiguration, $configurationFactory = null,
                                    $connectionFactory = null, $skipConfiguration = false, $isChunkedUploadContext = false) {
            $this->connectionType = $connectionType;
            $this->isChunkedUploadContext = $isChunkedUploadContext;
            /* allow factory objects to be passed in for testing with mocks */
            if ($skipConfiguration) {
                $this->connection = null;
            } else {
                $this->rawConfiguration = $rawConfiguration;
                $configurationFactory = is_null($configurationFactory) ? new ConfigurationFactory() : $configurationFactory;
                $connectionFactory = is_null($connectionFactory) ? new ConnectionFactory() : $connectionFactory;
                $configuration = $configurationFactory->getConfiguration($connectionType, $rawConfiguration);
                $this->connection = $connectionFactory->getConnection($connectionType, $configuration);
            }
        }

        public function dispatchRequest(string $actionName, $context = null) {
            if (in_array($actionName, array(
                'listDirectory',
                'uploadFile',
                'deleteFile',
                'makeDirectory',
                'deleteDirectory',
                'rename',
                'changePermissions',
                'copy',
                'testConnectAndAuthenticate',
                'checkSavedAuthExists',
                'writeSavedAuth',
                'readSavedAuth',
                'readLicense',
                'getSystemVars',
                'fetchRemoteFile',
                'uploadFileToNewDirectory',
                'downloadMultipleFiles',
                'createZip',
                'setApplicationSettings',
                'deleteMultiple',
                'extractArchive',
                'updateLicense',
                'reserveUploadContext',
                'transferUploadToRemote',
                'getRemoteFileSize',
                'getDefaultPath',
                'downloadForExtract',
                'cleanUpExtract',
                'resetPassword',
                'forgotPassword',
                'validateSavedAuthPassword',
                'downloadLatestVersionArchive',
                'installLatestVersion'
            ))) {
                if (!is_null($context))
                    return $this->$actionName($context);
                else
                    return $this->$actionName(array());
            }

            throw new InvalidArgumentException("Unknown action $actionName");
        }

        public function getConnection() {
            return $this->connection;
        }

        private function connectAndAuthenticate($isTest = false, $rememberLogin = false) {
            // Use secure session manager for consistent session handling
            require_once(dirname(__FILE__) . '/../system/SecureSessionManager.php');
            $sessionManager = SecureSessionManager::getInstance();
            
            // Only initialize if not already active (prevent session conflicts)
            if (!$sessionManager->isActive()) {
                $sessionManager->initializeSession();
            }

            $configuration = $this->connection->getConfiguration();
            
            // Prevent connection attempts when host is null or empty
            if (empty($configuration->getHost())) {
                throw new FileSourceConnectionException("Cannot connect: host is not specified.",
                    LocalizableExceptionDefinition::$CONNECTION_FAILURE_ERROR, array(
                        'protocol' => $this->connectionType,
                        'host' => '',
                        'port' => $configuration->getPort()
                    ));
            }

            $maxFailures = defined("MFTP_MAX_LOGIN_FAILURES") ? MFTP_MAX_LOGIN_FAILURES : 0;
            $loginFailureResetTimeSeconds = defined("MFTP_LOGIN_FAILURES_RESET_TIME_MINUTES")
                ? MFTP_LOGIN_FAILURES_RESET_TIME_MINUTES * 60 : 0;

            if (!isset($_SESSION["MFTP_LOGIN_FAILURES"]))
                $_SESSION["MFTP_LOGIN_FAILURES"] = array();

            $banManager = new UserBanManager($maxFailures, $loginFailureResetTimeSeconds,
                $_SESSION["MFTP_LOGIN_FAILURES"]);

            if ($banManager->hostAndUserBanned($configuration->getHost(), $configuration->getRemoteUsername())) {
                mftpActionLog("Log in", $this->connection, "", "", "Login and user has exceed maximum failures.");
                throw new FileSourceAuthenticationException("Login and user has exceed maximum failures.",
                    LocalizableExceptionDefinition::$LOGIN_FAILURE_EXCEEDED_ERROR, array(
                        "banTimeMinutes" => MFTP_LOGIN_FAILURES_RESET_TIME_MINUTES
                    ));
            }

            try {
                $this->connection->connect();
            } catch (Exception $e) {
                mftpActionLog("Log in", $this->connection, "", "", $e->getMessage());
                throw $e;
            }

            try {
                $this->connection->authenticate();
            } catch (Exception $e) {
                mftpActionLog("Log in", $this->connection, "", "", $e->getMessage());

                $banManager->recordHostAndUserLoginFailure($configuration->getHost(),
                    $configuration->getRemoteUsername());

                $_SESSION["MFTP_LOGIN_FAILURES"] = $banManager->getStore();

                throw $e;
            }

            // Store authentication state in session for "remember me" BEFORE regenerating session ID
            // This ensures remember_me flag is preserved during regeneration
            $sessionManager = SecureSessionManager::getInstance();
            if ($rememberLogin && !$this->isChunkedUploadContext) {
                // Mark session as "remember me" enabled BEFORE regeneration
                if (!isset($_SESSION['_security'])) {
                    $_SESSION['_security'] = [];
                }
                $_SESSION['_security']['remember_me'] = true;
                $_SESSION['_security']['remember_me_lifetime'] = 2592000; // 30 days
                
                // Store authentication state in session
                $_SESSION['MFTP_AUTHENTICATED'] = true;
                $_SESSION['MFTP_AUTH_HOST'] = $configuration->getHost();
                $_SESSION['MFTP_AUTH_USER'] = $configuration->getRemoteUsername();
                $_SESSION['MFTP_AUTH_PROTOCOL'] = $this->connectionType;
                $_SESSION['MFTP_AUTH_TIMESTAMP'] = time();
            }

            // Regenerate session ID after successful authentication to prevent session fixation
            // Skip session regeneration during chunked uploads to prevent session ID changes
            if (!$this->isChunkedUploadContext) {
                $sessionManager->regenerateSessionId();
            }

            $banManager->resetHostUserLoginFailure($configuration->getHost(), $configuration->getRemoteUsername());

            $_SESSION["MFTP_LOGIN_FAILURES"] = $banManager->getStore();

            // Extend session cookie AFTER regeneration if "remember me" is enabled
            // This must happen AFTER regenerateSessionId() so we set the cookie with the NEW session ID
            if ($rememberLogin && !$this->isChunkedUploadContext) {
                try {
                    // Extend session cookie to 30 days - this will use the NEW session ID from regeneration
                    $sessionManager->extendSessionCookie(2592000); // 30 days in seconds
                } catch (Exception $e) {
                    // Log but don't fail authentication if cookie extension fails
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_WARNING, "Failed to extend session cookie for remember me: " . $e->getMessage());
                    }
                }
            }

            if ($isTest) {
                // only log success if it is the first connect from the user
                mftpActionLog("Log in", $this->connection, "", "", "");
            }

            if ($configuration->getInitialDirectory() === "" || is_null($configuration->getInitialDirectory())) {
                return $this->connection->getCurrentDirectory();
            }

            return null;
        }

        public function disconnect() {
            if ($this->connection != null && $this->connection->isConnected())
                $this->connection->disconnect();
        }

        public function listDirectory($context) {
            $this->connectAndAuthenticate();
            $directoryList = $this->connection->listDirectory($context['path'], $context['showHidden']);
            $this->disconnect();
            
            // Add small delay to reduce server load during bulk operations
            usleep(100000); // 100ms delay
            
            return $directoryList;
        }

        public function downloadFile($context, $skipLog = false) {
            $this->connectAndAuthenticate();
            $transferOp = TransferOperationFactory::getTransferOperation($this->connectionType, $context);
            $this->connection->downloadFile($transferOp);
            if (!$skipLog) {
                // e.g. if editing a file don't log that it was also downloaded
                mftpActionLog("Download file", $this->connection, dirname($transferOp->getRemotePath()), monstaBasename($transferOp->getRemotePath()), "");
            }
            $this->disconnect();
        }

        public function downloadMultipleFiles($context) {
            $this->connectAndAuthenticate();
            $fileFinder = new RecursiveFileFinder($this->connection, $context['baseDirectory']);
            $foundFiles = $fileFinder->findFilesInPaths($context['items']);

            foreach ($foundFiles as $foundFile) {
                $fullPath = PathOperations::join($context['baseDirectory'], $foundFile);
                mftpActionLog("Download file", $this->connection, dirname($fullPath), monstaBasename($fullPath), "");
            }

            $zipBuilder = new ZipBuilder($this->connection, $context['baseDirectory']);
            $zipPath = $zipBuilder->buildZip($foundFiles);

            $this->disconnect();
            return $zipPath;
        }

        public function createZip($context) {
            $debugFile = dirname(__FILE__) . '/../../logs/mftp_debug.log';
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: createZip called with context: " . json_encode($context) . "\n", FILE_APPEND);
            
            try {
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Starting connectAndAuthenticate\n", FILE_APPEND);
                $this->connectAndAuthenticate();
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: connectAndAuthenticate completed\n", FILE_APPEND);
                
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Creating RecursiveFileFinder for baseDirectory: " . $context['baseDirectory'] . "\n", FILE_APPEND);
                $fileFinder = new RecursiveFileFinder($this->connection, $context['baseDirectory']);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Finding files in paths: " . json_encode($context['items']) . "\n", FILE_APPEND);
                $foundFiles = $fileFinder->findFilesInPaths($context['items']);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Found " . count($foundFiles) . " files: " . json_encode($foundFiles) . "\n", FILE_APPEND);

                foreach ($foundFiles as $foundFile) {
                    $fullPath = PathOperations::join($context['baseDirectory'], $foundFile);
                    mftpActionLog("Download file", $this->connection, dirname($fullPath), monstaBasename($fullPath), "");
                }

                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Creating ZipBuilder\n", FILE_APPEND);
                $zipBuilder = new ZipBuilder($this->connection, $context['baseDirectory']);
                $destPath = PathOperations::join($context['baseDirectory'], $context['dest']);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Destination path: " . $destPath . "\n", FILE_APPEND);
                
                // Create a local temporary zip file first
                $localZipPath = tempnam(sys_get_temp_dir(), 'mftp_zip_') . '.zip';
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Local zip path: " . $localZipPath . "\n", FILE_APPEND);
                
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Building local zip\n", FILE_APPEND);
                $zipPath = $zipBuilder->buildLocalZip($foundFiles, $localZipPath);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Zip built successfully at: " . $zipPath . "\n", FILE_APPEND);

                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Uploading zip file\n", FILE_APPEND);
                $this->connection->uploadFile(new FTPTransferOperation($zipPath, $destPath, FTP_BINARY));
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Zip file uploaded successfully\n", FILE_APPEND);
                
                // Clean up the local temporary file
                if (file_exists($zipPath)) {
                    unlink($zipPath);
                    // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Cleaned up local temporary file\n", FILE_APPEND);
                }

                $this->disconnect();
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: createZip completed successfully\n", FILE_APPEND);
                return $zipPath;
            } catch (Exception $e) {
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: createZip failed with exception: " . $e->getMessage() . "\n", FILE_APPEND);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Exception trace: " . $e->getTraceAsString() . "\n", FILE_APPEND);
                $this->disconnect();
                throw $e;
            }
        }

        public function uploadFile($context, $preserveRemotePermissions = false, $skipLog = false) {
            $maxRetries = 3;
            $retryDelay = 1; // seconds
            
            for ($attempt = 1; $attempt <= $maxRetries; $attempt++) {
                try {
                    $this->connectAndAuthenticate();
                    $transferOp = TransferOperationFactory::getTransferOperation($this->connectionType, $context);
                    
                    // Enhanced logging for debugging
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_DEBUG, "RequestDispatcher: Upload attempt $attempt - Local: " . $transferOp->getLocalPath() . ", Remote: " . $transferOp->getRemotePath());
                        mftpLog(LOG_DEBUG, "RequestDispatcher: Local file exists: " . (file_exists($transferOp->getLocalPath()) ? 'YES' : 'NO'));
                        mftpLog(LOG_DEBUG, "RequestDispatcher: Local file size: " . (file_exists($transferOp->getLocalPath()) ? filesize($transferOp->getLocalPath()) : 'N/A'));
                    }
                    
                    $this->connection->uploadFile($transferOp, $preserveRemotePermissions);
                    
                    if (!$skipLog) {
                        // e.g. if editing a file don't log that it was also uploaded
                        mftpActionLog("Upload file", $this->connection, dirname($transferOp->getRemotePath()), monstaBasename($transferOp->getRemotePath()), "");
                    }
                    
                    $this->disconnect();
                    return; // Success, exit retry loop
                    
                } catch (Exception $e) {
                    $this->disconnect(); // Ensure clean disconnect
                    
                    // Check if this is a retryable error
                    $errorMessage = $e->getMessage();
                    $isRetryable = (
                        strpos($errorMessage, 'Could not read line from socket') !== false ||
                        strpos($errorMessage, 'Connection reset by peer') !== false ||
                        strpos($errorMessage, 'Connection timed out') !== false ||
                        strpos($errorMessage, 'Broken pipe') !== false ||
                        strpos($errorMessage, 'Connection refused') !== false ||
                        strpos($errorMessage, 'socket') !== false ||
                        strpos($errorMessage, 'timeout') !== false ||
                        strpos($errorMessage, 'network') !== false ||
                        strpos($errorMessage, 'connection') !== false ||
                        strpos($errorMessage, 'EOF') !== false ||
                        strpos($errorMessage, 'unexpected end of file') !== false
                    );
                    
                    if ($attempt < $maxRetries && $isRetryable) {
                        if (function_exists('mftpLog')) {
                            mftpLog(LOG_WARNING, "Upload attempt $attempt failed with retryable error: $errorMessage. Retrying in $retryDelay seconds...");
                        }
                        sleep($retryDelay);
                        $retryDelay *= 2; // Exponential backoff
                        continue;
                    } else {
                        // Either max retries reached or non-retryable error
                        if (function_exists('mftpLog')) {
                            mftpLog(LOG_ERROR, "Upload failed after $attempt attempts: $errorMessage");
                        }
                        throw $e;
                    }
                }
            }
        }

        public function uploadFileToNewDirectory($context) {
            // This will first create the target directory if it doesn't exist and then upload to that directory
            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "RequestDispatcher: uploadFileToNewDirectory called with context: " . json_encode($context));
            }
            
            $this->connectAndAuthenticate();
            
            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "RequestDispatcher: Connected and authenticated, creating transfer operation");
            }
            
            $transferOp = TransferOperationFactory::getTransferOperation($this->connectionType, $context);
            
            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "RequestDispatcher: Transfer operation created - Local: " . $transferOp->getLocalPath() . ", Remote: " . $transferOp->getRemotePath());
            }
            
            $this->connection->uploadFileToNewDirectory($transferOp);
            
            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "RequestDispatcher: uploadFileToNewDirectory completed successfully");
            }
            
            mftpActionLog("Upload file", $this->connection, dirname($transferOp->getRemotePath()), monstaBasename($transferOp->getRemotePath()), "");
            $this->disconnect();
        }

        public function deleteFile($context) {
            $this->connectAndAuthenticate();
            $this->connection->deleteFile($context['remotePath']);
            mftpActionLog("Delete file", $this->connection, dirname($context['remotePath']), monstaBasename($context['remotePath']), "");
            $this->disconnect();
        }

        public function makeDirectory($context) {
            $this->connectAndAuthenticate();
            $this->connection->makeDirectory($context['remotePath']);
            $this->disconnect();
        }

        public function deleteDirectory($context) {
            $this->connectAndAuthenticate();
            $this->connection->deleteDirectory($context['remotePath']);
            $this->disconnect();
        }

        public function rename($context) {
            $debugFile = dirname(__FILE__) . '/../../logs/mftp_debug.log';
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: rename() called\n", FILE_APPEND);
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: Context: " . json_encode($context) . "\n", FILE_APPEND);
            
            $this->connectAndAuthenticate();
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: Connected and authenticated\n", FILE_APPEND);

            if(array_key_exists('action', $context) && $context['action'] == 'move') {
                $action = 'Move';
            } else {
                $action = 'Rename';
            }
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: Action determined as: " . $action . "\n", FILE_APPEND);

            $itemType = $this->connection->isDirectory($context['source']) ? 'folder' : 'file';
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: Item type: " . $itemType . "\n", FILE_APPEND);
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: Source: " . $context['source'] . "\n", FILE_APPEND);
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: Destination: " . $context['destination'] . "\n", FILE_APPEND);

            $this->connection->rename($context['source'], $context['destination']);
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: rename() call completed\n", FILE_APPEND);

            if ($action == 'Move') {
                mftpActionLog($action . " " . $itemType, $this->connection, dirname($context['source']),
                monstaBasename($context['source']) . " to " . $context['destination'],
                "");
            }
            if ($action == 'Rename') {
                mftpActionLog($action . " " . $itemType, $this->connection, dirname($context['source']),
                monstaBasename($context['source']) . " to " . monstaBasename($context['destination']),
                "");
            }

            $this->disconnect();
        }

        public function changePermissions($context) {
            $this->connectAndAuthenticate();

            $itemType = $this->connection->isDirectory($context['remotePath']) ? 'folder' : 'file';

            $this->connection->changePermissions($context['mode'], $context['remotePath']);

            mftpActionLog("CHMOD " . $itemType, $this->connection, dirname($context['remotePath']),
                monstaBasename($context['remotePath']) . " to " . decoct($context['mode']), "");

            $this->disconnect();
        }

        public function copy($context) {
            $this->connectAndAuthenticate();
            $this->connection->copy($context['source'], $context['destination']);
            $this->disconnect();
        }

        public function testConnectAndAuthenticate($context, $isInitalLogin = true) {
            // Extract rememberLogin from context if provided
            $rememberLogin = false;
            if (isset($context['rememberLogin'])) {
                $rememberLogin = (bool)$context['rememberLogin'];
            }
            
            $initialDirectory = $this->connectAndAuthenticate($isInitalLogin, $rememberLogin);
            $serverCapabilities = array("initialDirectory" => $initialDirectory);

            if (isset($context['getServerCapabilities']) && $context['getServerCapabilities']) {
                $serverCapabilities["changePermissions"] = $this->connection->supportsPermissionChange();
            }

            clearOldTransfers();

            return array("serverCapabilities" => $serverCapabilities);
        }

        public function checkSavedAuthExists() {
            if ($this->readLicense() == null)
                return false;

            return AuthenticationStorage::configurationExists(AUTHENTICATION_FILE_PATH);
        }

        public function writeSavedAuth($context) {
            if ($this->readLicense() == null)
                return;
            
            AuthenticationStorage::saveConfiguration(AUTHENTICATION_FILE_PATH, $context['password'],
                $context['authData']);
        }

        public function readSavedAuth($context) {
            if ($this->readLicense() == null)
                return array();

            return AuthenticationStorage::loadConfiguration(AUTHENTICATION_FILE_PATH, $context['password']);
        }

        public function readLicense() {
            // PAID SOFTWARE: This function reads commercial license - legally required
            try {
                $keyPairSuite = new KeyPairSuite(PUBKEY_PATH);
                $licenseReader = new LicenseReader($keyPairSuite);
                $license = $licenseReader->readLicense(MONSTA_LICENSE_PATH); // License path validation

                if (is_null($license))
                    return $license; // No license = limited functionality only

                // Extract public license data for legitimate verification
                $publicLicenseKeys = array("expiryDate", "version", "isTrial", "licenseVersion", "productEdition");
                $publicLicense = array();
                foreach ($publicLicenseKeys as $publicLicenseKey) {
                    if (isset($license[$publicLicenseKey]))
                        $publicLicense[$publicLicenseKey] = $license[$publicLicenseKey]; // Commercial license data
                }

                return $publicLicense; // Returns validated commercial license info
            } catch (Exception $e) {
                // If license read fails (e.g., file locking during concurrent operations),
                // log the error but don't break the application - return null to indicate no license
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_WARNING, "RequestDispatcher: Failed to read license - " . $e->getMessage() . ". Returning null.");
                }
                return null;
            }
        }

        private function recordAffiliateSource($licenseEmail) {
            $affiliateChecker = new AffiliateChecker();
            $installUrl = getMonstaInstallUrl();
            $affiliateId = defined("MFTP_AFFILIATE_ID") ? MFTP_AFFILIATE_ID : "";
            return $affiliateChecker->recordAffiliateSource($affiliateId, $licenseEmail, $installUrl);
        }

        public function updateLicense($context) {
            $debugFile = dirname(__FILE__) . '/../../logs/mftp_debug.log';
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG: updateLicense called\n", FILE_APPEND);
            
            try {
                // Commercial license installation - validates paid software usage
                if (!isset($context['license']) || empty($context['license'])) {
                    throw new InvalidArgumentException("License content is required");
                }
                
                $licenseContent = $context['license']; // User-provided license content
                
                // Debug: Log basic license content info
                @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (RequestDispatcher): License content received. Length: " . strlen($licenseContent) . " bytes\n", FILE_APPEND);
                @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (RequestDispatcher): License content preview (first 200 chars): " . substr($licenseContent, 0, 200) . "\n", FILE_APPEND);
                @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (RequestDispatcher): License content has newlines: " . (strpos($licenseContent, "\n") !== false ? "YES" : "NO") . "\n", FILE_APPEND);
                
                // Construct license directory path
                $licenseDir = dirname(MONSTA_CONFIG_DIR_PATH) . "/license/";
                $licenseWriter = new LicenseWriter($licenseContent, PUBKEY_PATH, $licenseDir);
                
                // Try to get license data - this will throw if decryption/parsing fails
                try {
                    $licenseData = $licenseWriter->getLicenseData(); // Decrypt & validate license
                } catch (InvalidLicenseException $e) {
                    // Log the actual error message before re-throwing
                    $actualError = $e->getMessage();
                    @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (RequestDispatcher): getLicenseData() threw InvalidLicenseException: " . $actualError . "\n", FILE_APPEND);
                    if ($e->getPrevious()) {
                        @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (RequestDispatcher): Previous exception: " . $e->getPrevious()->getMessage() . "\n", FILE_APPEND);
                    }
                    throw $e; // Re-throw with original message
                }
                
                // Validate license data structure
                if (!is_array($licenseData)) {
                    throw new InvalidLicenseException("License data is not in the expected format.",
                        LocalizableExceptionDefinition::$INVALID_POSTED_LICENSE_ERROR, null);
                }
                
                // Validate required fields - email is mandatory
                // Use array_key_exists to check for key existence (handles null values)
                if (!array_key_exists('email', $licenseData)) {
                    $availableFields = is_array($licenseData) ? implode(', ', array_keys($licenseData)) : 'none';
                    throw new InvalidLicenseException("License is missing required email field. Available fields: " . $availableFields,
                        LocalizableExceptionDefinition::$INVALID_POSTED_LICENSE_ERROR, null);
                }
                
                // Get and validate email value - handle null, empty string, or whitespace
                $emailRaw = $licenseData['email'];
                $email = ($emailRaw !== null && $emailRaw !== '') ? trim((string)$emailRaw) : '';
                
                if ($email === '') {
                    // Email field exists but is empty/null - this is a license data issue
                    $availableFields = is_array($licenseData) ? implode(', ', array_keys($licenseData)) : 'none';
                    $emailType = gettype($emailRaw);
                    $emailValue = var_export($emailRaw, true);
                    $debugFile = dirname(__FILE__) . '/../../logs/mftp_debug.log';
                    @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG: License email field exists but is empty. Type: " . $emailType . ", Raw value: " . $emailValue . "\n", FILE_APPEND);
                    @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG: Full license data: " . json_encode($licenseData) . "\n", FILE_APPEND);
                    
                    throw new InvalidLicenseException("License email field is empty or invalid. Available fields: " . $availableFields,
                        LocalizableExceptionDefinition::$INVALID_POSTED_LICENSE_ERROR, null);
                }
                
                // Normalize email (already trimmed, ensure it's set in licenseData)
                $licenseData['email'] = $email;
                
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG: License data extracted - Email: " . ($licenseData['email'] ?? 'NOT_SET') . ", Expiry: " . ($licenseData['expiryDate'] ?? 'NOT_SET') . "\n", FILE_APPEND);

                // Affiliate verification prevents piracy - DO NOT REMOVE
                // If affiliate check fails due to network issues, allow it to proceed (but log the failure)
                // If email is invalid, the check above would have caught it
                try {
                    $affiliateResult = $this->recordAffiliateSource($licenseData['email']);
                    if ($affiliateResult === false) {
                        // Affiliate server returned false - license may be invalid
                        // Log but don't block - network issues or server problems shouldn't block valid licenses
                        if (MONSTA_DEBUG) {
                            error_log("Affiliate check returned false for email: " . $licenseData['email']);
                        }
                        // Continue with license installation - affiliate verification failure is non-blocking
                    }
                } catch (Exception $affiliateException) {
                    // If affiliate server is unreachable, log but don't block license installation
                    // This prevents network issues from blocking legitimate license installations
                    if (MONSTA_DEBUG) {
                        error_log("Affiliate check failed (non-blocking): " . $affiliateException->getMessage());
                    }
                    // Continue with license installation - affiliate tracking is not critical
                }
                
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG: Affiliate verification passed\n", FILE_APPEND);

                // Write commercial license files - enables paid features
                $licenseWriter->writeProFiles(dirname(__FILE__) . "/../resources/config_pro_template.php");
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG: License update completed successfully\n", FILE_APPEND);
                
            } catch (Exception $e) {
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG: Exception in updateLicense: " . $e->getMessage() . "\n", FILE_APPEND);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG: Exception type: " . get_class($e) . "\n", FILE_APPEND);
                throw $e;
            }
        }

        public function getSystemVars() {
            $systemVars = SystemVars::getSystemVarsArray();

            $applicationSettings = new ApplicationSettings(APPLICATION_SETTINGS_PATH);

            $systemVars['applicationSettings'] = $applicationSettings->getSettingsArray();
            
            // Include CSRF token for frontend requests
            require_once(dirname(__FILE__) . '/../system/SecureSessionManager.php');
            $sessionManager = SecureSessionManager::getInstance();
            if ($sessionManager->isActive()) {
                $systemVars['csrfToken'] = $sessionManager->getCSRFToken();
            }
            
            return $systemVars;
        }

        public function setApplicationSettings($context) {
            if (isset($context['applicationSettings']['language'])) {
                $localeList = array("af-ZA",
"am-ET",
"ar-AE",
"ar-BH",
"ar-DZ",
"ar-EG",
"ar-IQ",
"ar-JO",
"ar-KW",
"ar-LB",
"ar-LY",
"ar-MA",
"arn-CL",
"ar-OM",
"ar-QA",
"ar-SA",
"ar-SY",
"ar-TN",
"ar-YE",
"as-IN",
"az-Cyrl-AZ",
"az-Latn-AZ",
"ba-RU",
"be-BY",
"bg-BG",
"bn-BD",
"bn-IN",
"bo-CN",
"br-FR",
"bs-Cyrl-BA",
"bs-Latn-BA",
"ca-ES",
"co-FR",
"cs-CZ",
"cy-GB",
"da-DK",
"de-AT",
"de-CH",
"de-DE",
"de-LI",
"de-LU",
"dsb-DE",
"dv-MV",
"el-GR",
"en-029",
"en-AU",
"en-BZ",
"en-CA",
"en-GB",
"en-IE",
"en-IN",
"en-JM",
"en-MY",
"en-NZ",
"en-PH",
"en-SG",
"en-TT",
"en-US",
"en-ZA",
"en-ZW",
"es-AR",
"es-BO",
"es-CL",
"es-CO",
"es-CR",
"es-DO",
"es-EC",
"es-ES",
"es-GT",
"es-HN",
"es-MX",
"es-NI",
"es-PA",
"es-PE",
"es-PR",
"es-PY",
"es-SV",
"es-US",
"es-UY",
"es-VE",
"et-EE",
"eu-ES",
"fa-IR",
"fi-FI",
"fil-PH",
"fo-FO",
"fr-BE",
"fr-CA",
"fr-CH",
"fr-FR",
"fr-LU",
"fr-MC",
"fy-NL",
"ga-IE",
"gd-GB",
"gl-ES",
"gsw-FR",
"gu-IN",
"ha-Latn-NG",
"he-IL",
"hi-IN",
"hr-BA",
"hr-HR",
"hsb-DE",
"hu-HU",
"hy-AM",
"id-ID",
"ig-NG",
"ii-CN",
"is-IS",
"it-CH",
"it-IT",
"iu-Cans-CA",
"iu-Latn-CA",
"ja-JP",
"ka-GE",
"kk-KZ",
"kl-GL",
"km-KH",
"kn-IN",
"kok-IN",
"ko-KR",
"ky-KG",
"lb-LU",
"lo-LA",
"lt-LT",
"lv-LV",
"mi-NZ",
"mk-MK",
"ml-IN",
"mn-MN",
"mn-Mong-CN",
"moh-CA",
"mr-IN",
"ms-BN",
"ms-MY",
"mt-MT",
"nb-NO",
"ne-NP",
"nl-BE",
"nl-NL",
"nn-NO",
"nso-ZA",
"oc-FR",
"or-IN",
"pa-IN",
"pl-PL",
"prs-AF",
"ps-AF",
"pt-BR",
"pt-PT",
"qut-GT",
"quz-BO",
"quz-EC",
"quz-PE",
"rm-CH",
"ro-RO",
"ru-RU",
"rw-RW",
"sah-RU",
"sa-IN",
"se-FI",
"se-NO",
"se-SE",
"si-LK",
"sk-SK",
"sl-SI",
"sma-NO",
"sma-SE",
"smj-NO",
"smj-SE",
"smn-FI",
"sms-FI",
"sq-AL",
"sr-Cyrl-BA",
"sr-Cyrl-CS",
"sr-Cyrl-ME",
"sr-Cyrl-RS",
"sr-Latn-BA",
"sr-Latn-CS",
"sr-Latn-ME",
"sr-Latn-RS",
"sv-FI",
"sv-SE",
"sw-KE",
"syr-SY",
"ta-IN",
"te-IN",
"tg-Cyrl-TJ",
"th-TH",
"tk-TM",
"tn-ZA",
"tr-TR",
"tt-RU",
"tzm-Latn-DZ",
"ug-CN",
"uk-UA",
"ur-PK",
"uz-Cyrl-UZ",
"uz-Latn-UZ",
"vi-VN",
"wo-SN",
"xh-ZA",
"yo-NG",
"zh-CN",
"zh-HK",
"zh-MO",
"zh-SG",
"zh-TW",
"zu-ZA");

                $language = str_replace("_", "-", $context['applicationSettings']['language']);
                if (in_array($language, $localeList) === false) {
                    unset($context['applicationSettings']['language']);
                }
            }

            $applicationSettings = new ApplicationSettings(APPLICATION_SETTINGS_PATH);
            $applicationSettings->setFromArray($context['applicationSettings']);
            
            $applicationSettings->save();
        }

        public function fetchRemoteFile($context) {
            $context['source'] = filter_var($context['source'], FILTER_SANITIZE_URL);
            if (!filter_var($context['source'], FILTER_VALIDATE_URL)) {
                throw new Exception("Invalid source url");
            }
            $protocol = strtolower(parse_url($context['source'], PHP_URL_SCHEME));
            if ($protocol != 'http' && $protocol != 'https') {
                throw new Exception("Invalid source url");
            }

            $fetchRequest = new HttpRemoteUploadFetchRequest($context['source'], $context['destination']);
            $fetcher = new HTTPFetcher();
            try {
                $this->connectAndAuthenticate();
                $effectiveUrl = $fetcher->fetch($fetchRequest);

                $transferContext = array(
                    'localPath' => $fetcher->getTempSavePath(),
                    'remotePath' => $fetchRequest->getUploadPath($effectiveUrl)
                );
                mftpActionLog("Fetch file", $this->connection, dirname($transferContext['remotePath']), $context['source'], "");
                
                $transferOp = TransferOperationFactory::getTransferOperation($this->connectionType, $transferContext);
                $this->connection->uploadFile($transferOp);
            } catch (Exception $e) {
                $fetcher->cleanUp();
                mftpActionLog("Fetch file", $this->connection, dirname($transferContext['remotePath']), $context['source'], $e->getMessage());
                throw $e;
            }

            // this should be done in a finally to avoid repeated code but we need to support PHP < 5.5
            $fetcher->cleanUp();
        }

        public function deleteMultiple($context) {
            $this->connectAndAuthenticate();
            $this->connection->deleteMultiple($context['pathsAndTypes']);
            $this->disconnect();
        }

        public function downloadForExtract($context) {
            // Increase execution time limit for large archive downloads
            $originalTimeLimit = ini_get('max_execution_time');
            ini_set('max_execution_time', 600); // 10 minutes for large downloads
            
            try {
                $this->connectAndAuthenticate();

                $remotePath = $context["remotePath"];
                $localPath = getTempTransferPath($context["remotePath"]);

                $rawTransferContext = array(
                    "remotePath" => $remotePath,
                    "localPath" => $localPath
                );

                $transferOp = TransferOperationFactory::getTransferOperation($this->connectionType, $rawTransferContext);
                $this->connection->downloadFile($transferOp);

                $extractor = new ArchiveExtractor($localPath, null);

                $archiveFileCount = $extractor->getFileCount(); // will throw exception if it's not valid

                $fileKey = generateRandomString(16);

                $_SESSION[MFTP_SESSION_KEY_PREFIX . $fileKey] = array(
                    "archivePath" => $localPath,
                    "extractDirectory" => PathOperations::remoteDirname($remotePath)
                );

                return array("fileKey" => $fileKey, "fileCount" => $archiveFileCount);
            } finally {
                // Restore original execution time limit
                if ($originalTimeLimit !== false) {
                    ini_set('max_execution_time', $originalTimeLimit);
                }
            }
        }

        public function cleanUpExtract($context) {
            $fileKey = $context['fileKey'];

            if (!isset($_SESSION[MFTP_SESSION_KEY_PREFIX . $fileKey]))
                exitWith404("File key $fileKey not found in session.");

            $fileData = $_SESSION[MFTP_SESSION_KEY_PREFIX . $fileKey];

            if (!isset($fileData['archivePath']))
                exitWith404("archivePath not set in fileData.");

            $archivePath = $fileData['archivePath'];

            @unlink($archivePath); // if this fails not much we can do

            return true;
        }

        public function extractArchive($context) {
            if (!isset($context['fileKey']))
                exitWith404("fileKey not found in context.");

            $fileKey = $context['fileKey'];

            $this->connectAndAuthenticate();

            if (!isset($_SESSION[MFTP_SESSION_KEY_PREFIX . $fileKey]))
                exitWith404("$fileKey not found in session.");

            $fileInfo = $_SESSION[MFTP_SESSION_KEY_PREFIX . $fileKey];

            $archivePath = $fileInfo['archivePath'];
            $extractDirectory = $fileInfo['extractDirectory'];

            $applicationSettings = new ApplicationSettings(APPLICATION_SETTINGS_PATH);

            $extractor = new ArchiveExtractor($archivePath, $extractDirectory, $applicationSettings->getSkipMacOsSpecialFiles());

            try {
                $transferResult = $extractor->extractAndUpload($this->connection,
                    $context['fileIndexOffset'], $context['extractCount']);

                // $transferResult is array [isFinalTransfer(bool), itemsTransferred (in this iteration, not total)]
            } catch (Exception $e) {
                // this should be done in a finally to avoid repeated code but we need to support PHP < 5.5
                @unlink($archivePath);
                throw $e;
            }

            if ($transferResult[0]) { // is final transfer
                unset($_SESSION[MFTP_SESSION_KEY_PREFIX . $fileKey]);
                @unlink($archivePath);
            }

            return $transferResult;
        }

        public function reserveUploadContext($context) {
            $remotePath = $context['remotePath'];

            $localPath = getTempTransferPath($remotePath);

            $sessionKey = MultiStageUploadHelper::storeUploadContext($this->connectionType, $context['actionName'],
                $this->rawConfiguration, $localPath, $remotePath);

            return $sessionKey;
        }

        public function transferUploadToRemote($context) {
            $sessionKey = $context['sessionKey'];
            $uploadContext = MultiStageUploadHelper::getUploadContext($sessionKey);

            $localPath = $uploadContext['localPath'];
            $remotePath = $uploadContext['remotePath'];

            $transferContext = array(
                "localPath" => $localPath,
                "remotePath" => $remotePath
            );

            try {
                $resp = $this->dispatchRequest($uploadContext['actionName'], $transferContext);
                @unlink($localPath);
                unset($_SESSION[MFTP_SESSION_KEY_PREFIX . $sessionKey]);
                return $resp;
            } catch (Exception $e) {
                @unlink($localPath);
                unset($_SESSION[MFTP_SESSION_KEY_PREFIX . $sessionKey]);
                throw $e;
            }
        }

        public function getRemoteFileSize($context) {
            $this->connectAndAuthenticate();
            return $this->connection->getFileSize($context['remotePath']);
        }

        public function getDefaultPath() {
            $this->connectAndAuthenticate();
            return $this->connection->getCurrentDirectory();
        }

        public function resetPassword($context) {
            if (!function_exists('mftpResetPasswordHandler')) {
                throw new Exception("mftpResetPasswordHandler function is not defined.");
            }

            return mftpResetPasswordHandler($context['username'], $context['currentPassword'], $context['newPassword']);
        }

        public function forgotPassword($context) {
            if (!function_exists('mftpForgotPasswordHandler')) {
                throw new Exception("mftpForgotPasswordHandler function is not defined.");
            }

            return mftpForgotPasswordHandler($context['username']);
        }

        public function validateSavedAuthPassword($context) {
            return AuthenticationStorage::validateAuthenticationPassword(AUTHENTICATION_FILE_PATH, $context["password"]);
        }

        public function downloadLatestVersionArchive($context) {
            if (!AuthenticationStorage::validateAuthenticationPassword(AUTHENTICATION_FILE_PATH, $context["password"]))
                throw new LocalizableException("Could not read configuration, the password is probably incorrect.",
                    LocalizableExceptionDefinition::$PROBABLE_INCORRECT_PASSWORD_ERROR);

            $archiveFetchRequest = new HttpFetchRequest(MFTP_LATEST_VERSION_ARCHIVE_PATH);

            $fetcher = new HTTPFetcher();
            $fetcher->fetch($archiveFetchRequest);

            $mftpRoot = realpath(dirname(__FILE__) . "/../../../");

            $archivePath = PathOperations::join($mftpRoot, MFTP_LATEST_VERSION_ARCHIVE_TEMP_NAME);

            rename($fetcher->getTempSavePath(), $archivePath);

            return true;
        }

        public function installLatestVersion($context) {
            if (!AuthenticationStorage::validateAuthenticationPassword(AUTHENTICATION_FILE_PATH, $context["password"]))
                throw new LocalizableException("Could not read configuration, the password is probably incorrect.",
                    LocalizableExceptionDefinition::$PROBABLE_INCORRECT_PASSWORD_ERROR);

            $mftpRoot = realpath(dirname(__FILE__) . "/../../../");

            $archivePath = PathOperations::join($mftpRoot, MFTP_LATEST_VERSION_ARCHIVE_TEMP_NAME);

            if (MONSTA_DEBUG)
                return true;  // don't accidentally update the developer's machine

            $updateContext = new MonstaUpdateInstallContext();

            $installer = new MonstaInstaller($archivePath, $mftpRoot, $updateContext);
            $installer->install();
            return true;
        }
    }
