<?php
    require_once(dirname(__FILE__) . "/../file_sources/PathOperations.php");
    require_once(dirname(__FILE__) . "/LicenseReader.php");
    require_once(dirname(__FILE__) . "/KeyPairSuite.php");
    require_once(dirname(__FILE__) . "/ProPackageIDGenerator.php");
    require_once(dirname(__FILE__) . "/ProConfigBuilder.php");
    require_once(dirname(__FILE__) . "/../lib/LocalizableException.php");

    class LicenseWriter {
        /**
         * @var string
         */
        private $licenseContent;

        /**
         * @var string
         */
        private $outputDirectory;

        /**
         * @var KeyPairSuite
         */
        private $keyPairSuite;

        /**
         * LicenseWriter constructor.
         * @param $licenseContent string
         * @param $publicKeyPath string
         * @param $outputDirectory string
         */
        public function __construct($licenseContent, $publicKeyPath, $outputDirectory) {
            $this->licenseContent = trim($licenseContent);
            $this->outputDirectory = $outputDirectory;
            $this->keyPairSuite = new KeyPairSuite($publicKeyPath);
        }

        public function throwInvalidLicenseException($previousException = null) {
            // If we have a previous exception with a message, use it; otherwise use generic message
            $message = "The license entered was not valid.";
            if ($previousException instanceof Exception) {
                $prevMessage = $previousException->getMessage();
                if (!empty($prevMessage) && $prevMessage !== $message) {
                    $message = $prevMessage;
                }
            }
            throw new InvalidLicenseException($message,
                LocalizableExceptionDefinition::$INVALID_POSTED_LICENSE_ERROR, null, $previousException);
        }

        public function getLicenseData() {
            $licenseReader = new LicenseReader($this->keyPairSuite);
            $debugFile = dirname(__FILE__) . '/../../logs/mftp_debug.log';
            
            // Ensure logs directory exists
            $logsDir = dirname($debugFile);
            if (!is_dir($logsDir)) {
                @mkdir($logsDir, 0755, true);
            }
            
            try {
                // Log the license content we're about to process (first 200 chars for debugging)
                $licenseContentPreview = substr($this->getLicenseContent(), 0, 200);
                @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseWriter): Starting license processing. Content preview: " . $licenseContentPreview . "...\n", FILE_APPEND);
                
                // readLicenseString expects (licensePath, licenseContent) but we're calling from POST
                // Use a placeholder path for POST requests
                $licenseData = $licenseReader->readLicenseString("POST", $this->getLicenseContent());
                
                // Validate that license data is valid and contains required fields
                if (!is_array($licenseData) || empty($licenseData)) {
                    // Log what we got for debugging
                    @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseWriter): getLicenseData returned non-array or empty. Type: " . gettype($licenseData) . ", Value: " . var_export($licenseData, true) . "\n", FILE_APPEND);
                    $this->throwInvalidLicenseException();
                }
                
                // Validate email field exists and is not empty (same validation as RequestDispatcher)
                if (!array_key_exists('email', $licenseData)) {
                    $availableFields = implode(', ', array_keys($licenseData));
                    @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseWriter): License missing email field. Available fields: " . $availableFields . "\n", FILE_APPEND);
                    @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseWriter): Full license data: " . json_encode($licenseData) . "\n", FILE_APPEND);
                    $this->throwInvalidLicenseException();
                }
                
                // Check if email value is empty/null/whitespace
                $emailRaw = $licenseData['email'];
                $email = ($emailRaw !== null && $emailRaw !== '') ? trim((string)$emailRaw) : '';
                if ($email === '') {
                    $availableFields = implode(', ', array_keys($licenseData));
                    $emailType = gettype($emailRaw);
                    $emailValue = var_export($emailRaw, true);
                    @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseWriter): License email field exists but is empty. Type: " . $emailType . ", Raw value: " . $emailValue . "\n", FILE_APPEND);
                    @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseWriter): Full license data: " . json_encode($licenseData) . "\n", FILE_APPEND);
                    $this->throwInvalidLicenseException();
                }
                
                @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseWriter): License validation passed. Email: " . $email . "\n", FILE_APPEND);
                return $licenseData;
            } catch (InvalidLicenseException $e) {
                // Re-throw InvalidLicenseException directly to preserve the original message
                $errorMsg = $e->getMessage();
                @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseWriter): Re-throwing InvalidLicenseException: " . $errorMsg . "\n", FILE_APPEND);
                throw $e;
            } catch (LicensingException $e) {
                // For other LicensingExceptions, wrap but preserve the message
                $errorMsg = $e->getMessage();
                @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseWriter): Caught LicensingException: " . $errorMsg . "\n", FILE_APPEND);
                // Create new exception with the original message
                throw new InvalidLicenseException($errorMsg,
                    LocalizableExceptionDefinition::$INVALID_POSTED_LICENSE_ERROR, null, $e);
            } catch (Exception $e) {
                // Catch any other exceptions and convert to invalid license
                $errorMsg = $e->getMessage();
                $errorType = get_class($e);
                @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseWriter): Caught Exception ($errorType): " . $errorMsg . "\n", FILE_APPEND);
                @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseWriter): Exception trace: " . $e->getTraceAsString() . "\n", FILE_APPEND);
                throw new InvalidLicenseException($errorMsg,
                    LocalizableExceptionDefinition::$INVALID_POSTED_LICENSE_ERROR, null, $e);
            }
        }

        public function getProPackageID($email) {
            $proIDGenerator = new ProPackageIDGenerator(strtolower($email));
            // Without anything else to work with, we're basically building the ID by sha($email + $email)
            // Not super secure any more but should still be good enough for our purposes

            return $proIDGenerator->idFromEmail($email);
        }

        public function getExistingLicense() {
            if (MONSTA_LICENSE_PATH !== "") {
                $licenseReader = new LicenseReader($this->keyPairSuite);
                try {
                    return $licenseReader->readLicense(MONSTA_LICENSE_PATH);
                } catch (Exception $e) {
                }
            }
            return null;
        }

        private function validateNewLicenseAgainstExisting() {
            $newLicense = $this->getLicenseData();
            $existingLicense = $this->getExistingLicense();
            if ($existingLicense === null)
                return;

            if ($newLicense["expiryDate"] < $existingLicense["expiryDate"])
                throw new LicensingException("The new license was not saved as its expiry date is earlier than the 
                current license.", LocalizableExceptionDefinition::$REPLACEMENT_LICENSE_OLDER_ERROR);
        }

        /**
         * @return string
         */
        public function getOutputDirectory() {
            return normalizePath($this->outputDirectory);
        }

        /**
         * @return string
         */
        public function getLicenseContent() {
            return $this->licenseContent;
        }

        public function getConfigOutputPath() {
            return PathOperations::join($this->getOutputDirectory(), "config_pro.php");
        }

        public function writeLicense($proPackageID, $licenseContent) {
            $this->checkOutputDirectoryWritable();

            $configBuilder = new ProConfigBuilder($proPackageID);

            $licensePath = PathOperations::join($this->getOutputDirectory(),
                $configBuilder->generateRelativeLicensePath());

            if (file_put_contents($licensePath, $licenseContent) === false) {
                $errorPath = basename(dirname($licensePath)) . "/" . basename($licensePath);
                throw new LocalizableException("Could not write license file to $errorPath",
                    LocalizableExceptionDefinition::$LICENSE_WRITE_ERROR, array("path" => $errorPath));
            }
        }

        public function writeConfig($configContent) {
            $this->checkOutputDirectoryWritable();

            if (file_put_contents($this->getConfigOutputPath(), $configContent) === false) {
                $path = normalizePath($this->getConfigOutputPath());
                $errorPath = basename(dirname($path)) . "/" . basename($path);
                throw new LocalizableException("Could not write pro config to " . $errorPath,
                    LocalizableExceptionDefinition::$LICENSE_WRITE_ERROR,
                    array("path" => $errorPath));
            }
        }

        public function renderConfig($configTemplatePath, $proPackageID) {
            $configBuilder = new ProConfigBuilder($proPackageID);
            return $configBuilder->renderProConfig($configTemplatePath);
        }

        public function writeProFiles($configTemplatePath) {
            $this->validateNewLicenseAgainstExisting();
            $newLicense = $this->getLicenseData();
            $proPackageId = $this->getProPackageID($newLicense['email']);
            $this->writeLicense($proPackageId, $this->getLicenseContent());
            $configContent = $this->renderConfig($configTemplatePath, $proPackageId);
            $this->writeConfig($configContent);
        }

        private function checkOutputDirectoryWritable() {
            $outputDir = $this->getOutputDirectory();
            
            // Create directory if it doesn't exist
            if (!file_exists($outputDir)) {
                if (!@mkdir($outputDir, 0755, true)) {
                    $path = normalizePath($outputDir);
                    $errorPath = basename(dirname($path)) . "/" . basename($path);
                    throw new LocalizableException("Could not create license directory " . $errorPath . ", check parent directory permissions.",
                        LocalizableExceptionDefinition::$LICENSE_DIRECTORY_NOT_WRITABLE_ERROR,
                        array("path" => $errorPath));
                }
            }
            
            // Check if directory is writable
            if (!is_writable($outputDir)) {
                $path = normalizePath($outputDir);
                $errorPath = basename(dirname($path)) . "/" . basename($path);
                throw new LocalizableException("License directory " . $errorPath . " not writable, check its permissions.",
                    LocalizableExceptionDefinition::$LICENSE_DIRECTORY_NOT_WRITABLE_ERROR,
                    array("path" => $errorPath));
            }
        }
    }