<?php

    require_once(dirname(__FILE__) . '/Exceptions.php');
    require_once(dirname(__FILE__) . '/../lib/file_operations.php');

    class LicenseReader {
        /**
         * @var KeyPairSuite
         */
        private $keyPairSuite;

        public function __construct($keyPairSuite) {
            $this->keyPairSuite = $keyPairSuite;
        }

        public function readLicense($licensePath) {
            if(!file_exists($licensePath))
                return null;

            // Retry logic for transient file system issues (common in containerized environments)
            // This handles cases where the file system is temporarily unavailable, file is locked,
            // or there are permission issues that resolve after a short delay
            $maxRetries = 3;
            $retryDelay = 0.1; // 100ms between retries
            $lastException = null;
            
            for ($attempt = 1; $attempt <= $maxRetries; $attempt++) {
                try {
                    // Use the existing file reading function which has proper error handling
                    $licenseContent = mftpFileGetContents($licensePath);
                    return $this->readLicenseString($licensePath, $licenseContent);
                    
                } catch (Exception $e) {
                    $lastException = $e;
                    
                    // Log the error for debugging (only log on first attempt to avoid spam)
                    if ($attempt === 1 && function_exists('mftpLog')) {
                        $errorDetails = sprintf(
                            "LicenseReader: Attempt %d/%d failed to read license file '%s': %s",
                            $attempt,
                            $maxRetries,
                            basename($licensePath),
                            $e->getMessage()
                        );
                        mftpLog(LOG_WARNING, $errorDetails);
                    }
                    
                    // Check if this is a transient error that might resolve on retry
                    $errorMessage = strtolower($e->getMessage());
                    $isTransientError = (
                        strpos($errorMessage, 'locked') !== false ||
                        strpos($errorMessage, 'permission') !== false ||
                        strpos($errorMessage, 'temporarily') !== false ||
                        strpos($errorMessage, 'resource temporarily') !== false ||
                        strpos($errorMessage, 'inaccessible') !== false
                    );
                    
                    // If this is not the last attempt and it might be a transient error, wait and retry
                    if ($attempt < $maxRetries && $isTransientError) {
                        usleep($retryDelay * 1000000); // Convert to microseconds
                        continue;
                    }
                    
                    // If it's not a transient error or we're on the last attempt, throw
                    if ($attempt === $maxRetries) {
                        // Log final failure
                        if (function_exists('mftpLog')) {
                            mftpLog(LOG_ERROR, sprintf(
                                "LicenseReader: Failed to read license file after %d attempts. Final error: %s",
                                $maxRetries,
                                $e->getMessage()
                            ));
                        }
                        // Wrap the exception with more context
                        throw new Exception("Failed to read license file after $maxRetries attempts. Last error: " . $e->getMessage(), 0, $e);
                    }
                }
            }
            
            // Should never reach here, but just in case
            if ($lastException) {
                throw $lastException;
            }
            
            return null;
        }

        public function extractEncodedDataFromLicense($licenseData) {
            $licenseLines = preg_split('/(?:=\s+|\r\n|\n|\r)/', $licenseData);
            $encodedData = "";

            foreach ($licenseLines as $line) {
                $line = trim($line);

                if ($line == "")
                    continue;

                if(strlen($line) >= 3 && substr($line, 0, 3) == "===" || substr($line, -3, 3) == "===")
                    continue;

                $encodedData .= $line;
            }

            return $encodedData;
        }

        /**
         * @param $licensePath
         * @param $licenseContent
         * @return mixed
         * @throws InvalidLicenseException
         */
        public function readLicenseString($licensePath, $licenseContent) {
            $encodedData = $this->extractEncodedDataFromLicense($licenseContent);

            try {
                $rawLicenseData = $this->keyPairSuite->base64DecodeAndDecrypt($encodedData);
            } catch (KeyPairException $e) {
                $errorPath = basename(dirname($licensePath)) . "/" . basename($licensePath);
                throw new InvalidLicenseException("Unable to read the license file at '$errorPath'.",
                    LocalizableExceptionDefinition::$LICENSE_READ_FAILED_ERROR, array('path' => $errorPath));
            }

            // Debug: Log the raw decrypted data before JSON parsing
            $debugFile = dirname(__FILE__) . '/../../logs/mftp_debug.log';
            @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseReader): Raw decrypted data length: " . strlen($rawLicenseData) . " bytes\n", FILE_APPEND);
            @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseReader): Raw decrypted data: " . $rawLicenseData . "\n", FILE_APPEND);
            
            $licenseData = json_decode($rawLicenseData, true);
            
            // Check if JSON decode failed
            if ($licenseData === null && json_last_error() !== JSON_ERROR_NONE) {
                @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseReader): JSON decode failed: " . json_last_error_msg() . "\n", FILE_APPEND);
                throw new InvalidLicenseException("License data is not valid JSON: " . json_last_error_msg(),
                    LocalizableExceptionDefinition::$LICENSE_READ_FAILED_ERROR, array('path' => $licensePath));
            }
            
            // Debug: Log the parsed JSON structure
            @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseReader): JSON decoded successfully. Fields: " . implode(', ', array_keys($licenseData)) . "\n", FILE_APPEND);
            if (isset($licenseData['email'])) {
                $emailType = gettype($licenseData['email']);
                $emailLen = is_string($licenseData['email']) ? strlen($licenseData['email']) : 'N/A';
                @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseReader): Email field - Type: $emailType, Length: $emailLen, Value: '" . $licenseData['email'] . "'\n", FILE_APPEND);
            } else {
                @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] LICENSE DEBUG (LicenseReader): Email field NOT PRESENT\n", FILE_APPEND);
            }
            
            return $licenseData;
        }
    }