<?php

require_once(dirname(__FILE__) . '/../constants.php');
require_once(dirname(__FILE__) . '/SecureTempFileManager.php');
require_once(dirname(__FILE__) . '/helpers.php');
require_once(dirname(__FILE__) . '/logging.php');

/**
 * Temporary File Cleanup Scheduler
 * 
 * Handles scheduled cleanup operations for temporary files across the application.
 * Can be run via cron job, web request, or manual execution.
 * 
 * Usage:
 * - Via CLI: php temp_cleanup_scheduler.php
 * - Via web: include this file and call performScheduledCleanup()
 * - Via cron: every 15 minutes - /usr/bin/php /path/to/temp_cleanup_scheduler.php
 */
class TempCleanupScheduler {
    
    private $secureTempManager;
    private $config;
    private $stats;
    
    public function __construct() {
        $this->secureTempManager = SecureTempFileManager::getInstance();
        $this->initializeConfig();
        $this->stats = [
            'start_time' => time(),
            'cleaned_files' => 0,
            'cleaned_directories' => 0,
            'errors' => 0,
            'total_size_freed' => 0
        ];
    }
    
    /**
     * Initialize cleanup configuration
     */
    private function initializeConfig() {
        $this->config = [
            'enable_legacy_cleanup' => true,        // Clean up existing non-managed temp files
            'enable_secure_cleanup' => true,        // Clean up SecureTempFileManager files
            'enable_transfer_cleanup' => true,      // Clean up transfer directory
            'enable_session_cleanup' => true,       // Clean up session-related temp files
            'enable_upload_cleanup' => true,        // Clean up upload temp files
            'max_execution_time' => 300,            // 5 minutes max execution
            'memory_limit' => '128M',               // Memory limit for cleanup process
            'dry_run' => false,                     // Set to true to simulate cleanup without deleting
            'verbose_logging' => false              // Enable detailed logging
        ];
        
        // Allow configuration override
        if (defined('MFTP_CLEANUP_CONFIG')) {
            $this->config = array_merge($this->config, MFTP_CLEANUP_CONFIG);
        }
    }
    
    /**
     * Perform complete scheduled cleanup
     */
    public function performScheduledCleanup(): array {
        mftpLog(LOG_INFO, "TempCleanupScheduler: Starting scheduled cleanup");
        
        // Set execution limits
        $this->setExecutionLimits();
        
        try {
            // 1. Cleanup SecureTempFileManager tracked files
            if ($this->config['enable_secure_cleanup']) {
                $this->cleanupSecureManagedFiles();
            }
            
            // 2. Cleanup legacy transfer directories
            if ($this->config['enable_transfer_cleanup']) {
                $this->cleanupTransferDirectories();
            }
            
            // 3. Cleanup legacy temporary files
            if ($this->config['enable_legacy_cleanup']) {
                $this->cleanupLegacyTempFiles();
            }
            
            // 4. Cleanup session-related temporary files
            if ($this->config['enable_session_cleanup']) {
                $this->cleanupSessionTempFiles();
            }
            
            // 5. Cleanup upload temporary files
            if ($this->config['enable_upload_cleanup']) {
                $this->cleanupUploadTempFiles();
            }
            
            // 6. System maintenance
            $this->performSystemMaintenance();
            
        } catch (Exception $e) {
            $this->stats['errors']++;
            mftpLog(LOG_ERROR, "TempCleanupScheduler: Error during cleanup: " . $e->getMessage());
        }
        
        $this->finalizeStats();
        $this->logCleanupResults();
        
        return $this->stats;
    }
    
    /**
     * Set execution limits for cleanup process
     */
    private function setExecutionLimits() {
        if (function_exists('set_time_limit')) {
            set_time_limit($this->config['max_execution_time']);
        }
        
        if (function_exists('ini_set')) {
            ini_set('memory_limit', $this->config['memory_limit']);
        }
    }
    
    /**
     * Cleanup SecureTempFileManager tracked files
     */
    private function cleanupSecureManagedFiles() {
        $cleaned = $this->secureTempManager->cleanupExpiredFiles();
        $this->stats['cleaned_files'] += $cleaned;
        
        if ($this->config['verbose_logging']) {
            mftpLog(LOG_DEBUG, "TempCleanupScheduler: Cleaned $cleaned secure managed files");
        }
    }
    
    /**
     * Cleanup transfer directories (existing legacy cleanup)
     */
    private function cleanupTransferDirectories() {
        try {
            clearOldTransfers(); // Use existing function
            
            // Additional cleanup for orphaned transfer directories
            $transferDir = getMonstaSharedTransferDirectory();
            $cleaned = $this->cleanupDirectoryContents($transferDir, 3600); // 1 hour timeout
            $this->stats['cleaned_directories'] += $cleaned;
            
        } catch (Exception $e) {
            $this->stats['errors']++;
            mftpLog(LOG_ERROR, "TempCleanupScheduler: Error cleaning transfer directories: " . $e->getMessage());
        }
    }
    
    /**
     * Cleanup legacy temporary files
     */
    private function cleanupLegacyTempFiles() {
        $tempDirs = [
            sys_get_temp_dir(),
            monstaGetTempDirectory()
        ];
        
        foreach ($tempDirs as $tempDir) {
            if (!is_dir($tempDir)) {
                continue;
            }
            
            try {
                $cleaned = $this->cleanupLegacyFilesInDirectory($tempDir);
                $this->stats['cleaned_files'] += $cleaned;
                
            } catch (Exception $e) {
                $this->stats['errors']++;
                mftpLog(LOG_ERROR, "TempCleanupScheduler: Error cleaning legacy temp files in $tempDir: " . $e->getMessage());
            }
        }
    }
    
    /**
     * Cleanup session-related temporary files
     */
    private function cleanupSessionTempFiles() {
        try {
            $sessionDir = session_save_path();
            if (empty($sessionDir)) {
                $sessionDir = sys_get_temp_dir();
            }
            
            if (is_dir($sessionDir)) {
                $cleaned = $this->cleanupSessionFilesInDirectory($sessionDir);
                $this->stats['cleaned_files'] += $cleaned;
            }
            
        } catch (Exception $e) {
            $this->stats['errors']++;
            mftpLog(LOG_ERROR, "TempCleanupScheduler: Error cleaning session temp files: " . $e->getMessage());
        }
    }
    
    /**
     * Cleanup upload temporary files
     */
    private function cleanupUploadTempFiles() {
        try {
            $uploadTmpDir = ini_get('upload_tmp_dir') ?: sys_get_temp_dir();
            
            if (is_dir($uploadTmpDir)) {
                $cleaned = $this->cleanupUploadFilesInDirectory($uploadTmpDir);
                $this->stats['cleaned_files'] += $cleaned;
            }
            
        } catch (Exception $e) {
            $this->stats['errors']++;
            mftpLog(LOG_ERROR, "TempCleanupScheduler: Error cleaning upload temp files: " . $e->getMessage());
        }
    }
    
    /**
     * Clean directory contents with age-based filtering
     */
    private function cleanupDirectoryContents(string $directory, int $maxAge = 3600): int {
        $cleanedCount = 0;
        
        if (!is_dir($directory)) {
            return 0;
        }
        
        try {
            $iterator = new DirectoryIterator($directory);
            $currentTime = time();
            
            foreach ($iterator as $file) {
                if ($file->isDot()) {
                    continue;
                }
                
                $filePath = $file->getPathname();
                $fileAge = $currentTime - $file->getMTime();
                
                if ($fileAge > $maxAge) {
                    $size = $file->isFile() ? $file->getSize() : 0;
                    
                    if (!$this->config['dry_run']) {
                        if ($file->isDir()) {
                            if ($this->recursiveRemoveDirectory($filePath)) {
                                $cleanedCount++;
                                $this->stats['total_size_freed'] += $size;
                            }
                        } else {
                            if (unlink($filePath)) {
                                $cleanedCount++;
                                $this->stats['total_size_freed'] += $size;
                            }
                        }
                    } else {
                        $cleanedCount++; // Dry run simulation
                        $this->stats['total_size_freed'] += $size;
                    }
                }
            }
            
        } catch (Exception $e) {
            mftpLog(LOG_ERROR, "TempCleanupScheduler: Error iterating directory $directory: " . $e->getMessage());
        }
        
        return $cleanedCount;
    }
    
    /**
     * Clean legacy files in directory (Monsta-specific patterns)
     */
    private function cleanupLegacyFilesInDirectory(string $directory): int {
        $cleanedCount = 0;
        $patterns = [
            'mftp_*',           // Monsta FTP temp files
            'monsta-*',         // Monsta prefixed files
            'http_fetch*',      // HTTP fetch temp files
            'mftp_chunked_*',   // Chunked upload files
            'extract-*'         // Archive extraction files
        ];
        
        foreach ($patterns as $pattern) {
            $files = glob($directory . DIRECTORY_SEPARATOR . $pattern);
            if ($files === false) {
                continue;
            }
            
            foreach ($files as $file) {
                if (is_file($file) && (time() - filemtime($file)) > 3600) { // 1 hour old
                    $size = filesize($file);
                    
                    if (!$this->config['dry_run']) {
                        if (unlink($file)) {
                            $cleanedCount++;
                            $this->stats['total_size_freed'] += $size;
                        }
                    } else {
                        $cleanedCount++;
                        $this->stats['total_size_freed'] += $size;
                    }
                }
            }
        }
        
        return $cleanedCount;
    }
    
    /**
     * Clean session files in directory
     */
    private function cleanupSessionFilesInDirectory(string $directory): int {
        $cleanedCount = 0;
        $sessionFiles = glob($directory . DIRECTORY_SEPARATOR . 'sess_*');
        
        if ($sessionFiles === false) {
            return 0;
        }
        
        $sessionMaxLifetime = ini_get('session.gc_maxlifetime') ?: 1440; // Default 24 minutes
        
        foreach ($sessionFiles as $file) {
            if ((time() - filemtime($file)) > $sessionMaxLifetime) {
                $size = filesize($file);
                
                if (!$this->config['dry_run']) {
                    if (unlink($file)) {
                        $cleanedCount++;
                        $this->stats['total_size_freed'] += $size;
                    }
                } else {
                    $cleanedCount++;
                    $this->stats['total_size_freed'] += $size;
                }
            }
        }
        
        return $cleanedCount;
    }
    
    /**
     * Clean upload files in directory
     */
    private function cleanupUploadFilesInDirectory(string $directory): int {
        $cleanedCount = 0;
        $uploadFiles = glob($directory . DIRECTORY_SEPARATOR . 'php*');
        
        if ($uploadFiles === false) {
            return 0;
        }
        
        foreach ($uploadFiles as $file) {
            if ((time() - filemtime($file)) > 7200) { // 2 hours old
                $size = filesize($file);
                
                if (!$this->config['dry_run']) {
                    if (unlink($file)) {
                        $cleanedCount++;
                        $this->stats['total_size_freed'] += $size;
                    }
                } else {
                    $cleanedCount++;
                    $this->stats['total_size_freed'] += $size;
                }
            }
        }
        
        return $cleanedCount;
    }
    
    /**
     * Perform system maintenance tasks
     */
    private function performSystemMaintenance() {
        // Clear opcode cache if available
        if (function_exists('opcache_reset')) {
            opcache_reset();
        }
        
        // Trigger garbage collection
        if (function_exists('gc_collect_cycles')) {
            gc_collect_cycles();
        }
    }
    
    /**
     * Recursively remove directory
     */
    private function recursiveRemoveDirectory(string $dir): bool {
        return $this->secureTempManager->cleanupTempFile($dir);
    }
    
    /**
     * Finalize statistics
     */
    private function finalizeStats() {
        $this->stats['end_time'] = time();
        $this->stats['execution_time'] = $this->stats['end_time'] - $this->stats['start_time'];
        $this->stats['total_cleaned'] = $this->stats['cleaned_files'] + $this->stats['cleaned_directories'];
    }
    
    /**
     * Log cleanup results
     */
    private function logCleanupResults() {
        $message = sprintf(
            "TempCleanupScheduler: Cleanup completed in %d seconds. Files: %d, Dirs: %d, Errors: %d, Size freed: %s",
            $this->stats['execution_time'],
            $this->stats['cleaned_files'],
            $this->stats['cleaned_directories'],
            $this->stats['errors'],
            $this->formatBytes($this->stats['total_size_freed'])
        );
        
        if ($this->stats['errors'] > 0) {
            mftpLog(LOG_WARNING, $message);
        } else {
            mftpLog(LOG_INFO, $message);
        }
    }
    
    /**
     * Format bytes for human readability
     */
    private function formatBytes(int $bytes): string {
        $units = ['B', 'KB', 'MB', 'GB'];
        $index = 0;
        
        while ($bytes >= 1024 && $index < count($units) - 1) {
            $bytes /= 1024;
            $index++;
        }
        
        return round($bytes, 2) . ' ' . $units[$index];
    }
    
    /**
     * Get cleanup statistics
     */
    public function getStatistics(): array {
        return $this->stats;
    }
    
    /**
     * Set configuration option
     */
    public function setConfig(string $key, $value): void {
        $this->config[$key] = $value;
    }
}

// CLI execution support
if (php_sapi_name() === 'cli' && basename(__FILE__) === basename($_SERVER['SCRIPT_NAME'])) {
    echo "Monsta FTP Temporary File Cleanup Scheduler\n";
    echo "==========================================\n\n";
    
    $scheduler = new TempCleanupScheduler();
    
    // Parse CLI arguments
    $options = getopt('dv', ['dry-run', 'verbose']);
    
    if (isset($options['d']) || isset($options['dry-run'])) {
        $scheduler->setConfig('dry_run', true);
        echo "Running in DRY RUN mode (no files will be deleted)\n";
    }
    
    if (isset($options['v']) || isset($options['verbose'])) {
        $scheduler->setConfig('verbose_logging', true);
        echo "Verbose logging enabled\n";
    }
    
    echo "\nStarting cleanup...\n";
    $stats = $scheduler->performScheduledCleanup();
    
    echo "\nCleanup Results:\n";
    echo "Files cleaned: " . $stats['cleaned_files'] . "\n";
    echo "Directories cleaned: " . $stats['cleaned_directories'] . "\n";
    echo "Total items: " . $stats['total_cleaned'] . "\n";
    echo "Errors: " . $stats['errors'] . "\n";
    echo "Execution time: " . $stats['execution_time'] . " seconds\n";
    echo "Memory freed: " . number_format($stats['total_size_freed']) . " bytes\n";
    
    echo "\nCleanup completed.\n";
}

/**
 * Helper function for web-based cleanup execution
 */
function performScheduledTempCleanup(): array {
    $scheduler = new TempCleanupScheduler();
    return $scheduler->performScheduledCleanup();
} 