<?php

    require_once(dirname(__FILE__) . "/../constants.php");
    require_once(dirname(__FILE__) . "/LocalizableException.php");

    function handleExceptionInRequest($exception) {
        if (MONSTA_DEBUG)
            error_log($exception->getTraceAsString());

        if(!headers_sent())
            header('HTTP/1.1 577 Monsta Exception', true, 577);
            // Custom code so we can determine if it's an exception we've handled. 77 is M in ASCII

        // Determine appropriate data value based on the type of request
        $defaultData = null;
        
        // For directory-related operations, return empty array to prevent sort() errors
        if (isset($_POST['request'])) {
            $request = json_decode($_POST['request'], true);
            if ($request && isset($request['actionName'])) {
                $directoryActions = [
                    'listDirectory',
                    'getDefaultPath', 
                    'changeDirectory',
                    'navigateToParent'
                ];
                
                if (in_array($request['actionName'], $directoryActions)) {
                    $defaultData = array(); // Empty array instead of null
                }
            }
        }

        // Return format that frontend expects: {"success": false, "data": array_or_null, "localizedErrors": array}
        $errResponse = array(
            'success' => false,
            'data' => $defaultData,
            'errors' => array($exception->getMessage()),
            'localizedErrors' => array(array(
                'errorName' => 'GENERIC_ERROR',
                'context' => array('message' => $exception->getMessage())
            ))
        );

        // Only add localization info if the exception has a getContext method
        if (is_a($exception, "LocalizableException") && method_exists($exception, 'getContext')) {
            $context = $exception->getContext();
            // Get error name from exception code using the lookup
            $errorName = LocalizableExceptionCodeLookup::codeToName($exception->getCode());
            if ($errorName === false) {
                $errorName = 'GENERIC_ERROR';
            }
            
            if ($context !== null) {
                $errResponse['context'] = $context;
                // Update localizedErrors with proper context
                $errResponse['localizedErrors'] = array(array(
                    'errorName' => $errorName,
                    'context' => $context
                ));
            } else {
                // Even without context, set the correct error name
                $errResponse['localizedErrors'] = array(array(
                    'errorName' => $errorName,
                    'context' => array('message' => $exception->getMessage())
                ));
            }
        }

        print json_encode($errResponse);
        exit();
    }

    function exitWith404($error = null) {
        header('HTTP/1.1 404 Not Found', true, 404);

        if(!is_null($error)){
            header("Content-type: text/plain");
            print($error);
        }

        exit();
    }

    function dieIfNotPOST() {
        if (!isset($_SERVER['REQUEST_METHOD']) || $_SERVER['REQUEST_METHOD'] != 'POST') {
            header('HTTP/1.1 405 Method Not Allowed', true, 405);
            header("Allow: POST");
            exit();
        }
    }