<?php

declare(strict_types=1);

require_once(dirname(__FILE__) . "/../constants.php");
require_once(dirname(__FILE__) . "/../enums/LogPriority.php");

includeMonstaConfig();

/**
 * Modern logging with PHP 8.1+ features
 */
function _mftpLogPriorityToText(int $priority): string {
    try {
        return LogPriority::fromLegacyValue($priority)->getText();
    } catch (InvalidArgumentException) {
        return "UNKNOWN";
    }
}

    function _mftpLogToFile(string $path, int $priority, string $message): void {
        // Ensure log directory exists
        $logDir = dirname($path);
        if (!is_dir($logDir)) {
            @mkdir($logDir, 0755, true);
        }
        
        // Only try to write if directory exists or was created successfully
        if (!is_dir($logDir)) {
            // Silently fail if we can't create the directory - don't break the application
            return;
        }
        
        $handle = fopen($path, "a");
        if ($handle === false) {
            // Silently fail if we can't open log file - don't break the application
            return;
        }

        fprintf($handle, "[%s] (%s) %s\n", _mftpLogPriorityToText($priority), date("c"), $message);

        fclose($handle);
    }

    function _mftpLogToSyslog(int $priority, string $message): void {
        $facility = defined("MFTP_LOG_SYSLOG_FACILITY") ? MFTP_LOG_SYSLOG_FACILITY : LOG_USER;

        if (!openlog("MONSTAFTP", LOG_ODELAY, $facility)) {
            // Silently fail if we can't open syslog - don't break the application
            return;
        }

        syslog($priority, $message);

        closelog();
    }

    function _mftpShouldLogToFile(): bool {
        return defined("MFTP_LOG_TO_FILE") && MFTP_LOG_TO_FILE
            && defined("MFTP_LOG_FILE_PATH") && !is_null(MFTP_LOG_FILE_PATH)
            && defined("MFTP_LOG_LEVEL_THRESHOLD");
    }

    function _mftpShouldLogToSyslog(): bool {
        return defined("MFTP_LOG_TO_SYSLOG") && MFTP_LOG_TO_SYSLOG;
    }

    function mftpLog(int $priority, string $message): void {
        // Modern logging with enum support
        $logPriority = LogPriority::fromLegacyValue($priority);
        
        if (_mftpShouldLogToFile()) {
            $threshold = defined('MFTP_LOG_LEVEL_THRESHOLD') 
                ? LogPriority::fromLegacyValue(MFTP_LOG_LEVEL_THRESHOLD)
                : LogPriority::getProductionThreshold();
                
            if ($logPriority->shouldLog($threshold)) {
                _mftpLogToFile(MFTP_LOG_FILE_PATH, $priority, $message);
            }
        }

        if (_mftpShouldLogToSyslog()) {
            _mftpLogToSyslog($priority, $message);
        }
    }

    function defaultMftpActionLog($action, $clientIp, $host, $username, $folderPath, $fileName, $error) {
        if (!defined("MFTP_ACTION_LOG_PATH")) {
            return;
        }

        if (is_null(MFTP_ACTION_LOG_PATH)) {
            return;
        }

        $handle = fopen(MFTP_ACTION_LOG_PATH, "a");
        if ($handle === false)
            return;

        fprintf($handle, "%s|%s|%s|%s|%s|%s|%s|%s\n",
            date("c"), $clientIp, $host, $username, $action, $folderPath, $fileName, $error);
        fclose($handle);
    }

    function mftpActionLog($action, $connection, $folderPath, $fileName, $error) {
        $clientIp = $_SERVER['REMOTE_ADDR'] ?? 'unknown';

        $configuration = $connection->getConfiguration();
        $host = $configuration->getHost();
        $username = $configuration->getRemoteUsername();

        // Modern callable syntax - avoid deprecated call_user_func
        if (defined("MFTP_ACTION_LOG_FUNCTION") && !is_null(MFTP_ACTION_LOG_FUNCTION)) {
            (MFTP_ACTION_LOG_FUNCTION)($action, $clientIp, $host, $username, $folderPath,
                $fileName, $error);
        } else {
            defaultMftpActionLog($action, $clientIp, $host, $username, $folderPath, $fileName, $error);
        }
    }