<?php
/**
 * Strict Mode Error Handler for Monsta FTP
 * 
 * This file provides comprehensive error handling for strict PHP environments.
 * It converts warnings and notices into exceptions for better error visibility
 * and provides graceful error handling for production environments.
 */

class StrictModeErrorHandler {
    
    private static $instance = null;
    private static $strictMode = false;
    private static $logFile = null;
    
    /**
     * Initialize the strict mode error handler
     * 
     * @param bool $enableStrictMode Whether to convert warnings/notices to exceptions
     * @param string|null $logFile Optional log file for error logging
     */
    public static function initialize(bool $enableStrictMode = true, ?string $logFile = null): void {
        if (self::$instance !== null) {
            return; // Already initialized
        }
        
        self::$strictMode = $enableStrictMode;
        self::$logFile = $logFile;
        self::$instance = new self();
        
        // Set custom error handler
        set_error_handler([self::class, 'handleError']);
        
        // Set custom exception handler for uncaught exceptions
        set_exception_handler([self::class, 'handleUncaughtException']);
        
        // Set shutdown function to catch fatal errors
        register_shutdown_function([self::class, 'handleShutdown']);
        
        // Configure error reporting based on mode
        if ($enableStrictMode) {
            // Use E_ALL for modern PHP versions (8.0+) where E_STRICT is deprecated
            if (PHP_VERSION_ID >= 80000) {
                error_reporting(E_ALL);
            } else {
                error_reporting(E_ALL | E_STRICT);
            }
            ini_set('display_errors', '1');
        }
    }
    
    /**
     * Custom error handler
     * 
     * @param int $severity Error severity level
     * @param string $message Error message
     * @param string $file File where error occurred
     * @param int $line Line number where error occurred
     * @throws ErrorException In strict mode for errors that should halt execution
     * @return bool
     */
    public static function handleError(int $severity, string $message, string $file, int $line): bool {
        // Don't handle errors that are suppressed with @
        if (!(error_reporting() & $severity)) {
            return false;
        }
        
        $errorInfo = [
            'severity' => $severity,
            'message' => $message,
            'file' => $file,
            'line' => $line,
            'time' => date('Y-m-d H:i:s'),
            'type' => self::getErrorTypeName($severity)
        ];
        
        // Log the error
        self::logError($errorInfo);
        
        // In strict mode, convert warnings and notices to exceptions
        if (self::$strictMode) {
            // Convert certain error types to exceptions
            if ($severity & (E_ERROR | E_WARNING | E_PARSE | E_CORE_ERROR | E_COMPILE_ERROR | E_USER_ERROR)) {
                throw new ErrorException($message, 0, $severity, $file, $line);
            }
        }
        
        // For notices and other minor errors, just log them in strict mode
        $minorErrors = E_NOTICE | E_USER_NOTICE | E_DEPRECATED | E_USER_DEPRECATED;
        if (PHP_VERSION_ID < 80000) {
            $minorErrors |= E_STRICT; // Only add E_STRICT for PHP < 8.0
        }
        
        if (self::$strictMode && ($severity & $minorErrors)) {
            // Log but don't throw exception - these are typically non-fatal
            return true;
        }
        
        // Return false to allow PHP's default error handler to also run
        return false;
    }
    
    /**
     * Handle uncaught exceptions
     * 
     * @param Throwable $exception The uncaught exception
     */
    public static function handleUncaughtException(Throwable $exception): void {
        $errorInfo = [
            'type' => 'Uncaught Exception',
            'class' => get_class($exception),
            'message' => $exception->getMessage(),
            'file' => $exception->getFile(),
            'line' => $exception->getLine(),
            'trace' => $exception->getTraceAsString(),
            'time' => date('Y-m-d H:i:s')
        ];
        
        self::logError($errorInfo);
        
        // In development/debug mode, show detailed error
        if (defined('MONSTA_DEBUG') && MONSTA_DEBUG) {
            self::displayDetailedError($errorInfo);
        } else {
            // In production, show generic error
            self::displayGenericError();
        }
        
        exit(1);
    }
    
    /**
     * Handle fatal errors during shutdown
     */
    public static function handleShutdown(): void {
        $error = error_get_last();
        
        if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            $errorInfo = [
                'type' => 'Fatal Error',
                'severity' => $error['type'],
                'message' => $error['message'],
                'file' => $error['file'],
                'line' => $error['line'],
                'time' => date('Y-m-d H:i:s')
            ];
            
            self::logError($errorInfo);
            
            // Don't display error details if output has already started
            if (!headers_sent()) {
                if (defined('MONSTA_DEBUG') && MONSTA_DEBUG) {
                    self::displayDetailedError($errorInfo);
                } else {
                    self::displayGenericError();
                }
            }
        }
    }
    
    /**
     * Get human-readable error type name
     * 
     * @param int $severity Error severity constant
     * @return string Error type name
     */
    private static function getErrorTypeName(int $severity): string {
        $errorTypes = [
            E_ERROR => 'Fatal Error',
            E_WARNING => 'Warning',
            E_PARSE => 'Parse Error',
            E_NOTICE => 'Notice',
            E_CORE_ERROR => 'Core Error',
            E_CORE_WARNING => 'Core Warning',
            E_COMPILE_ERROR => 'Compile Error',
            E_COMPILE_WARNING => 'Compile Warning',
            E_USER_ERROR => 'User Error',
            E_USER_WARNING => 'User Warning',
            E_USER_NOTICE => 'User Notice',
            E_RECOVERABLE_ERROR => 'Recoverable Error',
            E_DEPRECATED => 'Deprecated',
            E_USER_DEPRECATED => 'User Deprecated'
        ];
        
        // Only add E_STRICT for PHP versions < 8.0 where it's not deprecated
        if (PHP_VERSION_ID < 80000 && defined('E_STRICT')) {
            $errorTypes[E_STRICT] = 'Strict Standards';
        }
        
        return $errorTypes[$severity] ?? 'Unknown Error';
    }
    
    /**
     * Log error information
     * 
     * @param array $errorInfo Error information array
     */
    private static function logError(array $errorInfo): void {
        $logMessage = sprintf(
            "[%s] %s: %s in %s on line %d",
            $errorInfo['time'],
            $errorInfo['type'],
            $errorInfo['message'],
            $errorInfo['file'],
            $errorInfo['line']
        );
        
        // Add stack trace for exceptions
        if (isset($errorInfo['trace'])) {
            $logMessage .= "\nStack trace:\n" . $errorInfo['trace'];
        }
        
        // Log to file if specified
        if (self::$logFile) {
            error_log($logMessage . "\n", 3, self::$logFile);
        }
        
        // Also log to system error log
        error_log($logMessage);
    }
    
    /**
     * Display detailed error for development
     * 
     * @param array $errorInfo Error information array
     */
    private static function displayDetailedError(array $errorInfo): void {
        if (!headers_sent()) {
            header('HTTP/1.1 500 Internal Server Error');
            header('Content-Type: text/html; charset=UTF-8');
        }
        
        echo "<!DOCTYPE html>\n";
        echo "<html><head><title>Monsta FTP Error</title>";
        echo "<style>body{font-family:Arial,sans-serif;margin:20px;} .error{background:#f8d7da;border:1px solid #f5c6cb;padding:15px;border-radius:5px;} .trace{background:#f8f9fa;border:1px solid #dee2e6;padding:10px;margin-top:10px;font-family:monospace;font-size:12px;}</style>";
        echo "</head><body>";
        echo "<h1>Monsta FTP Error</h1>";
        echo "<div class='error'>";
        echo "<h2>{$errorInfo['type']}</h2>";
        echo "<p><strong>Message:</strong> " . htmlspecialchars($errorInfo['message']) . "</p>";
        echo "<p><strong>File:</strong> " . htmlspecialchars($errorInfo['file']) . "</p>";
        echo "<p><strong>Line:</strong> {$errorInfo['line']}</p>";
        echo "<p><strong>Time:</strong> {$errorInfo['time']}</p>";
        
        if (isset($errorInfo['trace'])) {
            echo "<div class='trace'><strong>Stack Trace:</strong><br><pre>" . htmlspecialchars($errorInfo['trace']) . "</pre></div>";
        }
        
        echo "</div></body></html>";
    }
    
    /**
     * Display generic error for production
     */
    private static function displayGenericError(): void {
        if (!headers_sent()) {
            header('HTTP/1.1 500 Internal Server Error');
            header('Content-Type: application/json');
        }
        
        echo json_encode([
            'success' => false,
            'error' => 'An internal server error occurred. Please try again later.',
            'data' => null
        ]);
    }
    
    /**
     * Enable strict mode temporarily
     */
    public static function enableStrictMode(): void {
        self::$strictMode = true;
        // Use appropriate error reporting level based on PHP version
        if (PHP_VERSION_ID >= 80000) {
            error_reporting(E_ALL);
        } else {
            error_reporting(E_ALL | E_STRICT);
        }
    }
    
    /**
     * Disable strict mode temporarily
     */
    public static function disableStrictMode(): void {
        self::$strictMode = false;
        error_reporting(E_ERROR | E_WARNING | E_PARSE);
    }
    
    /**
     * Check if strict mode is enabled
     * 
     * @return bool
     */
    public static function isStrictModeEnabled(): bool {
        return self::$strictMode;
    }
} 