<?php

require_once(dirname(__FILE__) . '/../constants.php');
require_once(dirname(__FILE__) . '/helpers.php');
require_once(dirname(__FILE__) . '/logging.php');

/**
 * SecureTempFileManager - Comprehensive temporary file management with security focus
 * 
 * Addresses security audit recommendations:
 * 1. Systematic temporary file cleanup
 * 2. Proper file permissions (600) for temporary files
 * 3. Secure temporary directory configurations
 * 4. Central file registry and tracking
 * 5. Automatic cleanup on script termination
 */
class SecureTempFileManager {
    
    private static $instance = null;
    private $tempFileRegistry = [];
    private $tempDirRegistry = [];
    private $config;
    private $shutdownRegistered = false;
    
    // Security configuration constants
    const SECURE_FILE_PERMISSIONS = 0600;  // Read/write for owner only
    const SECURE_DIR_PERMISSIONS = 0700;   // Read/write/execute for owner only
    const DEFAULT_CLEANUP_TIMEOUT = 3600;  // 1 hour default timeout
    const MAX_TEMP_FILES = 1000;           // Prevent memory exhaustion
    const CLEANUP_BATCH_SIZE = 50;         // Process files in batches
    
    private function __construct() {
        $this->initializeConfiguration();
        $this->registerShutdownHandler();
        
        mftpLog(LOG_DEBUG, "SecureTempFileManager: Initialized with secure temp management");
    }
    
    /**
     * Get singleton instance
     */
    public static function getInstance(): SecureTempFileManager {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Initialize security configuration
     */
    private function initializeConfiguration() {
        $this->config = [
            'enable_permission_enforcement' => true,
            'enable_automatic_cleanup' => true,
            'cleanup_timeout_seconds' => self::DEFAULT_CLEANUP_TIMEOUT,
            'max_tracked_files' => self::MAX_TEMP_FILES,
            'secure_temp_prefix' => 'mftp_secure_',
            'log_cleanup_actions' => true,
            'validate_paths' => true,
            'emergency_cleanup_threshold' => 0.9 // Clean when 90% full
        ];
        
        // Allow configuration override
        if (defined('MFTP_SECURE_TEMP_CONFIG')) {
            $this->config = array_merge($this->config, MFTP_SECURE_TEMP_CONFIG);
        }
    }
    
    /**
     * Create a secure temporary file with proper permissions
     */
    public function createSecureTemp(string $prefix = "", string $suffix = ""): string {
        $this->enforceRegistryLimits();
        
        $tempDir = $this->getSecureTempDirectory();
        $securePrefix = $this->config['secure_temp_prefix'] . $prefix;
        
        // Create temporary file
        $tempPath = monstaTempnam($tempDir, $securePrefix);
        if ($tempPath === false) {
            throw new RuntimeException("Failed to create secure temporary file");
        }
        
        // Add suffix if provided
        if (!empty($suffix)) {
            $newPath = $tempPath . $suffix;
            if (rename($tempPath, $newPath)) {
                $tempPath = $newPath;
            }
        }
        
        // Set secure permissions
        $this->setSecurePermissions($tempPath, self::SECURE_FILE_PERMISSIONS);
        
        // Register for cleanup
        $this->registerTempFile($tempPath);
        
        mftpLog(LOG_DEBUG, "SecureTempFileManager: Created secure temp file: $tempPath");
        
        return $tempPath;
    }
    
    /**
     * Create a secure temporary directory with proper permissions
     */
    public function createSecureTempDir(string $prefix = ""): string {
        $this->enforceRegistryLimits();
        
        $tempDir = $this->getSecureTempDirectory();
        $securePrefix = $this->config['secure_temp_prefix'] . $prefix;
        
        // Create unique directory name
        $tempDirPath = $tempDir . DIRECTORY_SEPARATOR . $securePrefix . uniqid();
        
        if (!mkdir($tempDirPath, self::SECURE_DIR_PERMISSIONS, true)) {
            throw new RuntimeException("Failed to create secure temporary directory: $tempDirPath");
        }
        
        // Ensure permissions are correct
        $this->setSecurePermissions($tempDirPath, self::SECURE_DIR_PERMISSIONS);
        
        // Register for cleanup
        $this->registerTempDir($tempDirPath);
        
        mftpLog(LOG_DEBUG, "SecureTempFileManager: Created secure temp directory: $tempDirPath");
        
        return $tempDirPath;
    }
    
    /**
     * Get secure temporary directory with proper configuration
     */
    private function getSecureTempDirectory(): string {
        $tempDir = monstaGetTempDirectory();
        
        // Create secure subdirectory if it doesn't exist
        $secureTempDir = $tempDir . DIRECTORY_SEPARATOR . 'mftp_secure';
        if (!file_exists($secureTempDir)) {
            if (!mkdir($secureTempDir, self::SECURE_DIR_PERMISSIONS, true)) {
                throw new RuntimeException("Failed to create secure temp directory: $secureTempDir");
            }
            $this->setSecurePermissions($secureTempDir, self::SECURE_DIR_PERMISSIONS);
        }
        
        return $secureTempDir;
    }
    
    /**
     * Set secure file/directory permissions
     */
    private function setSecurePermissions(string $path, int $permissions): bool {
        if (!$this->config['enable_permission_enforcement']) {
            return true;
        }
        
        $result = chmod($path, $permissions);
        if (!$result) {
            mftpLog(LOG_WARNING, "SecureTempFileManager: Failed to set permissions for: $path");
        }
        
        return $result;
    }
    
    /**
     * Register temporary file for cleanup
     */
    private function registerTempFile(string $path): void {
        $this->tempFileRegistry[$path] = [
            'created_at' => time(),
            'type' => 'file',
            'size' => 0
        ];
        
        // Update size if file exists
        if (file_exists($path)) {
            $this->tempFileRegistry[$path]['size'] = filesize($path);
        }
    }
    
    /**
     * Register temporary directory for cleanup
     */
    private function registerTempDir(string $path): void {
        $this->tempDirRegistry[$path] = [
            'created_at' => time(),
            'type' => 'directory'
        ];
    }
    
    /**
     * Manually cleanup a specific temporary file
     */
    public function cleanupTempFile(string $path): bool {
        if (!file_exists($path)) {
            // Remove from registry even if file doesn't exist
            unset($this->tempFileRegistry[$path]);
            return true;
        }
        
        $result = false;
        if (is_file($path)) {
            $result = unlink($path);
            if ($result) {
                unset($this->tempFileRegistry[$path]);
                if ($this->config['log_cleanup_actions']) {
                    mftpLog(LOG_DEBUG, "SecureTempFileManager: Cleaned up temp file: $path");
                }
            }
        } elseif (is_dir($path)) {
            $result = $this->recursiveRemoveDirectory($path);
            if ($result) {
                unset($this->tempDirRegistry[$path]);
                if ($this->config['log_cleanup_actions']) {
                    mftpLog(LOG_DEBUG, "SecureTempFileManager: Cleaned up temp directory: $path");
                }
            }
        }
        
        if (!$result) {
            mftpLog(LOG_ERROR, "SecureTempFileManager: Failed to cleanup: $path");
        }
        
        return $result;
    }
    
    /**
     * Cleanup all expired temporary files
     */
    public function cleanupExpiredFiles(): int {
        $cleanedCount = 0;
        $currentTime = time();
        $timeout = $this->config['cleanup_timeout_seconds'];
        
        // Cleanup expired files
        $expiredFiles = [];
        foreach ($this->tempFileRegistry as $path => $info) {
            if (($currentTime - $info['created_at']) > $timeout) {
                $expiredFiles[] = $path;
            }
        }
        
        // Process in batches to prevent memory issues
        $batches = array_chunk($expiredFiles, self::CLEANUP_BATCH_SIZE);
        foreach ($batches as $batch) {
            foreach ($batch as $path) {
                if ($this->cleanupTempFile($path)) {
                    $cleanedCount++;
                }
            }
        }
        
        // Cleanup expired directories
        $expiredDirs = [];
        foreach ($this->tempDirRegistry as $path => $info) {
            if (($currentTime - $info['created_at']) > $timeout) {
                $expiredDirs[] = $path;
            }
        }
        
        foreach ($expiredDirs as $path) {
            if ($this->cleanupTempFile($path)) {
                $cleanedCount++;
            }
        }
        
        if ($cleanedCount > 0 && $this->config['log_cleanup_actions']) {
            mftpLog(LOG_INFO, "SecureTempFileManager: Cleaned up $cleanedCount expired temp files/directories");
        }
        
        return $cleanedCount;
    }
    
    /**
     * Cleanup all registered temporary files (emergency cleanup)
     */
    public function cleanupAllFiles(): int {
        $cleanedCount = 0;
        
        // Cleanup all files
        foreach (array_keys($this->tempFileRegistry) as $path) {
            if ($this->cleanupTempFile($path)) {
                $cleanedCount++;
            }
        }
        
        // Cleanup all directories
        foreach (array_keys($this->tempDirRegistry) as $path) {
            if ($this->cleanupTempFile($path)) {
                $cleanedCount++;
            }
        }
        
        mftpLog(LOG_INFO, "SecureTempFileManager: Emergency cleanup completed, cleaned $cleanedCount items");
        
        return $cleanedCount;
    }
    
    /**
     * Recursively remove directory and contents
     */
    private function recursiveRemoveDirectory(string $dir): bool {
        if (!is_dir($dir)) {
            return false;
        }
        
        try {
            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
                RecursiveIteratorIterator::CHILD_FIRST
            );
            
            foreach ($iterator as $file) {
                if ($file->isDir()) {
                    if (!rmdir($file->getPathname())) {
                        return false;
                    }
                } else {
                    if (!unlink($file->getPathname())) {
                        return false;
                    }
                }
            }
            
            return rmdir($dir);
        } catch (Exception $e) {
            mftpLog(LOG_ERROR, "SecureTempFileManager: Error during recursive directory removal: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Enforce registry size limits to prevent memory exhaustion
     */
    private function enforceRegistryLimits(): void {
        $totalFiles = count($this->tempFileRegistry) + count($this->tempDirRegistry);
        
        if ($totalFiles >= $this->config['max_tracked_files']) {
            mftpLog(LOG_WARNING, "SecureTempFileManager: Registry limit reached, performing cleanup");
            $this->cleanupExpiredFiles();
        }
    }
    
    /**
     * Register shutdown handler for automatic cleanup
     */
    private function registerShutdownHandler(): void {
        if (!$this->shutdownRegistered) {
            register_shutdown_function([$this, 'shutdownCleanup']);
            $this->shutdownRegistered = true;
        }
    }
    
    /**
     * Shutdown cleanup handler
     */
    public function shutdownCleanup(): void {
        if ($this->config['enable_automatic_cleanup']) {
            $cleanedCount = $this->cleanupExpiredFiles();
            if ($cleanedCount > 0) {
                mftpLog(LOG_DEBUG, "SecureTempFileManager: Shutdown cleanup completed, cleaned $cleanedCount items");
            }
        }
    }
    
    /**
     * Get temporary file statistics
     */
    public function getStatistics(): array {
        $totalSize = 0;
        $fileCount = count($this->tempFileRegistry);
        $dirCount = count($this->tempDirRegistry);
        
        foreach ($this->tempFileRegistry as $info) {
            $totalSize += $info['size'];
        }
        
        return [
            'tracked_files' => $fileCount,
            'tracked_directories' => $dirCount,
            'total_items' => $fileCount + $dirCount,
            'estimated_size_bytes' => $totalSize,
            'cleanup_timeout' => $this->config['cleanup_timeout_seconds'],
            'secure_temp_dir' => $this->getSecureTempDirectory()
        ];
    }
    
    /**
     * Validate and secure an existing temporary file
     */
    public function secureExistingFile(string $path): bool {
        if (!file_exists($path)) {
            return false;
        }
        
        // Set secure permissions
        $result = $this->setSecurePermissions($path, self::SECURE_FILE_PERMISSIONS);
        
        // Register for cleanup if successful
        if ($result) {
            $this->registerTempFile($path);
        }
        
        return $result;
    }
    
    /**
     * Get configuration setting
     */
    public function getConfig(string $key) {
        return $this->config[$key] ?? null;
    }
    
    /**
     * Update configuration setting
     */
    public function setConfig(string $key, $value): void {
        $this->config[$key] = $value;
    }
} 