<?php

declare(strict_types=1);

require_once(dirname(__FILE__) . '/../enums/ConnectionState.php');
require_once(dirname(__FILE__) . '/../enums/LogPriority.php');

/**
 * Security Configuration Class (PHP 7.4+ Compatible)
 * 
 * Configuration class for security settings across different environments
 */
class ModernSecurityConfig
{
    public string $environment;
    public bool $strictMode;
    public LogPriority $logLevel;
    public int $maxLoginAttempts;
    public int $sessionTimeout;
    public bool $enforceHttps;
    public ?string $customDomain;
    public array $allowedHosts;
    public bool $enableCsrfProtection;
    public array $securityHeaders;

    /**
     * Constructor
     */
    public function __construct(
        string $environment,
        bool $strictMode = true,
        ?LogPriority $logLevel = null,
        int $maxLoginAttempts = 5,
        int $sessionTimeout = 1800,
        bool $enforceHttps = true,
        ?string $customDomain = null,
        array $allowedHosts = [],
        bool $enableCsrfProtection = true,
        array $securityHeaders = []
    ) {
        $this->environment = $environment;
        $this->strictMode = $strictMode;
        $this->logLevel = $logLevel ?? LogPriority::WARNING;
        $this->maxLoginAttempts = $maxLoginAttempts;
        $this->sessionTimeout = $sessionTimeout;
        $this->enforceHttps = $enforceHttps;
        $this->customDomain = $customDomain;
        $this->allowedHosts = $allowedHosts;
        $this->enableCsrfProtection = $enableCsrfProtection;
        $this->securityHeaders = $securityHeaders;
        
        // Validate environment
        $this->validateEnvironment();
    }
    
    /**
     * Create configuration for development environment
     */
    public static function forDevelopment(
        ?string $customDomain = null,
        ?array $allowedHosts = null,
        ?LogPriority $logLevel = null
    ): self {
        if ($allowedHosts === null) {
            $allowedHosts = ['localhost', '127.0.0.1'];
        }
        if ($logLevel === null) {
            $logLevel = LogPriority::DEBUG;
        }
        
        return new self(
            'development',
            false,
            $logLevel,
            10,
            3600,
            false,
            $customDomain,
            $allowedHosts,
            true,
            self::getDefaultDevelopmentHeaders()
        );
    }
    
    /**
     * Create configuration for production environment
     */
    public static function forProduction(
        string $domain,
        ?array $allowedHosts = null,
        ?LogPriority $logLevel = null
    ): self {
        if ($allowedHosts === null) {
            $allowedHosts = [];
        }
        if ($logLevel === null) {
            $logLevel = LogPriority::WARNING;
        }
        
        return new self(
            'production',
            true,
            $logLevel,
            3,
            1800,
            true,
            $domain,
            $allowedHosts,
            true,
            self::getDefaultProductionHeaders()
        );
    }
    
    /**
     * Get security level
     */
    public function getSecurityLevel(): string
    {
        switch ($this->environment) {
            case 'development':
                return 'low';
            case 'staging':
                return 'medium';
            case 'production':
                return 'high';
            default:
                return 'unknown';
        }
    }
    
    /**
     * Check if feature is enabled
     */
    public function isFeatureEnabled(string $feature): bool
    {
        switch ($feature) {
            case 'https':
                return $this->enforceHttps;
            case 'csrf':
                return $this->enableCsrfProtection;
            case 'strict_mode':
                return $this->strictMode;
            case 'session_security':
            case 'input_validation':
                return true;
            case 'rate_limiting':
                return $this->environment === 'production';
            case 'debug_logging':
                return $this->environment === 'development';
            default:
                return false;
        }
    }
    
    /**
     * Get session configuration
     */
    public function getSessionConfig(): array
    {
        $baseConfig = [
            'cookie_lifetime' => $this->sessionTimeout,
            'cookie_secure' => $this->enforceHttps,
            'cookie_httponly' => true,
            'cookie_samesite' => 'Strict',
            'use_strict_mode' => true,
            'use_cookies' => true,
            'use_only_cookies' => true,
        ];
        
        // Add environment-specific settings
        switch ($this->environment) {
            case 'development':
                return array_merge($baseConfig, [
                    'gc_maxlifetime' => 3600,
                    'gc_probability' => 1,
                    'gc_divisor' => 10,
                ]);
            case 'production':
                return array_merge($baseConfig, [
                    'gc_maxlifetime' => $this->sessionTimeout,
                    'gc_probability' => 1,
                    'gc_divisor' => 100,
                    'sid_length' => 48,
                    'sid_bits_per_character' => 6,
                ]);
            default:
                return $baseConfig;
        }
    }
    
    /**
     * Apply security headers
     */
    public function applySecurityHeaders(?callable $headerCallback = null): void
    {
        $headers = $this->getEffectiveHeaders();
        
        // Use callback if provided, otherwise use header function
        if ($headerCallback === null) {
            foreach ($headers as $name => $value) {
                header("$name: $value");
            }
        } else {
            foreach ($headers as $name => $value) {
                $headerCallback("$name: $value");
            }
        }
    }
    
    /**
     * Validate IP address
     */
    public function isAllowedHost(string $host): bool
    {
        // Return early if no restrictions
        if (empty($this->allowedHosts)) {
            return true;
        }
        
        // Check for exact match
        if (in_array($host, $this->allowedHosts, true)) {
            return true;
        }
        
        // Check for pattern match
        return array_any($this->allowedHosts, function($pattern) use ($host) {
            return fnmatch($pattern, $host);
        });
    }
    
    /**
     * Get rate limiting configuration based on environment
     */
    public function getRateLimitConfig(): array
    {
        switch ($this->environment) {
            case 'development':
                return [
                    'enabled' => false,
                    'max_requests' => 1000,
                    'time_window' => 3600,
                    'max_login_attempts' => $this->maxLoginAttempts,
                ];
            case 'staging':
                return [
                    'enabled' => true,
                    'max_requests' => 500,
                    'time_window' => 3600,
                    'max_login_attempts' => $this->maxLoginAttempts,
                ];
            case 'production':
                return [
                    'enabled' => true,
                    'max_requests' => 200,
                    'time_window' => 3600,
                    'max_login_attempts' => $this->maxLoginAttempts,
                ];
            default:
                return [
                    'enabled' => false,
                    'max_requests' => 100,
                    'time_window' => 3600,
                    'max_login_attempts' => 3,
                ];
        }
    }
    
    /**
     * Get effective headers merging defaults with custom
     */
    private function getEffectiveHeaders(): array
    {
        switch ($this->environment) {
            case 'development':
                $defaultHeaders = self::getDefaultDevelopmentHeaders();
                break;
            case 'production':
                $defaultHeaders = self::getDefaultProductionHeaders();
                break;
            default:
                $defaultHeaders = [];
                break;
        }
        
        // Merge arrays
        return array_merge($defaultHeaders, $this->securityHeaders);
    }
    
    /**
     * Get default development headers
     */
    private static function getDefaultDevelopmentHeaders(): array
    {
        return [
            'X-Frame-Options' => 'SAMEORIGIN',
            'X-Content-Type-Options' => 'nosniff',
            'X-XSS-Protection' => '1; mode=block',
            'Referrer-Policy' => 'strict-origin-when-cross-origin',
        ];
    }
    
    /**
     * Get default production headers
     */
    private static function getDefaultProductionHeaders(): array
    {
        return [
            'Strict-Transport-Security' => 'max-age=31536000; includeSubDomains; preload',
            'X-Frame-Options' => 'DENY',
            'X-Content-Type-Options' => 'nosniff',
            'X-XSS-Protection' => '1; mode=block',
            'Referrer-Policy' => 'strict-origin-when-cross-origin',
            'Content-Security-Policy' => "default-src 'self'; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-inline';",
            'X-Permitted-Cross-Domain-Policies' => 'none',
            'Cross-Origin-Embedder-Policy' => 'require-corp',
            'Cross-Origin-Opener-Policy' => 'same-origin',
            'Cross-Origin-Resource-Policy' => 'same-origin',
        ];
    }
    
    /**
     * Validate environment setting
     */
    private function validateEnvironment(): void
    {
        $validEnvironments = ['development', 'staging', 'production'];
        
        if (!in_array($this->environment, $validEnvironments, true)) {
            throw new InvalidArgumentException(
                "Invalid environment '{$this->environment}'. Valid environments: " . 
                implode(', ', $validEnvironments)
            );
        }
    }
    
    /**
     * Convert to array representation for debugging
     */
    public function toArray(): array
    {
        return [
            'environment' => $this->environment,
            'security_level' => $this->getSecurityLevel(),
            'strict_mode' => $this->strictMode,
            'log_level' => $this->logLevel->getText(),
            'max_login_attempts' => $this->maxLoginAttempts,
            'session_timeout' => $this->sessionTimeout,
            'enforce_https' => $this->enforceHttps,
            'custom_domain' => $this->customDomain,
            'allowed_hosts' => $this->allowedHosts,
            'csrf_protection' => $this->enableCsrfProtection,
            'security_headers_count' => count($this->securityHeaders),
            'rate_limiting' => $this->getRateLimitConfig(),
            'session_config' => $this->getSessionConfig(),
        ];
    }
    
    /**
     * Create JSON representation
     */
    public function toJson(int $flags = JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES): string
    {
        return json_encode($this->toArray(), $flags | JSON_THROW_ON_ERROR);
    }
}

/**
 * Helper function to check if array_any exists (PHP 8.4+) and provide fallback
 */
if (!function_exists('array_any')) {
    function array_any(array $array, callable $callback): bool
    {
        foreach ($array as $key => $value) {
            if ($callback($value, $key)) {
                return true;
            }
        }
        return false;
    }
} 