<?php

require_once(dirname(__FILE__) . '/../file_sources/Validation.php');
require_once(dirname(__FILE__) . '/../file_sources/PathOperations.php');
require_once(dirname(__FILE__) . '/LocalizableException.php');

/**
 * Enhanced Input Validator with security-focused validation
 * Extends the basic Validation class with comprehensive security checks
 */
class InputValidator extends Validation {
    
    // Maximum allowed file upload size (100MB by default)
    const MAX_UPLOAD_SIZE = 104857600;
    
    /**
     * Get the configured maximum upload size from system settings
     */
    private static function getMaxUploadSize() {
        if (defined('MFTP_MAX_UPLOAD_SIZE')) {
            return MFTP_MAX_UPLOAD_SIZE;
        }
        // Fallback to 100MB if not configured
        return 104857600; // 100MB
    }
    
    /**
     * Format bytes into human-readable units
     */
    private static function formatBytes($bytes, $precision = 2) {
        $units = array('B', 'KB', 'MB', 'GB', 'TB');
        
        if ($bytes == 0) {
            return '0 B';
        }
        
        $base = log($bytes, 1024);
        $unit = $units[floor($base)];
        $size = round(pow(1024, $base - floor($base)), $precision);
        
        return $size . ' ' . $unit;
    }
    

    
    // Maximum path length to prevent buffer overflow attacks
    const MAX_PATH_LENGTH = 4096;
    
    /**
     * Validate API request structure and required fields
     */
    public static function validateApiRequest($request) {
        if (!is_array($request)) {
            throw new InvalidArgumentException("Request must be an array");
        }
        
        if (!isset($request['actionName']) || !is_string($request['actionName'])) {
            throw new InvalidArgumentException("Request must contain a valid actionName");
        }
        
        // Validate action name against whitelist
        self::validateActionName($request['actionName']);
        
        // Validate context if present
        if (isset($request['context']) && !is_array($request['context'])) {
            throw new InvalidArgumentException("Request context must be an array");
        }
        
        return true;
    }
    
    /**
     * Validate action name against allowed actions
     */
    public static function validateActionName($actionName) {
        $allowedActions = [
            'listDirectory', 'uploadFile', 'deleteFile', 'makeDirectory',
            'deleteDirectory', 'rename', 'changePermissions', 'copy', 'testConnectAndAuthenticate',
            'checkSavedAuthExists', 'writeSavedAuth', 'readSavedAuth', 'readLicense',
            'getSystemVars', 'fetchRemoteFile', 'uploadFileToNewDirectory', 'downloadMultipleFiles',
            'createZip', 'setApplicationSettings', 'deleteMultiple', 'extractArchive',
            'updateLicense', 'reserveUploadContext', 'transferUploadToRemote', 'getRemoteFileSize',
            'getDefaultPath', 'downloadForExtract', 'cleanUpExtract', 'resetPassword',
            'forgotPassword', 'validateSavedAuthPassword', 'downloadLatestVersionArchive',
            'installLatestVersion', 'fetchFile', 'getFileContents', 'putFileContents',
            'uploadArchive'
        ];
        
        if (!in_array($actionName, $allowedActions, true)) {
            throw new InvalidArgumentException("Invalid action name: " . $actionName);
        }
        
        return true;
    }
    
    /**
     * Validate and sanitize file paths to prevent directory traversal
     */
    public static function validateFilePath($path, $allowAbsolute = false) {
        if (!is_string($path)) {
            throw new InvalidArgumentException("Path must be a string");
        }
        
        if (strlen($path) > self::MAX_PATH_LENGTH) {
            throw new InvalidArgumentException("Path too long (max " . self::MAX_PATH_LENGTH . " characters)");
        }
        
        // Check for null bytes
        if (strpos($path, "\0") !== false) {
            throw new InvalidArgumentException("Path contains null bytes");
        }
        
        // Check for directory traversal patterns
        if (self::containsDirectoryTraversal($path)) {
            throw new InvalidArgumentException("Path contains directory traversal sequences");
        }
        
        // Validate absolute paths if not allowed
        if (!$allowAbsolute && (substr($path, 0, 1) === '/' || preg_match('/^[a-zA-Z]:\\\\/', $path))) {
            throw new InvalidArgumentException("Absolute paths not allowed");
        }
        
        // Normalize the path
        return PathOperations::normalize($path);
    }
    
    /**
     * Check for directory traversal patterns
     */
    private static function containsDirectoryTraversal($path) {
        $patterns = [
            '../', '..\\', '..%2f', '..%2F', '..%5c', '..%5C',
            '%2e%2e%2f', '%2E%2E%2F', '%2e%2e%5c', '%2E%2E%5C',
            '....//....' // Double encoding attempts
        ];
        
        $lowerPath = strtolower($path);
        foreach ($patterns as $pattern) {
            if (strpos($lowerPath, $pattern) !== false) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Validate file upload parameters
     */
    public static function validateFileUpload($uploadPath, $remotePath, $fileSize = null) {
        // Validate remote path
        self::validateFilePath($remotePath, true);
        
        // Validate file size if provided
        if ($fileSize !== null) {
            self::validateInteger($fileSize);
            if ($fileSize > self::getMaxUploadSize()) {
                throw new InvalidArgumentException("File size exceeds maximum allowed (" . self::formatBytes(self::getMaxUploadSize()) . ")");
            }
            if ($fileSize < 0) {
                throw new InvalidArgumentException("Invalid file size");
            }
        }
        
        // Get file extension for MIME validation
        $extension = strtolower(pathinfo($remotePath, PATHINFO_EXTENSION));
        
        // Additional validation if local file exists
        if (file_exists($uploadPath)) {
            $actualSize = filesize($uploadPath);
            if ($actualSize > self::getMaxUploadSize()) {
                throw new InvalidArgumentException("File size exceeds maximum allowed (" . self::formatBytes(self::getMaxUploadSize()) . ")");
            }
            
            // Basic MIME type validation
            self::validateFileContent($uploadPath, $extension);
        }
        
        return true;
    }
    
    /**
     * Enhanced file content and MIME type validation
     * Implements comprehensive security audit requirements
     */
    private static function validateFileContent($filePath, $expectedExtension) {
        // FILE UPLOAD RESTRICTIONS DISABLED PER USER REQUEST
        // All file types are allowed without MIME type or content validation
        
        // Only perform basic file existence and size checks
        if (!file_exists($filePath)) {
            return;
        }
        
        // Get file size for validation
        $fileSize = filesize($filePath);
        if ($fileSize === false || $fileSize === 0) {
            throw new InvalidArgumentException("Unable to determine file size or file is empty");
        }
        
        // Skip all MIME type validation, signature validation, and content security checks
        // All file types are now allowed for upload
        
        return;
    }

    /**
     * Comprehensive MIME type validation with extensive mappings
     */
    private static function validateMimeType($filePath, $expectedExtension) {
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $detectedMimeType = finfo_file($finfo, $filePath);
        finfo_close($finfo);
        
        if ($detectedMimeType === false) {
            throw new InvalidArgumentException("Unable to detect MIME type for uploaded file");
        }
        
        // Comprehensive MIME type mappings for security audit compliance
        $mimeMap = [
            // Text files
            'txt' => ['text/plain', 'text/x-readme'],
            'csv' => ['text/csv', 'text/plain', 'application/csv'],
            'json' => ['application/json', 'text/plain', 'text/json'],
            'xml' => ['application/xml', 'text/xml', 'application/xhtml+xml'],
            'html' => ['text/html', 'application/xhtml+xml'],
            'htm' => ['text/html', 'application/xhtml+xml'],
            'css' => ['text/css', 'text/plain'],
            'js' => ['application/javascript', 'text/javascript', 'application/x-javascript', 'text/plain', 'text/csv'],
            'md' => ['text/plain', 'text/markdown', 'text/x-markdown'],
            'log' => ['text/plain', 'text/x-log'],
            
            // Programming languages
            'php' => ['text/x-php', 'text/plain', 'application/x-php', 'application/x-httpd-php', 'application/x-httpd-php-source', 'text/html'],
            'py' => ['text/x-python', 'text/plain', 'application/x-python-code'],
            'java' => ['text/x-java-source', 'text/plain'],
            'cpp' => ['text/x-c++src', 'text/plain', 'text/x-c'],
            'c' => ['text/x-csrc', 'text/plain', 'text/x-c'],
            'h' => ['text/x-chdr', 'text/plain', 'text/x-c'],
            'sql' => ['text/x-sql', 'text/plain', 'application/sql'],
            
            // Images
            'jpg' => ['image/jpeg'],
            'jpeg' => ['image/jpeg'],
            'png' => ['image/png'],
            'gif' => ['image/gif'],
            'bmp' => ['image/bmp', 'image/x-bmp', 'image/x-ms-bmp'],
            'svg' => ['image/svg+xml', 'text/xml'],
            'webp' => ['image/webp'],
            'ico' => ['image/x-icon', 'image/vnd.microsoft.icon'],
            'tiff' => ['image/tiff'],
            'tif' => ['image/tiff'],
            
            // Documents
            'pdf' => ['application/pdf'],
            'doc' => ['application/msword'],
            'docx' => ['application/vnd.openxmlformats-officedocument.wordprocessingml.document'],
            'xls' => ['application/vnd.ms-excel'],
            'xlsx' => ['application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'],
            'ppt' => ['application/vnd.ms-powerpoint'],
            'pptx' => ['application/vnd.openxmlformats-officedocument.presentationml.presentation'],
            'odt' => ['application/vnd.oasis.opendocument.text'],
            'ods' => ['application/vnd.oasis.opendocument.spreadsheet'],
            'odp' => ['application/vnd.oasis.opendocument.presentation'],
            
            // Archives
            'zip' => ['application/zip', 'application/x-zip-compressed'],
            'tar' => ['application/x-tar'],
            'gz' => ['application/gzip', 'application/x-gzip'],
            'bz2' => ['application/x-bzip2'],
            '7z' => ['application/x-7z-compressed'],
            'rar' => ['application/x-rar-compressed', 'application/vnd.rar'],
            'phar' => ['application/octet-stream', 'application/x-php', 'application/zip'], // PHP Archive
            
            // Audio/Video (if supported)
            'mp3' => ['audio/mpeg', 'audio/mp3'],
            'mp4' => ['video/mp4', 'application/octet-stream'],
            'wav' => ['audio/wav', 'audio/wave'],
            'avi' => ['video/x-msvideo', 'application/octet-stream'],
            'mov' => ['video/quicktime', 'application/octet-stream'],
            'mkv' => ['video/x-matroska', 'application/octet-stream'],
            'wmv' => ['video/x-ms-wmv', 'application/octet-stream'],
            'flv' => ['video/x-flv', 'application/octet-stream'],
            'webm' => ['video/webm', 'application/octet-stream'],
            'ogg' => ['audio/ogg', 'video/ogg', 'application/ogg'],
            
            // Fonts
            'ttf' => ['font/ttf', 'application/x-font-ttf'],
            'otf' => ['font/otf', 'application/x-font-otf'],
            'woff' => ['font/woff', 'application/font-woff'],
            'woff2' => ['font/woff2', 'application/font-woff2'],
            
            // Binary executables (should be blocked elsewhere but double-check)
            'exe' => [], // Empty array means always blocked
            'bat' => [],
            'cmd' => [],
            'com' => [],
            'scr' => [],
            'msi' => []
        ];
        
        // Check if extension is mapped
        if (!isset($mimeMap[$expectedExtension])) {
            // For unmapped extensions, log warning but allow
            if (function_exists('mftpLog')) {
                mftpLog(LOG_WARNING, "InputValidator: Unmapped file extension '$expectedExtension' with MIME type '$detectedMimeType'");
            }
            return;
        }
        
            $allowedMimes = $mimeMap[$expectedExtension];
        
        // Empty array means the file type should be blocked
        if (empty($allowedMimes)) {
            throw new InvalidArgumentException("File type '$expectedExtension' is not allowed for security reasons");
        }
        
        // Check if detected MIME type matches expected types
        if (!in_array($detectedMimeType, $allowedMimes, true)) {
            // For certain file types, be more lenient with MIME type validation
            // This handles cases where browsers misidentify MIME types during drag operations
            $lenientExtensions = ['js', 'css', 'txt', 'csv', 'json', 'xml', 'html', 'htm', 'php', 'py', 'java', 'cpp', 'c', 'h', 'sql'];
            
            if (in_array($expectedExtension, $lenientExtensions)) {
                // Allow upload but log the mismatch for audit purposes
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_WARNING, "InputValidator: MIME type mismatch allowed - detected '$detectedMimeType' for .$expectedExtension file");
                }
                return; // Allow the upload to proceed
            }
            
            // For other file types, maintain strict validation
            throw new InvalidArgumentException(
                "File content validation failed: detected MIME type '$detectedMimeType' " .
                "does not match expected types for '.$expectedExtension' extension. " .
                "Expected: " . implode(', ', $allowedMimes)
            );
        }
        
        // Log successful validation for audit trail
        if (function_exists('mftpLog')) {
            mftpLog(LOG_DEBUG, "InputValidator: File MIME type validation passed - '$expectedExtension' -> '$detectedMimeType'");
        }
    }

    /**
     * Validate file signature (magic numbers) for enhanced security
     */
    private static function validateFileSignature($filePath, $expectedExtension) {
        $handle = fopen($filePath, 'rb');
        if (!$handle) {
            throw new InvalidArgumentException("Unable to read file for signature validation");
        }
        
        // Read first 16 bytes for signature analysis
        $signature = fread($handle, 16);
        fclose($handle);
        
        if ($signature === false || strlen($signature) < 2) {
            throw new InvalidArgumentException("Unable to read file signature");
        }
        
        // Convert to hex for easier analysis
        $hexSignature = strtoupper(bin2hex($signature));
        
        // Common file signatures (magic numbers)
        $signatures = [
            'pdf' => ['255044462D'],  // %PDF-
            'zip' => ['504B0304', '504B0506'],  // PK..
            'jpg' => ['FFD8FFE0', 'FFD8FFE1', 'FFD8FFDB', 'FFD8FFEE'],  // JPEG markers
            'jpeg' => ['FFD8FFE0', 'FFD8FFE1', 'FFD8FFDB', 'FFD8FFEE'],
            'png' => ['89504E47'],  // PNG signature
            'gif' => ['474946383761', '474946383961'],  // GIF87a, GIF89a
            'bmp' => ['424D'],  // BM
            'gz' => ['1F8B'],  // gzip
            'tar' => [], // TAR files have signature at byte 257, skip for now
            'rar' => ['526172211A0700', '526172211A070100'],  // Rar!
            '7z' => ['377ABCAF271C'],  // 7z signature
            'exe' => ['4D5A'],  // MZ (DOS/Windows executable)
            'doc' => ['D0CF11E0A1B11AE1'],  // Microsoft Office
            'docx' => ['504B0304'],  // DOCX is a ZIP file
            'xlsx' => ['504B0304'],  // XLSX is a ZIP file
            'pptx' => ['504B0304'],  // PPTX is a ZIP file
        ];
        
        // Check signature if defined
        if (isset($signatures[$expectedExtension])) {
            $expectedSignatures = $signatures[$expectedExtension];
            
            if (!empty($expectedSignatures)) {
                $signatureMatched = false;
                foreach ($expectedSignatures as $expectedSig) {
                    if (substr($hexSignature, 0, strlen($expectedSig)) === $expectedSig) {
                        $signatureMatched = true;
                        break;
                    }
                }
                
                if (!$signatureMatched) {
                    throw new InvalidArgumentException(
                        "File signature validation failed for '.$expectedExtension': " .
                        "detected signature '$hexSignature' does not match expected signatures"
                    );
                }
                
                // Log successful signature validation
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_DEBUG, "InputValidator: File signature validation passed - '$expectedExtension'");
                }
            }
        }
    }

    /**
     * Enhanced content security validation
     */
    private static function validateFileContentSecurity($filePath, $expectedExtension, $fileSize) {
        // Get the configured maximum upload size
        $maxUploadSize = self::getMaxUploadSize();
        
        // Validate file size limits based on type (proportional to configured max)
        $typeSizeLimits = [
            'images' => [
                'extensions' => ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'svg', 'webp'],
                'limit' => min($maxUploadSize, 50 * 1024 * 1024) // 50MB or configured max, whichever is smaller
            ],
            'documents' => [
                'extensions' => ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx'],
                'limit' => min($maxUploadSize, 100 * 1024 * 1024) // 100MB or configured max
            ],
            'archives' => [
                'extensions' => ['zip', 'tar', 'gz', 'bz2', '7z', 'rar', 'phar'],
                'limit' => $maxUploadSize // Use full configured limit for archives
            ],
            'videos' => [
                'extensions' => ['mp4', 'avi', 'mov', 'mkv', 'wmv', 'flv', 'webm'],
                'limit' => $maxUploadSize // Use full configured limit for videos
            ],
            'audio' => [
                'extensions' => ['mp3', 'wav', 'ogg'],
                'limit' => min($maxUploadSize, 100 * 1024 * 1024) // 100MB or configured max
            ],
            'text' => [
                'extensions' => ['txt', 'csv', 'json', 'xml', 'html', 'css', 'js', 'php', 'py'],
                'limit' => min($maxUploadSize, 10 * 1024 * 1024) // 10MB or configured max
            ]
        ];
        
        foreach ($typeSizeLimits as $typeConfig) {
            if (in_array($expectedExtension, $typeConfig['extensions'], true)) {
                if ($fileSize > $typeConfig['limit']) {
                    throw new InvalidArgumentException(
                        "File size validation failed: $expectedExtension file size (" . self::formatBytes($fileSize) . ") " .
                        "exceeds limit for file type (" . self::formatBytes($typeConfig['limit']) . ")"
                    );
                }
                break;
            }
        }
        
        // Content validation disabled - allow all file types without analysis
        // This allows PHP, JS, HTML, and other script files to be uploaded
        // without security validation that could block legitimate files
    }

    /**
     * Validate script file content for security
     */
    private static function validateScriptFileContent($filePath, $extension) {
        $content = file_get_contents($filePath);
        if ($content === false) {
            throw new InvalidArgumentException("Unable to read script file content");
        }
        
        // Debug: Log what file is being validated
        error_log("MFTP DEBUG: Validating script file: $filePath (extension: $extension, size: " . strlen($content) . " bytes)");
        
        // Check for potentially dangerous patterns (more specific to avoid false positives)
        $dangerousPatterns = [
            '/\bexec\s*\(/i',  // word boundary to avoid matching "max_execution_time"
            '/\bsystem\s*\(/i',
            '/\bshell_exec\s*\(/i',
            '/\bpassthru\s*\(/i',
            '/\beval\s*\(/i',
            '/file_get_contents\s*\(\s*["\']php:\/\/input/i',
            '/\$_GET\s*\[\s*["\'][^"\']*["\']\s*\]/i',
            '/\$_POST\s*\[\s*["\'][^"\']*["\']\s*\]/i',
            '/<\?php.*?(eval|exec|system)/is',
        ];
        
        foreach ($dangerousPatterns as $i => $pattern) {
            if (preg_match($pattern, $content, $matches)) {
                error_log("MFTP DEBUG: Pattern " . ($i + 1) . " matched: " . $pattern);
                error_log("MFTP DEBUG: Match: " . $matches[0]);
                error_log("MFTP DEBUG: File content preview: " . substr($content, 0, 500));
                throw new InvalidArgumentException(
                    "Script file validation failed: potentially dangerous code pattern detected in $extension file"
                );
            }
        }
        
        error_log("MFTP DEBUG: Script file validation passed for: $filePath");
    }

    /**
     * Validate HTML content for security
     */
    private static function validateHtmlContent($filePath) {
        $content = file_get_contents($filePath);
        if ($content === false) {
            throw new InvalidArgumentException("Unable to read HTML file content");
        }
        
        // Check for dangerous HTML patterns
        $dangerousPatterns = [
            '/<script[^>]*>.*?<\/script>/is',
            '/javascript:/i',
            '/on\w+\s*=/i', // onclick, onload, etc.
            '/<iframe[^>]*>/i',
            '/<object[^>]*>/i',
            '/<embed[^>]*>/i',
        ];
        
        foreach ($dangerousPatterns as $pattern) {
            if (preg_match($pattern, $content)) {
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_WARNING, "InputValidator: Potentially dangerous HTML content detected");
                }
                // Note: For HTML files, we log warning but don't block as it might be legitimate
                break;
            }
        }
    }

    /**
     * Validate SVG content for security
     */
    private static function validateSvgContent($filePath) {
        $content = file_get_contents($filePath);
        if ($content === false) {
            throw new InvalidArgumentException("Unable to read SVG file content");
        }
        
        // SVG files can contain JavaScript, so check for dangerous patterns
        $dangerousPatterns = [
            '/<script[^>]*>.*?<\/script>/is',
            '/javascript:/i',
            '/on\w+\s*=/i',
            '/<foreignObject[^>]*>/i',
        ];
        
        foreach ($dangerousPatterns as $pattern) {
            if (preg_match($pattern, $content)) {
                throw new InvalidArgumentException(
                    "SVG file validation failed: potentially dangerous content detected"
                );
            }
        }
    }

    /**
     * Validate JSON content
     */
    private static function validateJsonContent($filePath) {
        $content = file_get_contents($filePath);
        if ($content === false) {
            throw new InvalidArgumentException("Unable to read JSON file content");
        }
        
        $decoded = json_decode($content);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new InvalidArgumentException(
                "JSON file validation failed: " . json_last_error_msg()
            );
        }
    }

    /**
     * Validate XML content
     */
    private static function validateXmlContent($filePath) {
        $content = file_get_contents($filePath);
        if ($content === false) {
            throw new InvalidArgumentException("Unable to read XML file content");
        }
        
        // Disable external entity loading for security
        $prevSetting = libxml_disable_entity_loader(true);
        $prevUseErrors = libxml_use_internal_errors(true);
        
        try {
            $xml = simplexml_load_string($content);
            if ($xml === false) {
                $errors = libxml_get_errors();
                $errorMessage = 'XML validation failed';
                if (!empty($errors)) {
                    $errorMessage .= ': ' . $errors[0]->message;
                }
                throw new InvalidArgumentException($errorMessage);
            }
        } finally {
            libxml_disable_entity_loader($prevSetting);
            libxml_use_internal_errors($prevUseErrors);
        }
    }

    /**
     * Optional virus scanning if antivirus is available
     */
    private static function performVirusScan($filePath) {
        // Check for ClamAV PHP extension
        if (extension_loaded('clamav')) {
            try {
                $scanResult = cl_scanfile($filePath);
                if ($scanResult !== CL_CLEAN) {
                    throw new InvalidArgumentException("File failed virus scan");
                }
                
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_INFO, "InputValidator: File passed virus scan");
                }
            } catch (Exception $e) {
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_WARNING, "InputValidator: Virus scan failed: " . $e->getMessage());
                }
                // Don't fail upload if virus scanner is not working properly
            }
        }
        
        // Check for command-line ClamAV
        if (!extension_loaded('clamav') && function_exists('exec')) {
            $clamdPath = '/usr/bin/clamdscan';
            if (file_exists($clamdPath)) {
                $escapedPath = escapeshellarg($filePath);
                $output = [];
                $returnCode = 0;
                
                exec("$clamdPath --no-summary $escapedPath 2>&1", $output, $returnCode);
                
                if ($returnCode !== 0) {
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_WARNING, "InputValidator: File may have failed virus scan (return code: $returnCode)");
                    }
                    // Don't fail upload as clamdscan might not be properly configured
                } else {
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_DEBUG, "InputValidator: File passed command-line virus scan");
                    }
                }
            }
        }
    }
    
    /**
     * Validate connection configuration parameters
     */
    public static function validateConnectionConfig($connectionType, $config) {
        if (!is_array($config)) {
            throw new InvalidArgumentException("Connection configuration must be an array");
        }
        
        $connectionType = strtolower($connectionType);
        
        switch ($connectionType) {
            case 'ftp':
                self::validateFTPConfig($config);
                break;
            case 'sftp':
                self::validateSFTPConfig($config);
                break;
            default:
                throw new InvalidArgumentException("Unknown connection type: " . $connectionType);
        }
        
        return true;
    }
    
    /**
     * Validate FTP connection configuration
     */
    private static function validateFTPConfig($config) {
        // Required fields - only validate if host is provided and not null
        if (isset($config['host']) && $config['host'] !== null && !is_string($config['host'])) {
            throw new InvalidArgumentException("FTP host must be a string");
        }
        
        // Validate hostname/IP only if provided and not null
        if (isset($config['host']) && $config['host'] !== null) {
            self::validateHostname($config['host']);
        }
        
        // Validate port
        if (isset($config['port'])) {
            self::validatePort($config['port']);
        }
        
        // Validate username
        if (isset($config['username'])) {
            self::validateNonEmptyString($config['username'], 'username', true);
        }
        
        // Validate boolean flags
        if (isset($config['passive'])) {
            self::validateBoolean($config['passive']);
        }
        
        if (isset($config['ssl'])) {
            self::validateBoolean($config['ssl']);
        }
    }
    
    /**
     * Validate SFTP connection configuration
     */
    private static function validateSFTPConfig($config) {
        // Required fields - only validate if host is provided and not null
        if (isset($config['host']) && $config['host'] !== null && !is_string($config['host'])) {
            throw new InvalidArgumentException("SFTP host must be a string");
        }
        
        // Validate hostname/IP only if provided and not null
        if (isset($config['host']) && $config['host'] !== null) {
            self::validateHostname($config['host']);
        }
        
        // Validate port
        if (isset($config['port'])) {
            self::validatePort($config['port']);
        }
        
        // Validate username
        if (isset($config['remoteUsername'])) {
            self::validateNonEmptyString($config['remoteUsername'], 'remoteUsername', true);
        }
        
        // Validate authentication method
        if (isset($config['authenticationModeName'])) {
            $validMethods = ['Password', 'PublicKeyFile', 'HostKeyFile'];
            if (!in_array($config['authenticationModeName'], $validMethods, true)) {
                throw new InvalidArgumentException("Invalid authentication method");
            }
        }
    }
    
    /**
     * Validate hostname or IP address
     */
    private static function validateHostname($host) {
        $original = $host;
        
        // Remove any potential protocol prefix (http, https, ftp, sftp, ftps)
        $host = preg_replace('/^(https?|s?ftps?):\/\//', '', $host);
        
        // Remove username if present (user@host format)
        if (strpos($host, '@') !== false) {
            $parts = explode('@', $host);
            $host = end($parts); // Get the part after the last @
        }
        
        // Remove port if present
        if (strpos($host, ':') !== false) {
            $parts = explode(':', $host);
            $host = $parts[0];
        }
        
        // Remove path if present
        if (strpos($host, '/') !== false) {
            $parts = explode('/', $host);
            $host = $parts[0];
        }
        
        // Check for valid hostname or IP
        if (!filter_var($host, FILTER_VALIDATE_IP) && !filter_var($host, FILTER_VALIDATE_DOMAIN, FILTER_FLAG_HOSTNAME)) {
            throw new InvalidArgumentException("Invalid hostname or IP address: " . $original);
        }
        
        // Block obviously malicious hostname patterns (not in URLs, but in the actual hostname)
        $blockedPatterns = [
            'javascript:', 'data:', 'file:', 'gopher:',
            'telnet:', 'ldap:', 'dict:'
        ];
        
        foreach ($blockedPatterns as $pattern) {
            if (stripos($host, $pattern) !== false) {
                throw new InvalidArgumentException("Blocked hostname pattern detected: " . $original);
            }
        }
    }
    
    /**
     * Validate port number
     */
    private static function validatePort($port) {
        self::validateInteger($port);
        
        if ($port < 1 || $port > 65535) {
            throw new InvalidArgumentException("Port must be between 1 and 65535");
        }
    }
    
    /**
     * Validate URL for remote file fetching
     */
    public static function validateUrl($url) {
        if (!is_string($url)) {
            throw new InvalidArgumentException("URL must be a string");
        }
        
        // Basic URL validation
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            throw new InvalidArgumentException("Invalid URL format");
        }
        
        $parsedUrl = parse_url($url);
        
        // Only allow HTTP/HTTPS
        if (!in_array($parsedUrl['scheme'], ['http', 'https'], true)) {
            throw new InvalidArgumentException("Only HTTP and HTTPS URLs are allowed");
        }
        
        // Validate hostname
        if (isset($parsedUrl['host'])) {
            self::validateHostname($parsedUrl['host']);
        }
        
        return filter_var($url, FILTER_SANITIZE_URL);
    }
    
    /**
     * Validate and sanitize application settings
     */
    public static function validateApplicationSettings($settings) {
        if (!is_array($settings)) {
            throw new InvalidArgumentException("Settings must be an array");
        }
        
        $validatedSettings = [];
        
        foreach ($settings as $key => $value) {
            switch ($key) {
                case 'language':
                    $validatedSettings[$key] = self::validateLanguage($value);
                    break;
                case 'connectionRestrictions':
                    $validatedSettings[$key] = self::validateConnectionRestrictions($value);
                    break;
                case 'allowedClientAddresses':
                    $validatedSettings[$key] = self::validateAllowedAddresses($value);
                    break;
                default:
                    // For unknown settings, apply basic sanitization
                    if (is_string($value)) {
                        $validatedSettings[$key] = filter_var($value, FILTER_SANITIZE_STRING);
                    } else {
                        $validatedSettings[$key] = $value;
                    }
            }
        }
        
        return $validatedSettings;
    }
    
    /**
     * Validate language code
     */
    private static function validateLanguage($language) {
        if (!is_string($language)) {
            throw new InvalidArgumentException("Language must be a string");
        }
        
        // Basic language code validation (e.g., en_US, fr_FR)
        if (!preg_match('/^[a-z]{2}(_[A-Z]{2})?$/', $language)) {
            throw new InvalidArgumentException("Invalid language code format");
        }
        
        return $language;
    }
    
    /**
     * Validate connection restrictions
     */
    private static function validateConnectionRestrictions($restrictions) {
        if (!is_array($restrictions)) {
            throw new InvalidArgumentException("Connection restrictions must be an array");
        }
        
        foreach ($restrictions as $type => $config) {
            if (!in_array($type, ['ftp', 'sftp'], true)) {
                throw new InvalidArgumentException("Invalid connection type in restrictions");
            }
            
            if (is_array($config)) {
                foreach ($config as $key => $value) {
                    if ($key === 'host' && is_array($value)) {
                        foreach ($value as $host) {
                            self::validateHostname($host);
                        }
                    }
                }
            }
        }
        
        return $restrictions;
    }
    
    /**
     * Validate allowed client addresses (CIDR notation)
     */
    private static function validateAllowedAddresses($addresses) {
        if (!is_array($addresses)) {
            throw new InvalidArgumentException("Allowed addresses must be an array");
        }
        
        foreach ($addresses as $address) {
            if (!is_string($address)) {
                throw new InvalidArgumentException("Address must be a string");
            }
            
            // Validate CIDR notation
            if (strpos($address, '/') !== false) {
                list($ip, $mask) = explode('/', $address, 2);
                if (!filter_var($ip, FILTER_VALIDATE_IP)) {
                    throw new InvalidArgumentException("Invalid IP address in CIDR: " . $address);
                }
                if (!is_numeric($mask) || $mask < 0 || $mask > 32) {
                    throw new InvalidArgumentException("Invalid CIDR mask: " . $address);
                }
            } else {
                if (!filter_var($address, FILTER_VALIDATE_IP)) {
                    throw new InvalidArgumentException("Invalid IP address: " . $address);
                }
            }
        }
        
        return $addresses;
    }
    
    /**
     * Sanitize and validate session key
     */
    public static function validateSessionKey($sessionKey) {
        if (!is_string($sessionKey)) {
            throw new InvalidArgumentException("Session key must be a string");
        }
        
        // Session keys should only contain alphanumeric characters
        if (!preg_match('/^[a-zA-Z0-9]+$/', $sessionKey)) {
            throw new InvalidArgumentException("Invalid session key format");
        }
        
        if (strlen($sessionKey) < 8 || strlen($sessionKey) > 64) {
            throw new InvalidArgumentException("Session key length must be between 8 and 64 characters");
        }
        
        return $sessionKey;
    }
} 