<?php

    require_once(dirname(__FILE__) . '/../PathOperations.php');
    require_once(dirname(__FILE__) . '/../transfers/TransferOperationFactory.php');
    require_once(dirname(__FILE__) . "/../../lib/helpers.php");

    class ZipBuilder {
        private $connection;
        private $baseDirectory;
        private $connectionType;
        private $zipFile;
        private $localPathsCleanup;

        public function __construct($connection, $baseDirectory) {
            $this->connection = $connection;
            $this->baseDirectory = $baseDirectory;
            $this->connectionType = strtolower($connection->getProtocolName());
            $this->localPathsCleanup = array();
        }

        private function cleanupLocalPaths() {
            foreach ($this->localPathsCleanup as $localPath) {
                @unlink($localPath);
            }
        }

        public function buildZip($fileList) {
            $zipPath = monstaTempnam(getMonstaSharedTransferDirectory(), "monsta-download-zip");
            
            // Remove the empty temp file created by monstaTempnam to avoid deprecation warning
            if (file_exists($zipPath)) {
                unlink($zipPath);
            }

            $this->zipFile = new ZipArchive();
            $this->zipFile->open($zipPath, ZipArchive::CREATE);

            try {
                foreach ($fileList as $relativeFilePath) {
                    $this->addFileToZip($relativeFilePath);
                }
                $this->zipFile->close();
            } catch (Exception $e) {
                $this->cleanupLocalPaths();
                throw $e;
            }

            // this should be done in a finally to avoid repeated code but we need to support PHP < 5.5
            $this->cleanupLocalPaths();

            return $zipPath;
        }

        public function buildLocalZip($fileList, $zipPath) {
            $debugFile = dirname(__FILE__) . '/../../../logs/mftp_debug.log';
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: buildLocalZip called with zipPath: " . $zipPath . "\n", FILE_APPEND);
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: File list: " . json_encode($fileList) . "\n", FILE_APPEND);
            
            // Use the provided zipPath instead of creating a new temp file
            
            // Remove the empty temp file if it exists to avoid deprecation warning
            if (file_exists($zipPath)) {
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Removing existing file: " . $zipPath . "\n", FILE_APPEND);
                unlink($zipPath);
            }

            // Check if ZipArchive class is available
            if (!class_exists('ZipArchive')) {
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: ZipArchive class not available\n", FILE_APPEND);
                throw new Exception("ZipArchive class is not available. Please install the zip extension for PHP.");
            }
            
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Creating ZipArchive instance\n", FILE_APPEND);
            $this->zipFile = new ZipArchive();
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Opening zip file: " . $zipPath . "\n", FILE_APPEND);
            $result = $this->zipFile->open($zipPath, ZipArchive::CREATE);
            
            if ($result !== TRUE) {
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: ZipArchive::open failed with result: " . $result . "\n", FILE_APPEND);
                $errorMessages = [
                    ZipArchive::ER_OK => 'No error',
                    ZipArchive::ER_MULTIDISK => 'Multi-disk zip archives not supported',
                    ZipArchive::ER_RENAME => 'Renaming temporary file failed',
                    ZipArchive::ER_CLOSE => 'Closing zip archive failed',
                    ZipArchive::ER_SEEK => 'Seek error',
                    ZipArchive::ER_READ => 'Read error',
                    ZipArchive::ER_WRITE => 'Write error',
                    ZipArchive::ER_CRC => 'CRC error',
                    ZipArchive::ER_ZIPCLOSED => 'Containing zip archive was closed',
                    ZipArchive::ER_NOENT => 'No such file',
                    ZipArchive::ER_EXISTS => 'File already exists',
                    ZipArchive::ER_OPEN => 'Can\'t open file',
                    ZipArchive::ER_TMPOPEN => 'Failure to create temporary file',
                    ZipArchive::ER_ZLIB => 'Zlib error',
                    ZipArchive::ER_MEMORY => 'Memory allocation failure',
                    ZipArchive::ER_CHANGED => 'Entry has been changed',
                    ZipArchive::ER_COMPNOTSUPP => 'Compression method not supported',
                    ZipArchive::ER_EOF => 'Premature EOF',
                    ZipArchive::ER_INVAL => 'Invalid argument',
                    ZipArchive::ER_NOZIP => 'Not a zip archive',
                    ZipArchive::ER_INTERNAL => 'Internal error',
                    ZipArchive::ER_INCONS => 'Zip archive inconsistent',
                    ZipArchive::ER_REMOVE => 'Can\'t remove file',
                    ZipArchive::ER_DELETED => 'Entry has been deleted'
                ];
                
                $errorMessage = $errorMessages[$result] ?? "Unknown error (code: $result)";
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: ZipArchive error: " . $errorMessage . "\n", FILE_APPEND);
                throw new Exception("Failed to create zip archive at '$zipPath': $errorMessage");
            }
            
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: ZipArchive opened successfully\n", FILE_APPEND);

            try {
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Starting to add " . count($fileList) . " files to zip\n", FILE_APPEND);
                foreach ($fileList as $relativeFilePath) {
                    // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Adding file to zip: " . $relativeFilePath . "\n", FILE_APPEND);
                    $this->addFileToZip($relativeFilePath);
                    // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Successfully added file: " . $relativeFilePath . "\n", FILE_APPEND);
                }
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Closing zip file\n", FILE_APPEND);
                $this->zipFile->close();
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Zip file closed successfully\n", FILE_APPEND);
            } catch (Exception $e) {
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Exception during zip processing: " . $e->getMessage() . "\n", FILE_APPEND);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Exception trace: " . $e->getTraceAsString() . "\n", FILE_APPEND);
                $this->cleanupLocalPaths();
                throw $e;
            }

            // this should be done in a finally to avoid repeated code but we need to support PHP < 5.5
            $this->cleanupLocalPaths();

            return $zipPath;
        }

        private function addFileToZip($relativeFilePath) {
            $debugFile = dirname(__FILE__) . '/../../../logs/mftp_debug.log';
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: addFileToZip called for: " . $relativeFilePath . "\n", FILE_APPEND);
            try {
                $fileName = monstaBasename($relativeFilePath);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: File name: " . $fileName . "\n", FILE_APPEND);
                $fileOutputPath = monstaTempnam(getMonstaSharedTransferDirectory(), $fileName);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Temp file path: " . $fileOutputPath . "\n", FILE_APPEND);
                
                $remotePath = PathOperations::join($this->baseDirectory, $relativeFilePath);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Remote path: " . $remotePath . "\n", FILE_APPEND);
                
                $rawConfiguration = array(
                    'localPath' => $fileOutputPath,
                    'remotePath' => $remotePath
                );

                $this->localPathsCleanup[] = $fileOutputPath;

                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Creating transfer operation\n", FILE_APPEND);
                $transferOperation = TransferOperationFactory::getTransferOperation($this->connectionType,
                    $rawConfiguration);

                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Downloading file from remote\n", FILE_APPEND);
                $this->connection->downloadFile($transferOperation);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: File download completed\n", FILE_APPEND);

                // Check if the downloaded file exists and is readable
                if (!file_exists($fileOutputPath)) {
                    // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Downloaded file does not exist: " . $fileOutputPath . "\n", FILE_APPEND);
                    throw new Exception("Downloaded file does not exist: $fileOutputPath");
                }
                
                if (!is_readable($fileOutputPath)) {
                    // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Downloaded file is not readable: " . $fileOutputPath . "\n", FILE_APPEND);
                    throw new Exception("Downloaded file is not readable: $fileOutputPath");
                }
                
                $fileSize = filesize($fileOutputPath);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Downloaded file size: " . $fileSize . " bytes\n", FILE_APPEND);

                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Adding file to zip archive\n", FILE_APPEND);
                $result = $this->zipFile->addFile($fileOutputPath, $relativeFilePath);
                if (!$result) {
                    // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Failed to add file to zip archive\n", FILE_APPEND);
                    throw new Exception("Failed to add file '$relativeFilePath' to zip archive");
                }
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: File added to zip successfully\n", FILE_APPEND);
            } catch (Exception $e) {
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Exception in addFileToZip: " . $e->getMessage() . "\n", FILE_APPEND);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Exception trace: " . $e->getTraceAsString() . "\n", FILE_APPEND);
                throw new Exception("Error adding file '$relativeFilePath' to zip: " . $e->getMessage());
            }
        }
    }