<?php

    require_once(dirname(__FILE__) . "/../PathOperations.php");

    class RecursiveFileFinder {
        private $connection;
        private $baseDirectory;
        private $itemList;

        public function __construct($connection, $baseDirectory) {
            $this->connection = $connection;
            $this->baseDirectory = PathOperations::ensureTrailingSlash($baseDirectory);
            $this->itemList = array();
        }

        public function findFilesInPaths($items = null) {
            return $this->findItems($items, false);
        }

        public function findFilesAndDirectoriesInPaths($items = null) {
            return $this->findItems($items, true);
        }

        private function findItems($items, $includeDirectoriesAndData = false) {
            if($items == null)
                $items = array("");

            $debugFile = dirname(__FILE__) . '/../../../logs/mftp_debug.log';
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: RecursiveFileFinder: Processing " . count($items) . " items\n", FILE_APPEND);

            foreach ($items as $index => $itemPath) {
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: RecursiveFileFinder: Processing item " . ($index + 1) . "/" . count($items) . ": " . $itemPath . "\n", FILE_APPEND);
                $fullPath = PathOperations::join($this->baseDirectory, $itemPath);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: RecursiveFileFinder: Full path: " . $fullPath . "\n", FILE_APPEND);
                $this->handleFileOrDirectory($fullPath, $includeDirectoriesAndData);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: RecursiveFileFinder: Completed item " . ($index + 1) . "\n", FILE_APPEND);
            }

            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: RecursiveFileFinder: Found " . count($this->itemList) . " total items\n", FILE_APPEND);
            return $this->itemList;
        }

        private function handleFileOrDirectory($itemPath, $includeDirectoriesAndData) {
            $debugFile = dirname(__FILE__) . '/../../../logs/mftp_debug.log';
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: RecursiveFileFinder: handleFileOrDirectory called for: " . $itemPath . "\n", FILE_APPEND);
            
            if(!$includeDirectoriesAndData) {
                // First, try to determine if it's likely a file based on the path
                $fileName = basename($itemPath);
                $hasExtension = strpos($fileName, '.') !== false && strpos($fileName, '.') > 0;
                
                // If it has an extension and doesn't end with a slash, it's likely a file
                if ($hasExtension && substr($itemPath, -1) !== '/') {
                    // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: RecursiveFileFinder: Detected file by extension: " . $itemPath . "\n", FILE_APPEND);
                    $this->handleFile($itemPath);
                    return;
                }
                
                // For paths without extensions or ending with slash, check if it's a directory
                try {
                    $directoryList = $this->connection->listDirectory($itemPath, true);
                    // If listDirectory succeeds, it's a directory
                    // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: RecursiveFileFinder: Detected directory: " . $itemPath . "\n", FILE_APPEND);
                    $this->traverseDirectory($itemPath, $includeDirectoriesAndData, $directoryList);
                } catch (Exception $e) {
                    // If listDirectory fails, it's likely a file
                    // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: RecursiveFileFinder: Detected file: " . $itemPath . "\n", FILE_APPEND);
                    $this->handleFile($itemPath);
                }
            } else {
                // Original logic for when we need directory data
                try {
                    $directoryList = $this->connection->listDirectory($itemPath, true);
                    $this->traverseDirectory($itemPath, $includeDirectoriesAndData, $directoryList);
                } catch (Exception $e) {
                    // If it's not a directory, treat as file
                    $this->handleFile($itemPath);
                }
            }
        }

        private function handleFile($filePath) {
            $this->itemList[] = substr($filePath, strlen($this->baseDirectory));
        }

        private function handleItem($itemPath, $item) {
            $this->itemList[] = array(substr($itemPath, strlen($this->baseDirectory)), $item);
        }

        private function traverseDirectory($dirPath, $includeDirectoriesAndData, $directoryList = null, $depth = 0) {
            if($depth >= 50)
                return; // we've gone too deep, maybe recursive symlink. bail out

            if($directoryList == null)
                $directoryList = $this->connection->listDirectory($dirPath, true);

            foreach ($directoryList as $item) {
                $itemPath = PathOperations::join($dirPath, $item->getName());

                if($includeDirectoriesAndData)
                    $this->handleItem($itemPath, $item);
                else if(!$item->isDirectory())
                    $this->handleFile($itemPath);

                if ($item->isDirectory())
                    $this->traverseDirectory($itemPath, $includeDirectoriesAndData, null, $depth + 1);
            }
        }
    }