<?php

    require_once(dirname(__FILE__) . "/../../lib/helpers.php");
    require_once(dirname(__FILE__) . "/SFTPConnectionBase.php");
    require_once(dirname(__FILE__) . "/PHPSeclibListParser.php");
    require_once(dirname(__FILE__) . "/../../lib/logging.php");
    
    // Custom autoloader for PHPSecLib 3.x
    function loadPhpSecLib3($className) {
        if (strpos($className, 'phpseclib3\\') === 0) {
            $relativePath = str_replace('phpseclib3\\', '', $className);
            $relativePath = str_replace('\\', '/', $relativePath);
            $filePath = dirname(__FILE__) . '/../../vendor/phpseclib/phpseclib/' . $relativePath . '.php';
            
            if (file_exists($filePath)) {
                require_once $filePath;
                return true;
            }
        }
        return false;
    }
    
    // Register the autoloader
    spl_autoload_register('loadPhpSecLib3');
    
    // Include bootstrap for any initialization
    require_once(dirname(__FILE__) . '/../../vendor/phpseclib/phpseclib/phpseclib/bootstrap.php');

    use phpseclib3\Net\SFTP;
    use phpseclib3\Crypt\RSA;
    use phpseclib3\System\SSH\Agent;

    class PHPSeclibConnection extends SFTPConnectionBase {
        protected function handleConnect() {
            $conn = new SFTP(escapeIpAddress($this->configuration->getHost()), $this->configuration->getPort());
            mftpLog(LOG_DEBUG, "New SFTPSecLib connection created to '{$this->configuration->getHost()}:{$this->configuration->getPort()}'");
            return $conn;
        }

        protected function handleDisconnect() {
            $this->connection->disconnect();
        }

        protected function postAuthentication() {
            // TODO: Implement postAuthentication() method.
        }

        protected function handleListDirectory($path, $showHidden) {
            $stat = $this->statRemoteFile($path);

            if ($stat === false || $stat['type'] != NET_SFTP_TYPE_DIRECTORY) {
                if ($stat === false)
                    $errorMessage = "$path does not exist";
                else
                    $errorMessage = "$path is not a directory";

                mftpLog(LOG_DEBUG, "SFTPSecLib Unable to list directory: $errorMessage");
                $error = array('message' => $errorMessage);
                $this->handleOperationError('LIST_DIRECTORY_OPERATION', $path, $error);
                return false;
            }

            $rawList = $this->connection->rawlist($path);
            
            // Check if rawlist failed (returns false on connection/auth failures)
            if ($rawList === false) {
                $errorMessage = "Failed to list directory contents for $path";
                mftpLog(LOG_DEBUG, "SFTPSecLib rawlist failed for path: $path");
                $error = array('message' => $errorMessage);
                $this->handleOperationError('LIST_DIRECTORY_OPERATION', $path, $error);
                return false;
            }
            
            $dirList = new PHPSeclibListParser($rawList, $showHidden);

            mftpLog(LOG_DEBUG, "SFTPSecLib listed directory: $path. Returned " . count($rawList) . " results.");

            return $dirList;
        }

        protected function handleDownloadFile($transferOperation) {
            $transferSuccess = $this->connection->get($transferOperation->getRemotePath(),
                $transferOperation->getLocalPath());

            if ($transferSuccess) {
                mftpLog(LOG_DEBUG, "SFTPSecLib got '{$transferOperation->getRemotePath()}' to '{$transferOperation->getLocalPath()}'");
            } else {
                // Get SFTP error, ensuring we have a valid error message
                $sftpError = $this->connection->getLastSFTPError();
                $errorMessage = (!empty($sftpError) && is_string($sftpError)) 
                    ? $sftpError 
                    : 'SFTP download failed - connection may have been closed or file may be inaccessible';
                
                $this->setLastError($errorMessage, $transferOperation->getRemotePath());
                mftpLog(LOG_WARNING, "SFTPSecLib failed to get '{$transferOperation->getRemotePath()}' to '{$transferOperation->getLocalPath()}': $errorMessage");
            }

            return $transferSuccess;
        }

        protected function handleUploadFile($transferOperation) {
            $transferSuccess = $this->connection->put($transferOperation->getRemotePath(),
                $transferOperation->getLocalPath(), SFTP::SOURCE_LOCAL_FILE);

            if ($transferSuccess)
                mftpLog(LOG_DEBUG, "SFTPSecLib put '{$transferOperation->getRemotePath()}' to '{$transferOperation->getLocalPath()}'");
            else {
                $this->setLastError($this->connection->getLastSFTPError(), $transferOperation->getRemotePath());
                mftpLog(LOG_WARNING, "SFTPSecLib failed to put '{$transferOperation->getRemotePath()}' to '{$transferOperation->getLocalPath()}': {$this->lastError['message']}");
            }

            if (method_exists($transferOperation, 'getCreateMode')) {
                if ( $transferSuccess && !is_null($transferOperation->getCreateMode()) ) {
                    $this->changePermissions($transferOperation->getCreateMode(), $transferOperation->getRemotePath());
                }
            }

            return $transferSuccess;
        }

        protected function handleDeleteFile($remotePath) {
            $deleteSuccess = $this->connection->delete($remotePath);

            if ($deleteSuccess)
                mftpLog(LOG_DEBUG, "SFTPSecLib deleted file '$remotePath'");
            else {
                $this->setLastError($this->connection->getLastSFTPError(), $remotePath);
                mftpLog(LOG_WARNING, "SFTPSecLib failed to delete file '$remotePath': {$this->lastError['message']}");
            }

            return $deleteSuccess;
        }

        protected function handleMakeDirectory($remotePath) {
            $createSuccess = $this->connection->mkdir($remotePath);

            if ($createSuccess)
                mftpLog(LOG_DEBUG, "SFTPSecLib created directory '$remotePath'");
            else {
                $this->setLastError($this->connection->getLastSFTPError(), $remotePath);
                mftpLog(LOG_WARNING, "SFTPSecLib failed to create directory '$remotePath': {$this->lastError['message']}");
            }

            return $createSuccess;
        }

        protected function handleDeleteDirectory($remotePath) {
            $deleteSuccess = $this->connection->rmdir($remotePath);

            if ($deleteSuccess)
                mftpLog(LOG_DEBUG, "SFTPSecLib deleted directory '$remotePath'");
            else {
                $this->setLastError($this->connection->getLastSFTPError(), $remotePath);
                mftpLog(LOG_WARNING, "SFTPSecLib failed to directory directory '$remotePath': {$this->lastError['message']}");
            }

            return $deleteSuccess;
        }

        protected function handleRename($source, $destination) {
            $renameSuccess = $this->connection->rename($source, $destination);

            if ($renameSuccess)
                mftpLog(LOG_DEBUG, "SFTPSecLib renamed '$source' to '$destination'");
            else {
                $this->setLastError($this->connection->getLastSFTPError(), $source);
                mftpLog(LOG_WARNING, "SFTPSecLib failed to rename ''$source' to '$destination': {$this->lastError['message']}");
            }

            return $renameSuccess;
        }

        protected function handleChangePermissions($mode, $remotePath) {
            $operationSuccess = $this->connection->chmod($mode, $remotePath);

            if ($operationSuccess)
                mftpLog(LOG_DEBUG, sprintf("SFTPSecLib changed perms of '%s' to '%o'", $remotePath, $mode));
            else {
                $this->setLastError($this->connection->getLastSFTPError(), $remotePath);
                mftpLog(LOG_DEBUG, sprintf("SFTPSecLib failed to change perms of '%s' to '%o': ", $remotePath, $mode, $this->lastError['message']));
            }

            return $operationSuccess;
        }

        protected function statRemoteFile($remotePath) {
            return $this->connection->stat($remotePath);
        }

        protected function authenticateByPassword() {
            $authSuccess = $this->connection->login($this->configuration->getRemoteUsername(),
                $this->configuration->getPassword());

            if ($authSuccess)
                mftpLog(LOG_DEBUG, "SFTPSecLib password login success '{$this->configuration->getRemoteUsername()}@{$this->configuration->getHost()}'");
            else
                mftpLog(LOG_WARNING, "SFTPSecLib password authentication failed for '{$this->configuration->getRemoteUsername()}@{$this->configuration->getHost()}'");

            return $authSuccess;
        }

        protected function authenticateByPublicKey() {
            $key = RSA::loadFormat('PKCS1', file_get_contents($this->configuration->getPrivateKeyFilePath()));

            if (!is_null($this->configuration->getPassword()))
                $key = $key->withPassword($this->configuration->getPassword());

            $authSuccess = $this->connection->login($this->configuration->getRemoteUsername(), $key);

            if ($authSuccess)
                mftpLog(LOG_DEBUG, "SFTPSecLib key login success '{$this->configuration->getRemoteUsername()}@{$this->configuration->getHost()}'");
            else
                mftpLog(LOG_WARNING, "SFTPSecLib key authentication failed for '{$this->configuration->getRemoteUsername()}@{$this->configuration->getHost()}'");

            return $authSuccess;
        }

        protected function authenticateByAgent() {
            $agent = @new Agent();
            $authSuccess = $this->connection->login($this->configuration->getRemoteUsername(), $agent);

            if ($authSuccess)
                mftpLog(LOG_DEBUG, "SFTPSecLib agent login success '{$this->configuration->getRemoteUsername()}@{$this->configuration->getHost()}'");
            else
                mftpLog(LOG_WARNING, "SFTPSecLib agent authentication failed for '{$this->configuration->getRemoteUsername()}@{$this->configuration->getHost()}'");

            return $authSuccess;
        }

        protected function handleGetCurrentDirectory() {
            return $this->connection->realpath('.');
        }
    }