<?php

    require_once(dirname(__FILE__) . "/../../lib/helpers.php");
    require_once(dirname(__FILE__) . "/../../lib/logging.php");
    require_once(dirname(__FILE__) . '/mftp_functions.php');
    require_once(dirname(__FILE__) . '/FTPConnectionBase.php');

    class MFTPConnection extends FTPConnectionBase {
        protected function handleConnect() {
            $connectionOrFalse = mftp_connect($this->configuration->getHost(), $this->configuration->getPort());

            if($connectionOrFalse === false)
                mftpLog(LOG_WARNING, "MFTP failed to connect to '{$this->configuration->getHost()}:{$this->configuration->getPort()}'");
            else
                mftpLog(LOG_DEBUG, "MFTP connected to '{$this->configuration->getHost()}:{$this->configuration->getPort()}'");

            return $connectionOrFalse;
        }

        protected function configureUTF8() {
            $features = $this->getServerFeatures();
            if (array_search("UTF8", $features) !== false) {
                mftp_utf8_on($this->connection);
                // this may or may not work, but if it doesn't there's nothing we can do so just carry on
                mftpLog(LOG_DEBUG, "MFTP enabled UTF8");
            }
        }

        protected function handleChangeDirectory($newDirectory) {
            try {
                mftp_chdir($this->connection, $newDirectory);

                mftpLog(LOG_DEBUG, "MFTP changed directory to $newDirectory");

                return true;
            } catch (MFTPRemoteFileException $remoteFileException) {
                $this->setLastError($remoteFileException->getMessage(), $newDirectory);
                mftpLog(LOG_WARNING, "MFTP failed to change directory to '$newDirectory': {$remoteFileException->getMessage()}");
                return false;
            }
        }

        protected function handleGetCurrentDirectory() {
            $path = mftp_pwd($this->connection);

            mftpLog(LOG_DEBUG, "MFTP pwd is: '$path'");

            return $path;
        }

        function handleRawDirectoryList($listArgs) {
            try {
                $rawList = mftp_rawlist($this->connection, $listArgs);

                mftpLog(LOG_DEBUG, "MFTP listed directory: $listArgs. Returned " . count($rawList) . " results.");

                return $rawList;
            } catch (MFTPNoSuchRemoteFileException $remoteFileMissingException) {
                $this->setLastError($remoteFileMissingException->getMessage(), $listArgs);

                return false;
            }
        }

        protected function rawGetSysType() {
            try {
                $sysType = mftp_get_systype($this->connection);
            } catch (MFTPException $sysTypeException) {
                mftpLog(LOG_WARNING, "MFTP failed to get sysType: " . $sysTypeException->getMessage());

                return false;
            }

            mftpLog(LOG_DEBUG, "MFTP got sysType '$sysType'");

            return $sysType;
        }

        protected function handleDisconnect() {
            return mftp_disconnect($this->connection);
        }

        protected function handleAuthentication() {
            try {
                if($this->configuration->isSSLMode())
                    mftp_enable_ssl($this->connection);

                mftp_login($this->connection, $this->configuration->getUsername(),
                    $this->configuration->getPassword());

                mftpLog(LOG_INFO, "MFTP login success '{$this->configuration->getUsername()}@{$this->configuration->getHost()}'");

                return true;
            } catch(MFTPAuthenticationRequiresTlsException $tlsException) {
              throw new LocalizableException("The server you are connecting to requires TLS/SSL to be enabled.",
                  LocalizableExceptionDefinition::$TLS_REQUIRED_ERROR);
            } catch (MFTPAuthenticationException $e) {
                mftpLog(LOG_WARNING, "MFTP authentication failed for '{$this->configuration->getUsername()}': {$e->getMessage()}");
                return false;
            }
        }

        protected function handlePassiveModeSet($passiveMode) {
            mftp_pasv($this->connection, $passiveMode);

            mftpLog(LOG_DEBUG, "MFTP passive mode set to '$passiveMode'");

            return true;
        }

        protected function handleDownloadFile($transferOperation) {
            try {
                mftp_get($this->connection, $transferOperation->getLocalPath(),
                    $transferOperation->getRemotePath(), $transferOperation->getTransferMode());

                mftpLog(LOG_DEBUG, "MFTP got '{$transferOperation->getRemotePath()}' to '{$transferOperation->getLocalPath()}'");

                return true;
            } catch (MFTPException $exception) {
                $this->setLastError($exception->getMessage(), $transferOperation->getRemotePath());
                mftpLog(LOG_WARNING, "MFTP failed to get '{$transferOperation->getRemotePath()}' to '{$transferOperation->getLocalPath()}': {$exception->getMessage()}");
                return false;
            }
        }

        protected function handleUploadFile($transferOperation) {
            $maxRetries = 3;
            $retryDelay = 1; // seconds
            
            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "MFTPConnection: handleUploadFile called - Local: '{$transferOperation->getLocalPath()}', Remote: '{$transferOperation->getRemotePath()}'");
                mftpLog(LOG_DEBUG, "MFTPConnection: Local file exists: " . (file_exists($transferOperation->getLocalPath()) ? 'YES' : 'NO'));
                if (file_exists($transferOperation->getLocalPath())) {
                    mftpLog(LOG_DEBUG, "MFTPConnection: Local file size: " . filesize($transferOperation->getLocalPath()) . " bytes");
                }
            }
            
            for ($attempt = 1; $attempt <= $maxRetries; $attempt++) {
                try {
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_DEBUG, "MFTPConnection: Attempting mftp_put (attempt $attempt/$maxRetries)");
                    }
                    
                    mftp_put($this->connection, $transferOperation->getRemotePath(),
                        $transferOperation->getLocalPath(), $transferOperation->getTransferMode(),
                        MFTP_UPLOAD_PROGRESS_CALLBACK_TIME_SECONDS,
                        function ($totalBytes){
                            outputStreamKeepAlive();
                        });

                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_DEBUG, "MFTPConnection: mftp_put completed successfully (attempt $attempt)");
                    }

                    return true;
                    
                } catch (MFTPFileException $fileException) {
                    $this->setLastError($fileException->getMessage(), $transferOperation->getRemotePath());

                    mftpLog(LOG_WARNING, "MFTP failed to put '{$transferOperation->getLocalPath()}' to '{$transferOperation->getRemotePath()}': {$fileException->getMessage()}");

                    return false;
                    
                } catch (MFTPQuotaExceededException $quotaExceededException) {
                    // For concurrent uploads, quota errors might be false positives due to timing
                    // Retry with a shorter delay to allow other concurrent uploads to complete and quota to update
                    // Use shorter delays (0.5s, 1s) to minimize impact on upload speed
                    if ($attempt < $maxRetries) {
                        $retryDelaySeconds = 0.5 * $attempt; // 0.5s, 1s delays (reduced from 2s, 4s)
                        if (function_exists('mftpLog')) {
                            mftpLog(LOG_WARNING, sprintf(
                                "MFTP quota error on attempt %d/%d for '%s'. This may be a false positive due to concurrent uploads. Retrying in %.1f seconds...",
                                $attempt,
                                $maxRetries,
                                $transferOperation->getRemotePath(),
                                $retryDelaySeconds
                            ));
                        }
                        usleep((int)($retryDelaySeconds * 1000000)); // Convert to microseconds
                        continue; // Retry the upload
                    } else {
                        // Max retries reached - this is likely a real quota error
                        if (function_exists('mftpLog')) {
                            mftpLog(LOG_ERROR, sprintf(
                                "MFTP quota exceeded after %d attempts for '%s'. Server response: %s",
                                $maxRetries,
                                $transferOperation->getRemotePath(),
                                $quotaExceededException->getMessage()
                            ));
                        }
                        throw new LocalizableException(
                            sprintf("Could not upload the file '%s' as the account quota has been exceeded.", 
                                basename($transferOperation->getRemotePath())),
                            LocalizableExceptionDefinition::$QUOTA_EXCEEDED_MESSAGE
                        );
                    }
                } catch (Exception $e) {
                    $errorMessage = $e->getMessage();
                    
                    // Check if this is a retryable error
                    $isRetryable = (
                        strpos($errorMessage, 'Could not read line from socket') !== false ||
                        strpos($errorMessage, 'socket') !== false ||
                        strpos($errorMessage, 'timeout') !== false ||
                        strpos($errorMessage, 'network') !== false ||
                        strpos($errorMessage, 'connection') !== false ||
                        strpos($errorMessage, 'EOF') !== false ||
                        strpos($errorMessage, 'unexpected end of file') !== false
                    );
                    
                    if ($attempt < $maxRetries && $isRetryable) {
                        mftpLog(LOG_WARNING, "MFTP upload attempt $attempt failed with retryable error: $errorMessage. Retrying in $retryDelay seconds...");
                        sleep($retryDelay);
                        $retryDelay *= 2; // Exponential backoff
                        continue;
                    } else {
                        // Either max retries reached or non-retryable error
                        mftpLog(LOG_ERROR, "MFTP upload failed after $attempt attempts: $errorMessage");
                        throw $e;
                    }
                }
            }
        }

        protected function handleDeleteFile($remotePath) {
            try {
                mftp_delete($this->connection, $remotePath);

                mftpLog(LOG_DEBUG, "MFTP deleted $remotePath");

                return true;
            } catch (MFTPRemoteFileException $localFileException) {
                $this->setLastError($localFileException->getMessage(), $remotePath);
                mftpLog(LOG_WARNING, "MFTP failed to delete '$remotePath': {$localFileException->getMessage()}");
                return false;
            }
        }

        protected function handleMakeDirectory($remotePath) {
            try {
                mftp_mkdir($this->connection, $remotePath);

                mftpLog(LOG_DEBUG, "MFTP created directory '$remotePath'");
                return true;
            } catch (MFTPRemoteFileException $remoteException) {
                $this->setLastError($remoteException->getMessage(), $remotePath);
                mftpLog(LOG_WARNING, "MFTP failed to create directory '$remotePath': {$remoteException->getMessage()}");
                return false;
            }
        }

        protected function handleDeleteDirectory($remotePath) {
            try {
                mftp_rmdir($this->connection, $remotePath);

                mftpLog(LOG_DEBUG, "MFTP deleted directory '$remotePath'");

                return true;
            } catch (MFTPRemoteFileException $remoteException) {
                $this->setLastError($remoteException->getMessage(), $remotePath);
                mftpLog(LOG_WARNING, "MFTP failed to delete directory '$remotePath': {$remoteException->getMessage()}");
                return false;
            }
        }

        protected function handleRename($source, $destination) {
            try {
                mftp_rename($this->connection, $source, $destination);

                mftpLog(LOG_DEBUG, "MFTP renamed '$source' to '$destination'");

                return true;
            } catch (MFTPRemoteFileException $remoteException) {
                $this->setLastError($remoteException->getMessage(), $remoteException->getPath());

                mftpLog(LOG_WARNING, "MFTP failed to rename '$source' to '$destination': {$remoteException->getMessage()}");

                return false;
            }
        }

        protected function handleChangePermissions($mode, $remotePath) {
            try {
                mftp_chmod($this->connection, $mode, $remotePath);

                mftpLog(LOG_DEBUG, sprintf("MFTP chmod '%s' to '%o'", $remotePath, $mode));

                return true;
            } catch (MFTPRemoteFileException $remoteException) {
                $this->setLastError($remoteException->getMessage(), $remotePath);

                mftpLog(LOG_WARNING, sprintf("MFTP failed to chmod '%s' to '%o': %s", $remotePath, $mode,
                    $remoteException->getMessage()));

                return false;
            }
        }

        protected function getServerFeatures() {
            $cachedFeatures = $this->getCapabilitiesArrayValue("FEATURES");

            if(!is_null($cachedFeatures)) {
                if(is_null($this->connection->features))
                    $this->connection->features = $cachedFeatures;
                return $cachedFeatures;
            }
            if(!$this->isAuthenticated())
                return mftp_features($this->connection);
            else {
                mftpLog(LOG_INFO, "Attempting to get FEAT after authentication");
                return array(); // some FTP servers (ws_ftp) don't support getting FEAT after auth. default to empty array
            }
        }
    }
