<?php

    require_once(dirname(__FILE__) . '/FTPConnectionBase.php');
    require_once(dirname(__FILE__) . '/../PathOperations.php');
    require_once(dirname(__FILE__) . '/FTPListParser.php');
    require_once(dirname(__FILE__) . '/Exceptions.php');
    require_once(dirname(__FILE__) . "/../../lib/helpers.php");

    abstract class FTPTransferMode {
        public static function fromString($transferModeName) {
            switch ($transferModeName) {
                case "ASCII":
                    return FTP_ASCII;
                case "BINARY":
                    return FTP_BINARY;
                default:
                    throw new InvalidArgumentException("FTP Transfer mode must be ASCII or BINARY.");
            }
        }
    }

    class FTPConnection extends FTPConnectionBase {
        public function __construct($configuration) {
            parent::__construct($configuration);
            $this->sysType = null;
        }

        protected function handleConnect() {
            $host = $this->configuration->getHost();
            $port = $this->configuration->getPort();
            
            // Check if hostname can be resolved (if it's not already an IP address)
            $isIP = filter_var($host, FILTER_VALIDATE_IP) !== false;
            $dnsResolved = false;
            
            if (!$isIP) {
                // Try to resolve the hostname
                $ip = @gethostbyname($host);
                $dnsResolved = ($ip !== $host && $ip !== false);
                
                if (!$dnsResolved) {
                    // DNS resolution failed - try reverse lookup to see if it's a DNS issue
                    $debugFile = dirname(__FILE__) . '/../../../logs/mftp_debug.log';
                    @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] FTP DEBUG: DNS resolution failed for hostname: $host\n", FILE_APPEND);
                    @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] FTP DEBUG: Suggestion: Try using the IP address instead of the hostname\n", FILE_APPEND);
                }
            }
            
            // Suppress warnings to capture them ourselves
            $errorBefore = error_get_last();
            
            if ($this->configuration->isSSLMode()) {
                $connection = @ftp_ssl_connect($host, $port, 30); // 30 second timeout
            } else {
                $connection = @ftp_connect($host, $port, 30); // 30 second timeout
            }
            
            $errorAfter = error_get_last();
            
            // Log connection attempt details
            $debugFile = dirname(__FILE__) . '/../../../logs/mftp_debug.log';
            if ($connection === false) {
                $errorMsg = ($errorAfter !== $errorBefore && $errorAfter !== null) ? $errorAfter['message'] : 'Unknown error';
                @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] FTP DEBUG: Connection failed to $host:$port. Error: $errorMsg\n", FILE_APPEND);
                
                // If DNS resolution failed, add helpful message
                if (!$isIP && !$dnsResolved && strpos($errorMsg, 'getaddrinfo') !== false) {
                    @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] FTP DEBUG: DNS resolution error detected. The server cannot resolve '$host' to an IP address.\n", FILE_APPEND);
                    @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] FTP DEBUG: SOLUTION: Use the IP address instead of the hostname (e.g., use '123.45.67.89' instead of '$host')\n", FILE_APPEND);
                }
            } else {
                @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] FTP DEBUG: Connection successful to $host:$port\n", FILE_APPEND);
            }
            
            return $connection !== false ? $connection : false;
        }

        protected function handleDisconnect() {
            // Suppress SSL warnings when closing FTP connections - these are common and don't affect functionality
            $result = @ftp_close($this->connection);
            if ($result === false) {
                $errorMessage =  $this->getLastError();
                throw new FileSourceConnectionException(
                    sprintf("Failed to close %s connection: %s", $this->getProtocolName(), $errorMessage),
                    LocalizableExceptionDefinition::$FAILED_TO_CLOSE_CONNECTION_ERROR, array(
                    'protocol' => $this->getProtocolName(),
                    'message' => $errorMessage
                ));
            }
        }

        protected function handleAuthentication() {
            return ftp_login($this->connection, $this->configuration->getUsername(),
                $this->configuration->getPassword());
        }

        protected function configureUTF8() {
            // Attempt to enable UTF-8 support - this may or may not work depending on server
            $result = ftp_raw($this->connection, "OPTS UTF8 ON");
            // Intentionally ignore the result - if it doesn't work, there's nothing we can do
        }

        protected function handlePassiveModeSet($passiveMode) {
            return ftp_pasv($this->connection, $passiveMode);
        }

        function handleRawDirectoryList($listArgs) {
            return ftp_rawlist($this->connection, $listArgs);
        }

        protected function handleListDirectory($path, $showHidden) {
            // Validate connection before attempting directory listing
            if (!$this->isConnected()) {
                throw new FileSourceOperationException("FTP connection is not active",
                    LocalizableExceptionDefinition::$LIST_DIRECTORY_FAILED_ERROR,
                    array('path' => $path, 'error' => 'Connection lost'));
            }
            
            if (!$this->isAuthenticated()) {
                throw new FileSourceOperationException("FTP authentication failed",
                    LocalizableExceptionDefinition::$LIST_DIRECTORY_FAILED_ERROR,
                    array('path' => $path, 'error' => 'Not authenticated'));
            }
            
            if (!PathOperations::directoriesMatch($path, $this->getCurrentDirectory())) {
                $this->changeDirectory($path);
            }

            // Always send LIST without parameters for maximum compatibility
            // Some FTP servers (like ProFTPd) reject LIST with parameters
            // Hidden files are filtered client-side by FTPListParser
            $listArgs = null;

            // Add debugging for FTP connection issues
            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "FTP: Attempting to list directory '$path' (showHidden: " . ($showHidden ? 'true' : 'false') . ")");
                mftpLog(LOG_DEBUG, "FTP: Connection status - Connected: " . ($this->isConnected() ? 'true' : 'false') . ", Authenticated: " . ($this->isAuthenticated() ? 'true' : 'false'));
            }

            $dirList = ftp_rawlist($this->connection, $listArgs);

            if ($dirList === false) {
                // Get more detailed error information
                $lastError = error_get_last();
                $errorMsg = $lastError ? $lastError['message'] : 'Unknown error';
                
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_ERROR, "FTP: Failed to list directory '$path' - Error: $errorMsg");
                }
                
                throw new FileSourceOperationException(sprintf("Failed to list directory \"%s\" - %s", $path, $errorMsg),
                    LocalizableExceptionDefinition::$LIST_DIRECTORY_FAILED_ERROR,
                    array(
                        'path' => $path,
                        'error' => $errorMsg
                    ));
            }

            return new FTPListParser($dirList, $showHidden, $this->getSysType("unix"));
        }

        protected function rawGetSysType() {
            return ftp_systype($this->connection);
        }

        protected function handleGetCurrentDirectory() {
            return ftp_pwd($this->connection);
        }

        protected function handleChangeDirectory($newDirectory) {
            return ftp_chdir($this->connection, $newDirectory);
        }

        /**
         * @param $transferOperation FTPTransferOperation
         * @return bool
         */
        protected function handleDownloadFile($transferOperation) {
            return ftp_get($this->connection, $transferOperation->getLocalPath(), $transferOperation->getRemotePath(),
                $transferOperation->getTransferMode());
        }

        /**
         * @param $transferOperation FTPTransferOperation
         * @return bool
         */
        protected function handleUploadFile($transferOperation) {
            return ftp_put($this->connection, $transferOperation->getRemotePath(), $transferOperation->getLocalPath(),
                $transferOperation->getTransferMode());
        }

        protected function handleDeleteFile($remotePath) {
            return ftp_delete($this->connection, $remotePath);
        }

        protected function handleMakeDirectory($remotePath) {
            return ftp_mkdir($this->connection, $remotePath);
        }

        protected function handleDeleteDirectory($remotePath) {
            return ftp_rmdir($this->connection, $remotePath);
        }

        protected function handleRename($source, $destination) {
            return ftp_rename($this->connection, $source, $destination);
        }

        protected function handleChangePermissions($mode, $remotePath) {
            return ftp_chmod($this->connection, $mode, $remotePath);
        }
    }