<?php
    require_once(dirname(__FILE__) . "/../../lib/helpers.php");
    require_once(dirname(__FILE__) . "/../../lib/logging.php");
    require_once(dirname(__FILE__) . '/../Validation.php');
    require_once(dirname(__FILE__) . '/../PathOperations.php');
    require_once(dirname(__FILE__) . '/RecursiveFileFinder.php');
    require_once(dirname(__FILE__) . '/../../lib/ServerCapabilities.php');

    abstract class ConnectionBase {

        /**
         * @var boolean
         */
        protected $connected;
        /**
         * @var boolean
         */
        protected $authenticated;
        /**
         * @var FTPConfiguration
         */
        protected $configuration;
        /**
         * @var string;
         */
        protected $currentDirectory;
        /**
         * @var mftp_conn
         */
        protected $connection;
        /**
         * @var string
         */
        protected $protocolName = 'BASE_CLASS';

        /**
         * @var array null
         */
        protected $serverCapabilitiesArray = null;

        /**
         * @var string null
         */
        protected $lastError = null;

        /* subclasses should implement these abstract methods to do the actual stuff based on their protocol,
        then return bool for  success/failure, and this class will handle throwing the exception. optionally the handle*
        methods could throw their own exceptions for custom failures */

        abstract protected function handleConnect();

        abstract protected function handleDisconnect();

        /**
         * @return bool
         */
        abstract protected function handleAuthentication();

        abstract protected function postAuthentication();

        /**
         * @param $path string
         * @param $showHidden bool
         * @return ListParser
         */
        abstract protected function handleListDirectory($path, $showHidden);

        /**
         * @param $transferOperation TransferOperation
         * @return bool
         */
        abstract protected function handleDownloadFile($transferOperation);

        /**
         * @param $transferOperation TransferOperation
         * @return bool
         */
        abstract protected function handleUploadFile($transferOperation);

        /**
         * @param $remotePath string
         * @return bool
         */
        abstract protected function handleDeleteFile($remotePath);

        /**
         * @param $remotePath string
         * @return bool
         */
        abstract protected function handleMakeDirectory($remotePath);

        /**
         * @param $remotePath string
         * @return bool
         */
        abstract protected function handleDeleteDirectory($remotePath);

        /**
         * @param $source string
         * @param $destination string
         * @return bool
         */
        abstract protected function handleRename($source, $destination);

        /**
         * @param $mode int
         * @param $remotePath string
         * @return bool
         */
        abstract protected function handleChangePermissions($mode, $remotePath);

        /**
         * @param $source string
         * @param $destination string
         */
        abstract protected function handleCopy($source, $destination);

        abstract protected function handleGetFileInfo($remotePath);

        public function __construct($configuration) {
            $this->configuration = $configuration;
            $this->connected = false;
            $this->authenticated = false;
            $this->currentDirectory = null;
        }

        public function __destruct() {
            if ($this->isConnected())
                $this->disconnect();
        }

        public function connect() {
            $maxRetries = 3;
            $retryDelay = 1; // seconds
            $lastError = null;
            
            for ($attempt = 1; $attempt <= $maxRetries; $attempt++) {
                // Capture any PHP errors that might occur during connection
                $errorBefore = error_get_last();
                $this->connection = $this->handleConnect();
                $errorAfter = error_get_last();
                
                // Check if a new error occurred
                if ($errorAfter !== $errorBefore && $errorAfter !== null) {
                    $lastError = $errorAfter['message'];
                }
                
                if ($this->connection !== false) {
                    $this->connected = true;
                    $this->populateServerCapabilitiesArray();
                    $this->postConnection();
                    return; // Success, exit retry loop
                }
                
                // Connection failed, retry if attempts remaining
                if ($attempt < $maxRetries) {
                    $errorMsg = $lastError ? " (Error: $lastError)" : "";
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_WARNING, "Connection attempt $attempt failed for {$this->getProtocolName()} to {$this->configuration->getHost()}:{$this->configuration->getPort()}$errorMsg. Retrying in $retryDelay seconds...");
                    }
                    sleep($retryDelay);
                    $retryDelay *= 2; // Exponential backoff
                }
            }
            
            // All retries failed - include last error if available
            $host = $this->configuration->getHost();
            $isIP = filter_var($host, FILTER_VALIDATE_IP) !== false;
            $errorDetails = $lastError ? " Last error: $lastError." : "";
            
            // If DNS resolution failed, add helpful suggestion
            if (!$isIP && $lastError && strpos($lastError, 'getaddrinfo') !== false) {
                $errorDetails = " DNS resolution failed - the server cannot resolve '$host' to an IP address. Try using the IP address instead of the hostname.";
            }
            
            throw new FileSourceConnectionException(sprintf("%s connection to %s:%d failed after %d attempts.%s",
                $this->getProtocolName(), escapeIpAddress($host),
                $this->configuration->getPort(), $maxRetries, $errorDetails), LocalizableExceptionDefinition::$CONNECTION_FAILURE_ERROR, array(
                'protocol' => $this->getProtocolName(),
                'host' => escapeIpAddress($host),
                'port' => $this->configuration->getPort(),
                'last_error' => $lastError,
                'dns_hint' => (!$isIP && $lastError && strpos($lastError, 'getaddrinfo') !== false) ? "Try using the IP address instead of the hostname" : null
            ));
        }

        protected function postConnection() {
            // overridden in subclasses
        }

        public function disconnect() {
            if (!$this->isConnected())
                throw new FileSourceConnectionException("Can't disconnect a non-connected connection.",
                    LocalizableExceptionDefinition::$UNCONNECTED_DISCONNECT_ERROR);

            $this->handleDisconnect();

            $this->connected = false;
            $this->authenticated = false;
            $this->connection = null;
        }

        public function isConnected() {
            return $this->connected;
        }

        public function isAuthenticated() {
            return $this->authenticated;
        }

        public function getCurrentDirectory() {
            if (is_null($this->currentDirectory) && $this->isConnected() && $this->isAuthenticated())
                $this->syncCurrentDirectory();

            return $this->currentDirectory;
        }

        /**
         * @return string
         */
        public function getProtocolName() {
            return $this->protocolName;
        }

        protected function getLastError() {
            if (!is_null($this->lastError)) {
                $lastError = $this->lastError;
                $this->lastError = null;
                // Ensure error structure is valid
                if (!is_array($lastError) || !isset($lastError['message'])) {
                    return array('message' => 'Unknown error occurred');
                }
                return $lastError;
            }

            $phpError = error_get_last();
            // Ensure we return a valid error structure
            if ($phpError && is_array($phpError) && isset($phpError['message'])) {
                return array('message' => $phpError['message']);
            }
            
            // Fallback if no error is available
            return array('message' => 'Unknown error occurred');
        }

        protected function setLastError($message, $path = null) {
            $this->lastError = array(
                "message" => $message,
            );

            if (!is_null($path))
                $this->lastError["path"] = $path;
        }

        protected function handleOperationError($operationName, $path, $error, $secondaryPath = null) {
            $errorPreface = sprintf("Error during %s %s", $this->getProtocolName(), $operationName);

            if ($secondaryPath != null)
                $formattedPath = sprintf('"%s" / "%s"', $path, $secondaryPath);
            else
                $formattedPath = sprintf('"%s"', $path);

            $localizableContext = array(
                'protocol' => $this->getProtocolName(),
                'operation' => $operationName,
                'path' => $formattedPath
            );

            // Ensure error message is never empty
            $errorMessage = isset($error['message']) && !empty($error['message']) 
                ? $error['message'] 
                : 'Unknown error occurred';
            
            // Log the error for debugging
            if (function_exists('mftpLog')) {
                mftpLog(LOG_WARNING, sprintf(
                    "%s %s failed for path '%s': %s",
                    $this->getProtocolName(),
                    $operationName,
                    $formattedPath,
                    $errorMessage
                ));
            }

            if (strpos($errorMessage, "No such file or directory") !== FALSE
                || strpos($errorMessage, "file doesn't exist") !== FALSE
                || strpos($errorMessage, "does not exist") !== FALSE
                || strpos($errorMessage, "stat failed for") !== FALSE
                || strpos($errorMessage, "No such directory") !== FALSE
                || strpos($errorMessage, "No such file") !== FALSE
            )
                // latter is generated during rename, former for all others
                throw new FileSourceFileDoesNotExistException(sprintf("%s, file not found: %s", $errorPreface,
                    $formattedPath), LocalizableExceptionDefinition::$FILE_DOES_NOT_EXIST_ERROR, $localizableContext);
            else if (strpos($errorMessage, "Permission denied") !== FALSE
                || strpos($errorMessage, "failed to open stream: operation failed") !== FALSE)
                throw new FileSourceFilePermissionException(sprintf("%s, permission denied at: %s", $errorPreface,
                    $formattedPath), LocalizableExceptionDefinition::$FILE_PERMISSION_ERROR, $localizableContext);
            else if (strpos($errorMessage, "File exists") !== FALSE)
                throw new FileSourceFileExistsException(sprintf("%s, file exists at: %s", $errorPreface,
                    $formattedPath), LocalizableExceptionDefinition::$FILE_EXISTS_ERROR, $localizableContext);
            else {
                $localizableContext['message'] = $errorMessage;
                throw new FileSourceOperationException(
                    sprintf("%s, at %s: %s", $errorPreface, $formattedPath, $errorMessage),
                    LocalizableExceptionDefinition::$GENERAL_FILE_SOURCE_ERROR, $localizableContext);
            }
        }

        protected function ensureConnectedAndAuthenticated($operationName) {
            $errorContext = array('operation' => $operationName, 'protocol' => $this->getProtocolName());

            if (!$this->isConnected())
                throw new FileSourceConnectionException(sprintf("Can't %s file before %s is connected.",
                    $operationName, $this->getProtocolName()),
                    LocalizableExceptionDefinition::$OPERATION_BEFORE_CONNECTION_ERROR,
                    $errorContext);

            if (!$this->isAuthenticated())
                throw new FileSourceAuthenticationException(
                    sprintf("Can't %s file before authentication.", $operationName),
                    LocalizableExceptionDefinition::$OPERATION_BEFORE_AUTHENTICATION_ERROR,
                    $errorContext);
        }

        public function authenticate() {
            if (!$this->isConnected())
                throw new FileSourceConnectionException("Attempting to authenticate before connection.",
                    LocalizableExceptionDefinition::$AUTHENTICATION_BEFORE_CONNECTION_ERROR);

            $login_success = $this->handleAuthentication();

            if (!$login_success)
                throw new FileSourceAuthenticationException(sprintf("%s authentication failed.",
                    $this->getProtocolName()),
                    LocalizableExceptionDefinition::$AUTHENTICATION_FAILED_ERROR,
                    array('protocol' => $this->getProtocolName()));

            $this->authenticated = true;
            $this->postAuthentication();
        }

        /**
         * @param $path
         * @param bool $showHidden
         * @return array
         * @throws FileSourceAuthenticationException
         * @throws FileSourceConnectionException
         */
        public function listDirectory($path, $showHidden = null) {
            $this->ensureConnectedAndAuthenticated('LIST_DIRECTORY_OPERATION');
            return $this->handleListDirectory($path, is_null($showHidden) ? false : $showHidden);
        }

        private function getPathsFromLastError($lastError, $defaultPrimaryPath, $defaultSecondaryPath = null) {
            if (!is_null($lastError) && array_key_exists("path", $lastError)) {
                $primaryPath = $lastError["path"];
                $secondaryPath = null;
            } else {
                $primaryPath = $defaultPrimaryPath;
                $secondaryPath = $defaultSecondaryPath;
            }

            return array($primaryPath, $secondaryPath);
        }

        private function handleMultiTransferError($operationCode, $transferOperation) {
            $lastError = $this->getLastError();

            $paths = $this->getPathsFromLastError($lastError, $transferOperation->getLocalPath(),
                $transferOperation->getRemotePath());

            $this->handleOperationError($operationCode, $paths[0], $lastError, $paths[1]);
        }

        /**
         * @param $transferOperation TransferOperation
         * @throws FileSourceAuthenticationException
         * @throws FileSourceConnectionException
         * @throws FileSourceFileDoesNotExistException
         * @throws FileSourceFileExistsException
         * @throws FileSourceFilePermissionException
         * @throws FileSourceOperationException
         */
        public function downloadFile($transferOperation) {
            $this->ensureConnectedAndAuthenticated('DOWNLOAD_OPERATION');

            if (!$this->handleDownloadFile($transferOperation))
                $this->handleMultiTransferError('DOWNLOAD_OPERATION', $transferOperation);
        }

        /**
         * @param $transferOperation TransferOperation
         * @param $preserveRemotePermissions bool
         * @throws FileSourceAuthenticationException
         * @throws FileSourceConnectionException
         * @throws FileSourceFileDoesNotExistException
         * @throws FileSourceFileExistsException
         * @throws FileSourceFilePermissionException
         * @throws FileSourceOperationException
         */
        public function uploadFile($transferOperation, $preserveRemotePermissions = false) {
            $this->ensureConnectedAndAuthenticated('UPLOAD_OPERATION');

            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "uploadFile: Starting upload - Local: " . $transferOperation->getLocalPath() . ", Remote: " . $transferOperation->getRemotePath());
                mftpLog(LOG_DEBUG, "uploadFile: Local file exists: " . (file_exists($transferOperation->getLocalPath()) ? 'YES' : 'NO'));
            }

            // FIXED: Don't try to get file info before upload - file doesn't exist yet
            // This was causing "file not found" errors during upload
            $previousPermissions = null;

            if (!$this->handleUploadFile($transferOperation)) {
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_ERROR, "uploadFile: handleUploadFile failed for " . $transferOperation->getRemotePath());
                }
                $this->handleMultiTransferError('UPLOAD_OPERATION', $transferOperation);
            }
            else if ($preserveRemotePermissions && $this->supportsPermissionChange()) {
                // Get permissions after successful upload
                try {
                    $fileInfo = $this->handleGetFileInfo($transferOperation->getRemotePath());
                    $previousPermissions = $fileInfo->getNumericPermissions();
                } catch (Exception $e) {
                    // If we can't get file info after upload, continue without permission preservation
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_WARNING, "Could not get file info after upload for permission preservation: " . $e->getMessage());
                    }
                }
            }
            
            if ($preserveRemotePermissions && $previousPermissions !== null) {
                $this->changePermissions($previousPermissions, $transferOperation->getRemotePath());
            }
        }

        /**
         * @param $remotePath string
         * @throws FileSourceAuthenticationException
         * @throws FileSourceConnectionException
         * @throws FileSourceFileDoesNotExistException
         * @throws FileSourceFileExistsException
         * @throws FileSourceFilePermissionException
         * @throws FileSourceOperationException
         */
        public function deleteFile($remotePath) {
            $this->ensureConnectedAndAuthenticated('DELETE_FILE_OPERATION');

            if (!$this->handleDeleteFile($remotePath))
                $this->handleOperationError('DELETE_FILE_OPERATION', $remotePath, $this->getLastError());
        }

        /**
         * @param $remotePath string
         * @throws FileSourceAuthenticationException
         * @throws FileSourceConnectionException
         * @throws FileSourceFileDoesNotExistException
         * @throws FileSourceFileExistsException
         * @throws FileSourceFilePermissionException
         * @throws FileSourceOperationException
         */
        public function makeDirectory($remotePath) {
            $this->ensureConnectedAndAuthenticated('MAKE_DIRECTORY_OPERATION');

            if (!$this->handleMakeDirectory($remotePath))
                $this->handleOperationError('MAKE_DIRECTORY_OPERATION', $remotePath, $this->getLastError());
        }

        private function performIntermediateDirectoryCreate($fullPath) {
            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "performIntermediateDirectoryCreate: Creating directory '$fullPath'");
            }
            
            try {
                $this->makeDirectory($fullPath);
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_DEBUG, "performIntermediateDirectoryCreate: Directory '$fullPath' created successfully");
                }
            } catch (FileSourceFileExistsException $f) {
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_DEBUG, "performIntermediateDirectoryCreate: Directory '$fullPath' already exists");
                }
                return;
            } catch (Exception $e) {
                if (function_exists('mftpLog')) {
                    mftpLog(LOG_ERROR, "performIntermediateDirectoryCreate: Failed to create directory '$fullPath': " . $e->getMessage());
                }
                throw $e;
            }
        }

        /**
         * @param $remotePath
         * @throws FileSourceAuthenticationException
         * @throws FileSourceConnectionException
         * @throws FileSourceFileDoesNotExistException
         * @throws FileSourceFilePermissionException
         * @throws FileSourceOperationException
         */
        public function makeDirectoryWithIntermediates($remotePath) {
            $fullPath = '';

            $pathComponents = explode("/", $remotePath);

            foreach ($pathComponents as $pathComponent) {
                $fullPath .= "/" . $pathComponent;

                $fullPath = preg_replace("/^\\/+/", "/", $fullPath);

                if ($fullPath == "/" || ($this->getCurrentDirectory() != null && PathOperations::isParentPath($fullPath, $this->getCurrentDirectory())))
                    // / does not behave like other paths e.g. permission denied/does not exist so treat it special
                    continue;

                try {
                    $this->listDirectory($fullPath);
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_DEBUG, "makeDirectoryWithIntermediates: Directory '$fullPath' already exists");
                    }
                } catch (FileSourceFileDoesNotExistException $e) {
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_DEBUG, "makeDirectoryWithIntermediates: Directory '$fullPath' does not exist, creating it");
                    }
                    
                    try {
                        $this->performIntermediateDirectoryCreate($fullPath);
                        
                        // Verify the directory was actually created
                        $this->listDirectory($fullPath);
                        if (function_exists('mftpLog')) {
                            mftpLog(LOG_DEBUG, "makeDirectoryWithIntermediates: Directory '$fullPath' created and verified successfully");
                        }
                    } catch (Exception $createException) {
                        if (function_exists('mftpLog')) {
                            mftpLog(LOG_ERROR, "makeDirectoryWithIntermediates: Failed to create directory '$fullPath': " . $createException->getMessage());
                        }
                        throw new Exception("Failed to create directory '$fullPath': " . $createException->getMessage());
                    }
                } catch (Exception $e) {
                    if (function_exists('mftpLog')) {
                        mftpLog(LOG_ERROR, "makeDirectoryWithIntermediates: Unexpected error checking directory '$fullPath': " . $e->getMessage());
                    }
                    throw $e;
                }
            }
        }

        /**
         * @param $transferOperation
         */
        public function uploadFileToNewDirectory($transferOperation) {
            $remotePath = $transferOperation->getRemotePath();
            $remoteDirectory = PathOperations::remoteDirname($remotePath);
            
            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "uploadFileToNewDirectory: Creating directory '$remoteDirectory' for file '$remotePath'");
            }
            
            $this->makeDirectoryWithIntermediates($remoteDirectory);
            
            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "uploadFileToNewDirectory: Directory created, now uploading file '$remotePath'");
            }
            
            $this->uploadFile($transferOperation);
        }

        /**
         * @param $remotePath
         * @throws FileSourceAuthenticationException
         * @throws FileSourceConnectionException
         * @throws FileSourceFileDoesNotExistException
         * @throws FileSourceFileExistsException
         * @throws FileSourceFilePermissionException
         * @throws FileSourceOperationException
         */
        public function deleteDirectory($remotePath) {
            $this->ensureConnectedAndAuthenticated('DELETE_DIRECTORY_OPERATION');

            $dirList = $this->listDirectory($remotePath, true);

            foreach ($dirList as $item) {
                $childPath = PathOperations::join($remotePath, $item->getName());
                if ($item->isDirectory())
                    $this->deleteDirectory($childPath);
                else
                    $this->deleteFile($childPath);
            }

            if (!$this->handleDeleteDirectory($remotePath))
                $this->handleOperationError('DELETE_DIRECTORY_OPERATION', $remotePath, $this->getLastError());
        }

        public function deleteMultiple($remotePathsAndTypes) {
            $this->ensureConnectedAndAuthenticated('DELETE_MULTIPLE_OPERATION');

            foreach ($remotePathsAndTypes as $remotePathAndType) {
                $remotePath = $remotePathAndType[0];
                $isDirectory = $remotePathAndType[1];

                if ($isDirectory) {
                    $this->deleteDirectory($remotePath);
                    mftpActionLog("Delete directory", $this, dirname($remotePath), monstaBasename($remotePath), "");
                } else {
                    $this->deleteFile($remotePath);
                    mftpActionLog("Delete file", $this, dirname($remotePath), monstaBasename($remotePath), "");
                }
            }
        }

        /**
         * @param $source string
         * @param $destination string
         * @throws FileSourceAuthenticationException
         * @throws FileSourceConnectionException
         * @throws FileSourceFileDoesNotExistException
         * @throws FileSourceFileExistsException
         * @throws FileSourceFilePermissionException
         * @throws FileSourceOperationException
         */
        public function rename($source, $destination) {
            $debugFile = dirname(__FILE__) . '/../../../logs/mftp_debug.log';
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: ConnectionBase::rename() called\n", FILE_APPEND);
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: Source: " . $source . "\n", FILE_APPEND);
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: Destination: " . $destination . "\n", FILE_APPEND);
            
            $this->ensureConnectedAndAuthenticated('RENAME_OPERATION');
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: Connection ensured\n", FILE_APPEND);

            if (!$this->handleRename($source, $destination)) {
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: handleRename() failed\n", FILE_APPEND);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: Last error: " . $this->getLastError() . "\n", FILE_APPEND);
                $this->handleOperationError('RENAME_OPERATION', $source, $this->getLastError(), $destination);
            } else {
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: handleRename() succeeded\n", FILE_APPEND);
            }
        }

        /**
         * @param $mode int
         * @param $remotePath string
         * @throws FileSourceAuthenticationException
         * @throws FileSourceConnectionException
         * @throws FileSourceFileDoesNotExistException
         * @throws FileSourceFileExistsException
         * @throws FileSourceFilePermissionException
         * @throws FileSourceOperationException
         */
        public function changePermissions($mode, $remotePath) {
            $this->ensureConnectedAndAuthenticated('CHANGE_PERMISSIONS_OPERATION');

            Validation::validatePermissionMask($mode, false);

            if (!$this->handleChangePermissions($mode, $remotePath)) {
                $this->handleOperationError('CHANGE_PERMISSIONS_OPERATION', $remotePath, $this->getLastError());
            }
        }

        /**
         * @param $path
         * @return bool
         */
        public function isDirectory($path) {
            $debugFile = dirname(__FILE__) . '/../../../logs/mftp_debug.log';
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ISDIRECTORY DEBUG: Starting isDirectory check for: $path\n", FILE_APPEND);
            
            $this->ensureConnectedAndAuthenticated('IS_DIRECTORY');
            
            // Smart heuristic: if path ends with common file extensions, assume it's a file
            $fileExtensions = ['.zip', '.tar', '.gz', '.rar', '.7z', '.pdf', '.txt', '.doc', '.docx', '.xls', '.xlsx', '.jpg', '.jpeg', '.png', '.gif', '.mp4', '.avi', '.mov', '.mp3', '.wav', '.html', '.htm', '.css', '.js', '.php', '.xml', '.json'];
            $pathLower = strtolower($path);
            
            foreach ($fileExtensions as $ext) {
                if (substr($pathLower, -strlen($ext)) === $ext) {
                    // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ISDIRECTORY DEBUG: Detected file extension '$ext', returning false (file)\n", FILE_APPEND);
                    return false; // It's a file
                }
            }
            
            // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ISDIRECTORY DEBUG: No file extension detected, trying getFileInfo\n", FILE_APPEND);
            
            // Set a timeout for the operation to prevent hanging
            $originalTimeout = ini_get('default_socket_timeout');
            ini_set('default_socket_timeout', 10); // 10 second timeout
            
            try {
                $fileInfo = $this->handleGetFileInfo($path);
                $isDir = $fileInfo->isDirectory();
                ini_set('default_socket_timeout', $originalTimeout);
                return $isDir;
            } catch (Exception $e) {
                ini_set('default_socket_timeout', $originalTimeout);
                // If handleGetFileInfo fails, assume it's a file to avoid hanging
                return false;
            }
        }

        /**
         * @param $source string
         * @param $destination string
         * @throws FileSourceAuthenticationException
         * @throws FileSourceConnectionException
         */
        public function copy($source, $destination) {
            $this->ensureConnectedAndAuthenticated('COPY_OPERATION');

            $newPermissions = array();
            /*
            do the permissions update at the end, otherwise the copy may fail if a parent directory is made unreadable
            store the dest permissions here
            */

            $isDirectory = $this->isDirectory($source);

            if (!$isDirectory) {
                //mftpActionLog("Copy file", $this->connection, dirname($source), monstaBasename($source) . " to " . $destination, "");

                $sources = array(array($source, null));
                $destinations = array($destination);
            } else {
                //mftpActionLog("Copy folder", $this->connection, dirname($source), monstaBasename($source) . " to " . $destination, "");
                $fileFinder = new RecursiveFileFinder($this, $source);
                $sources = $fileFinder->findFilesAndDirectoriesInPaths();
                $destinations = array();

                for ($i = 0; $i < sizeof($sources); ++$i) {
                    $sourcePath = $sources[$i][0];

                    if (substr($sourcePath, 0, 1) == "/")
                        $sourcePath = substr($sourcePath, 1);

                    $destinations[] = PathOperations::join($destination, $sourcePath);
                    $sources[$i][0] = PathOperations::join($source, $sourcePath);
                }
            }

            if ($isDirectory && sizeof($sources) == 0) {
                $this->makeDirectory($destination);
                return;
            }

            $destinationDirs = array();

            for ($i = 0; $i < sizeof($sources); ++$i) {
                $destinationPath = $destinations[$i];
                $destinationDir = PathOperations::remoteDirname($destinationPath);

                $sourcePathAndItem = $sources[$i];

                $sourcePath = $sourcePathAndItem[0];
                $sourceItem = $sourcePathAndItem[1];

                if ($destinationDir != "" && $destinationDir != "/" &&
                    array_search($destinationDir, $destinationDirs) === false) {
                    $destinationDirs[] = $destinationDir;
                    $this->makeDirectoryWithIntermediates($destinationDir);
                }

                if ($sourceItem === null)
                    $this->handleCopy($sourcePath, $destinationPath);
                else {
                    if ($sourceItem->isDirectory()) {
                        if (array_search($destinationPath, $destinationDirs) === false) {
                            $destinationDirs[] = $destinationPath;
                            $this->makeDirectoryWithIntermediates($destinationPath);
                        }
                    } else {
                        $this->handleCopy($sourcePath, $destinationPath);
                    }

                    $newPermissions[$destinationPath] = $sourceItem->getNumericPermissions();
                }
            }

            if (!$this->supportsPermissionChange()) {
                return;
            }
            // Go through each copied path, deepest first, and apply permissions
            $destinationPaths = array_keys($newPermissions);

            usort($destinationPaths, "pathDepthCompare");

            foreach ($destinationPaths as $destinationPath) {
                $sourceNumericPermission = $newPermissions[$destinationPath];
                $this->changePermissions($sourceNumericPermission, $destinationPath);
            }

            /* this is kind of a special case so let the handleCopy raise an exception instead of going through
            handleOperationError, and downloadFile/uploadFile will call it anyway */
        }

        /**
         * @param $remotePath string
         * @return int
         */
        public function getFileSize($remotePath) {
            $directoryPath = PathOperations::remoteDirname($remotePath);
            $fileName = monstaBasename($remotePath);
            $directoryList = $this->listDirectory($directoryPath);

            foreach ($directoryList as $item) {
                if ($item->getName() == $fileName)
                    return $item->getSize();
            }

            return -1;
        }

        public function changeDirectory($newDirectory) {
            // for compatibility throughout the API; sometimes this does nothing
        }

        /**
         * @return bool
         * Some connections don't support CHMOD/permissions changing (namely FTP to Windows servers)
         */
        public function supportsPermissionChange() {
            return true;
        }

        public function getConfiguration() {
            return $this->configuration;
        }

        protected function syncCurrentDirectory() {
            $this->ensureConnectedAndAuthenticated('GET_CWD_OPERATION');

            $this->currentDirectory = $this->handleGetCurrentDirectory();
        }

        protected function handleGetCurrentDirectory() {
            // only overridden if supported
            return null;
        }

        protected function handleFetchServerCapabilities() {
            // only overridden if supported
            return null;
        }

        protected function getCapabilitiesArrayValue($capabilitiesKey) {
            if (is_null($this->serverCapabilitiesArray)) {
                return null;
            }

            return array_key_exists($capabilitiesKey, $this->serverCapabilitiesArray) ?
                $this->serverCapabilitiesArray[$capabilitiesKey] : null;
        }

        private function populateServerCapabilitiesArray() {
            $dataDir = dirname(__FILE__) . "/../../data/";
            $serverCapabilities = new ServerCapabilities(PathOperations::join($dataDir, "server_capabilities.php"));

            $capabilitiesArray = $serverCapabilities->getServerCapabilities($this->getProtocolName(),
                $this->configuration->getHost(), $this->configuration->getPort());

            if (is_null($capabilitiesArray)) {
                $capabilitiesArray = $this->handleFetchServerCapabilities();

                if (!is_null($capabilitiesArray)) {
                    $serverCapabilities->setServerCapabilities($this->getProtocolName(),
                        $this->configuration->getHost(), $this->configuration->getPort(), $capabilitiesArray);
                }
            }

            $this->serverCapabilitiesArray = $capabilitiesArray;
        }
    }