<?php

    require_once(dirname(__FILE__) . "/../../lib/helpers.php");
    require_once(dirname(__FILE__) . "/../PathOperations.php");
    require_once(dirname(__FILE__) . "/../../lib/LocalizableException.php");
    require_once(dirname(__FILE__) . "/../../lib/helpers.php");
    require_once(dirname(__FILE__) . "/ConnectionFactory.php");
    require_once(dirname(__FILE__) . "/../../vendor/autoload.php");

    use \wapmorgan\UnifiedArchive\UnifiedArchive;

    function recursiveDirectoryDelete($dir) {
        if (!file_exists($dir)) {
            return true;
        }

        if (!is_dir($dir)) {
            return unlink($dir);
        }

        foreach (scandir($dir) as $item) {
            if ($item == '.' || $item == '..') {
                continue;
            }

            if (!recursiveDirectoryDelete($dir . DIRECTORY_SEPARATOR . $item)) {
                return false;
            }

        }

        return rmdir($dir);
    }

    class ArchiveExtractor {
        private $archivePath;
        private $uploadDirectory;
        private $existingDirectories;
        private $extractDirectory;
        private $flatFileList;
        private $archiveHandle;
        private $archiveExtension;
        private $isSkipMacOsFiles;

        public function __construct($archivePath, $uploadDirectory, $skipMacOsFiles = false) {
            $this->archivePath = $archivePath;
            $this->uploadDirectory = $uploadDirectory;
            $this->existingDirectories = array();
            $this->flatFileList = null;
            $this->archiveHandle = null;
            $this->archiveExtension = pathinfo($archivePath, PATHINFO_EXTENSION);
            $this->isSkipMacOsFiles = $skipMacOsFiles;
        }

        private function getSessionKey() {
            return "archive_contents_" . md5($this->getArchivePath());
        }

        private function getArchivePath() {
            return $this->archivePath;
        }

        private function extractArchiveFilePath($fullFilePath) {
            return $fullFilePath;
        }

        private function setupArchiveHandle() {
            if ($this->archiveHandle !== null)
                return;

            $archivePath = $this->getArchivePath();

            $this->archiveHandle = UnifiedArchive::open($archivePath);
            
            if (!$this->archiveHandle) {
                throw new Exception("Failed to open archive: $archivePath");
            }

            $sessionKey = MFTP_SESSION_KEY_PREFIX . $this->getSessionKey();
            
            // Force refresh if we have cached data but it might have bad indexing
            if (isset($_SESSION[$sessionKey])) {
                $cached = $_SESSION[$sessionKey];
                if (is_array($cached) && count($cached) > 0 && !isset($cached[0])) {
                    error_log("ArchiveExtractor: Clearing bad cached file list with non-zero-based keys for reindexing");
                    unset($_SESSION[$sessionKey]);
                }
            }

            if (isset($_SESSION[$sessionKey])) {
                $this->flatFileList = $_SESSION[$sessionKey];
                error_log("ArchiveExtractor: Using cached file list with " . count($this->flatFileList) . " files");
            } else {
                $fileNames = $this->archiveHandle->getFileNames();
                if (empty($fileNames)) {
                    error_log("ArchiveExtractor: Warning - Archive appears to be empty or file list could not be retrieved: $archivePath");
                    $this->flatFileList = array(); // Ensure it's an array
                } else {
                    // Reindex the array to ensure 0-based sequential keys
                    // UnifiedArchive sometimes returns arrays with non-sequential keys starting from 1
                    $this->flatFileList = array_values($fileNames);
                    error_log("ArchiveExtractor: Reindexed file list from keys [" . implode(', ', array_slice(array_keys($fileNames), 0, 10)) . "] to 0-based indices for " . count($this->flatFileList) . " files");
                }
                $_SESSION[$sessionKey] = $this->flatFileList;
            }
            
            // Additional validation
            if (!is_array($this->flatFileList)) {
                error_log("ArchiveExtractor: flatFileList is not an array, converting to empty array");
                $this->flatFileList = array();
            }
        }

        public function getFileCount() {
            $this->setupArchiveHandle();
            
            if (!is_array($this->flatFileList)) {
                error_log("ArchiveExtractor: getFileCount called but flatFileList is not an array");
                return 0;
            }
            
            $count = count($this->flatFileList);
            if ($count === 0) {
                error_log("ArchiveExtractor: Archive contains no files or file list is empty");
            }
            
            return $count;
        }

        private function getFileInfoAtIndex($fileIndex) {
            try {
                // Ensure archive is set up and file list is initialized
                $this->setupArchiveHandle();
                
                // Validate that the index exists in our file list
                if (!isset($this->flatFileList[$fileIndex])) {
                    error_log("ArchiveExtractor: File index $fileIndex not found. Available indices: 0-" . (count($this->flatFileList) - 1));
                    return array('', false);
                }
                
                $fileName = $this->flatFileList[$fileIndex];
                if (empty($fileName)) {
                    error_log("ArchiveExtractor: Empty filename at index $fileIndex");
                    return array('', false);
                }
                
                $fileInfo = $this->archiveHandle->getFileData($fileName);

                if (!$fileInfo) {
                    error_log("ArchiveExtractor: Error getting file info at index $fileIndex: File $fileName does not exist in archive");
                    return array($fileName, false);
                }

                // Handle different property names in UnifiedArchive library
                $fileName = '';
                if (property_exists($fileInfo, 'path') && !empty($fileInfo->path)) {
                    $fileName = $fileInfo->path;
                } elseif (property_exists($fileInfo, 'filename') && !empty($fileInfo->filename)) {
            $fileName = $fileInfo->filename;
                } elseif (property_exists($fileInfo, 'name') && !empty($fileInfo->name)) {
                    $fileName = $fileInfo->name;
                } else {
                    // Fallback: use the flatFileList entry itself
                    $fileName = $this->flatFileList[$fileIndex] ?? '';
                }

                // Ensure fileName is not null and is a string
                if ($fileName === null || !is_string($fileName)) {
                    $fileName = '';
                }

                $isDirectory = false;
                if (strlen($fileName) > 0) {
            $isDirectory = substr($fileName, strlen($fileName) - 1) == "/";
                }

            return array($fileName, $isDirectory);
                
            } catch (Exception $e) {
                // Return safe fallback values if anything goes wrong
                error_log("ArchiveExtractor: Error getting file info at index $fileIndex: " . $e->getMessage());
                return array($this->flatFileList[$fileIndex] ?? '', false);
            }
        }

        public function extractAndUpload($connection, $fileOffset, $stepCount) {
            $connection->changeDirectory($this->uploadDirectory);

            $this->createExtractDirectory();

            $fileMax = min($this->getFileCount(), $fileOffset + $stepCount);

            $itemsTransferred = 0;

            $startTime = time();

            try {
                for (; $fileOffset < $fileMax; ++$fileOffset) {
                    $this->extractAndUploadItem($connection, $this->archiveHandle, $fileOffset);
                    ++$itemsTransferred;

                    outputStreamKeepAlive();

                    if (time() - $startTime >= MFTP_EXTRACT_UPLOAD_TIME_LIMIT_SECONDS)
                        break;
                }
            } catch (Exception $e) {
                recursiveDirectoryDelete($this->extractDirectory);
                throw $e;
                // this should be done in a finally to avoid repeated code but we need to support PHP < 5.5
            }

            recursiveDirectoryDelete($this->extractDirectory);

            $extractFinished = $this->getFileCount() == $fileOffset;

            if ($extractFinished)
                $this->cleanup();

            return array($extractFinished, $itemsTransferred);
        }

        private function cleanup() {
            if (isset($_SESSION[MFTP_SESSION_KEY_PREFIX . $this->getSessionKey()]))
                unset($_SESSION[MFTP_SESSION_KEY_PREFIX . $this->getSessionKey()]);
        }

        private function getTransferOperation($connection, $localPath, $remotePath) {
            return TransferOperationFactory::getTransferOperation(strtolower($connection->getProtocolName()),
                array(
                    "localPath" => $localPath,
                    "remotePath" => $remotePath
                )
            );
        }

        private function createExtractDirectory() {
            $tempPath = monstaTempnam(getMonstaSharedTransferDirectory(), monstaBasename($this->archivePath) . "extract-dir");

            if (file_exists($tempPath))
                unlink($tempPath);

            mkdir($tempPath);
            if (!is_dir($tempPath))
                throw new Exception("Temp archive dir was not a dir");

            $this->extractDirectory = $tempPath;
        }

        private function isPathTraversalPath($itemName) {
            try {
                // Use enhanced path validation from PathOperations
                PathOperations::validatePathSecurity($itemName, false);
                return false;
            } catch (InvalidArgumentException $e) {
                // If validation fails, it's unsafe
                return true;
            }
        }

        private function extractFileToDisk($archive, $extractDir, $itemPath) {
            try {
                // Suppress warnings to prevent JSON corruption
                $errorReporting = error_reporting(E_ERROR);
                
                if ($this->archiveExtension == "gz") {
                    // gzip may only be extracted all at once
                    if (@$archive->extract($extractDir) === false)
                        throw new Exception("Unable to extract from gzip archive");
                } else {
                    // Extract specific file - $files must be passed by reference
                    $files = [$itemPath];
                    if (@$archive->extractFiles($extractDir, $files) === false)
                        throw new Exception("Unable to extract $itemPath from archive");
                }
                
                // Restore error reporting
                error_reporting($errorReporting);
                
            } catch (Exception $e) {
                // Restore error reporting
                if (isset($errorReporting)) {
                    error_reporting($errorReporting);
                }
                
                // Fallback: extract all files if specific extraction fails
                try {
                    if (@$archive->extract($extractDir) === false)
                        throw new Exception("Unable to extract archive using fallback method");
                } catch (Exception $fallbackException) {
                    throw new Exception("Failed to extract $itemPath: " . $e->getMessage() . " (Fallback also failed: " . $fallbackException->getMessage() . ")");
                }
            }
        }

        private function extractAndUploadItem($connection, $archive, $itemIndex) {
            $itemInfo = $this->getFileInfoAtIndex($itemIndex);

            if ($this->isPathTraversalPath($itemInfo[0]))
                return;

            $itemIsDirectory = $itemInfo[1] === true;

            $archiveInternalPath = $this->extractArchiveFilePath($itemInfo[0]);

            if($this->isSkipMacOsFiles && (preg_match('/(^|\/)__MACOSX\//m', $archiveInternalPath) ||
                    preg_match('/(^|\/)\.DS_Store(\/|$)/m', $archiveInternalPath))) {
                return;
            }

            if (DIRECTORY_SEPARATOR == "\\")
                $archiveInternalPath = str_replace("\\", "/", $archiveInternalPath); // fix in windows

            if (!$itemIsDirectory)
                $this->extractFileToDisk($archive, $this->extractDirectory, $archiveInternalPath);

            $itemPath = PathOperations::join($this->extractDirectory, $archiveInternalPath);

            if (is_null($itemInfo[1]) && is_dir($itemPath))
                return;

            $uploadPath = PathOperations::join($this->uploadDirectory, $archiveInternalPath);

            $remoteDirectoryPath = $itemIsDirectory ? $uploadPath : PathOperations::remoteDirname($uploadPath);

            if (!$this->directoryRecordExists($remoteDirectoryPath)) {
                $connection->makeDirectoryWithIntermediates($remoteDirectoryPath);
                $this->recordExistingDirectories(PathOperations::directoriesInPath($remoteDirectoryPath));
            }

            if ($itemIsDirectory)
                return; // directory was created so just return, don't upload it

            $uploadOperation = $this->getTransferOperation($connection, $itemPath, $uploadPath);

            try {
                $connection->uploadFile($uploadOperation);
            } catch (Exception $e) {
                @unlink($itemPath);
                throw $e;
                // this should be done in a finally to avoid repeated code but we need to support PHP < 5.5
            }

            @unlink($itemPath);
        }

        private function directoryRecordExists($directoryPath) {
            // this is not true directory exists function, just if we have created it or a subdirectory in this object
            return array_search(PathOperations::normalize($directoryPath), $this->existingDirectories) !== false;
        }

        private function recordDirectoryExists($directoryPath) {
            if ($this->directoryRecordExists($directoryPath))
                return;

            $this->existingDirectories[] = PathOperations::normalize($directoryPath);
        }

        private function recordExistingDirectories($existingDirectories) {
            foreach ($existingDirectories as $existingDirectory) {
                $this->recordDirectoryExists($existingDirectory);
            }
        }
    }