<?php

    /**
     * @param mixed $val
     * @param bool $allowNull
     * @param callable $isTypeFunction
     * @param string $typeName
     */
    function _validateType($val, bool $allowNull, callable $isTypeFunction, string $typeName) {
        if ($allowNull && is_null($val))
            return;

        // Modern callable syntax - avoid deprecated call_user_func
        if (!$isTypeFunction($val))
            throw new InvalidArgumentException("Argument must be $typeName, got " . gettype($val) . " type '$val'.");
    }

    class Validation {
        /**
         * @param mixed $val
         * @param string $argName
         * @param bool $allowNull
         */
        public static function validateNonEmptyString($val, string $argName = '', bool $allowNull = false) {
            if ($allowNull && $val === null) {
                return; // Allow null values if explicitly permitted
            }
            if (!is_string($val) || strlen($val) == 0)
                throw new InvalidArgumentException("Argument $argName must be non zero-length string. Got: \"$val\"");
        }

        /**
         * @param int|null $mask
         * @param bool $allowNull
         */
        public static function validatePermissionMask(int|null $mask, bool $allowNull = false): void {
            Validation::validateInteger($mask, $allowNull);

            if ($mask < 0 || $mask > 0777)
                throw new InvalidArgumentException(sprintf("File mode out of range: 0%o.", $mask));
        }

        /**
         * @param int|null $val
         * @param bool $allowNull
         */
        public static function validateInteger(int|null $val, bool $allowNull = false): void {
            _validateType($val, $allowNull, 'is_int', "integer");
        }

        /**
         * @param string|null $val
         * @param bool $allowNull
         */
        public static function validateString(string|null $val, bool $allowNull = false): void {
            _validateType($val, $allowNull, 'is_string', "string");
        }

        public static function validateBoolean(bool|null $val, bool $allowNull = false): void {
            _validateType($val, $allowNull, 'is_bool', "boolean");
        }

        /**
         * @param $arrayOrNull array|null
         * @param $key string
         * @return mixed|null
         */
        public static function getArrayValueOrNull($arrayOrNull, $key) {
            if (!is_array($arrayOrNull))
                return null;

            if (!array_key_exists($key, $arrayOrNull))
                return null;

            return $arrayOrNull[$key];
        }
    }
    