<?php

require_once(dirname(__FILE__) . '/../constants.php');
require_once(dirname(__FILE__) . '/PerformanceOptimizer.php');
require_once(dirname(__FILE__) . '/../lib/helpers.php');
require_once(dirname(__FILE__) . '/../lib/logging.php');

/**
 * Transfer resume and file caching manager
 * Handles interrupted transfers, file caching, and resume capabilities
 */
class TransferResumeManager {
    
    private static $instance = null;
    private $resumeDirectory;
    private $cacheDirectory;
    private $performanceOptimizer;
    private $maxCacheSize;
    private $maxCacheAge;
    private $resumeData = [];
    
    // Resume file statuses
    const STATUS_PENDING = 'pending';
    const STATUS_ACTIVE = 'active';
    const STATUS_COMPLETED = 'completed';
    const STATUS_FAILED = 'failed';
    const STATUS_CANCELLED = 'cancelled';
    
    // Cache types
    const CACHE_TYPE_TRANSFER = 'transfer';
    const CACHE_TYPE_DIRECTORY = 'directory';
    const CACHE_TYPE_FILE_INFO = 'file_info';
    
    // Configuration
    const DEFAULT_MAX_CACHE_SIZE = 104857600; // 100MB
    const DEFAULT_MAX_CACHE_AGE = 86400; // 24 hours
    const RESUME_CHUNK_SIZE = 65536; // 64KB chunks for resume
    
    private function __construct() {
        $this->performanceOptimizer = PerformanceOptimizer::getInstance();
        
        // Initialize directories
        $tempDir = sys_get_temp_dir();
        $this->resumeDirectory = $tempDir . '/mftp_resume';
        $this->cacheDirectory = $tempDir . '/mftp_cache';
        
        $this->createDirectories();
        
        // Configuration
        $this->maxCacheSize = self::DEFAULT_MAX_CACHE_SIZE;
        $this->maxCacheAge = self::DEFAULT_MAX_CACHE_AGE;
        
        // Load existing resume data
        $this->loadResumeData();
        
        mftpLog(LOG_DEBUG, "TransferResumeManager: Initialized with resume dir: {$this->resumeDirectory}");
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Create a resumable transfer session
     */
    public function createResumableTransfer($remotePath, $localPath, $totalSize, $connectionType = 'ftp') {
        $transferId = $this->generateTransferId($remotePath, $localPath);
        
        $resumeInfo = [
            'transfer_id' => $transferId,
            'remote_path' => $remotePath,
            'local_path' => $localPath,
            'total_size' => $totalSize,
            'connection_type' => $connectionType,
            'status' => self::STATUS_PENDING,
            'created_at' => time(),
            'last_updated' => time(),
            'bytes_transferred' => 0,
            'chunks_completed' => [],
            'failed_chunks' => [],
            'retry_count' => 0,
            'checksum' => null
        ];
        
        $this->resumeData[$transferId] = $resumeInfo;
        $this->saveResumeData();
        
        mftpLog(LOG_INFO, "TransferResumeManager: Created resumable transfer $transferId");
        
        return $transferId;
    }
    
    /**
     * Resume an interrupted transfer
     */
    public function resumeTransfer($transferId, $connection) {
        if (!isset($this->resumeData[$transferId])) {
            throw new InvalidArgumentException("Resume data not found for transfer ID: $transferId");
        }
        
        $resumeInfo = &$this->resumeData[$transferId];
        $resumeInfo['status'] = self::STATUS_ACTIVE;
        $resumeInfo['last_updated'] = time();
        
        $startOffset = $resumeInfo['bytes_transferred'];
        $remainingSize = $resumeInfo['total_size'] - $startOffset;
        
        mftpLog(LOG_INFO, "TransferResumeManager: Resuming transfer $transferId from offset $startOffset");
        
        try {
            $result = $this->performResumeTransfer($resumeInfo, $connection, $startOffset);
            
            if ($result['success']) {
                $resumeInfo['status'] = self::STATUS_COMPLETED;
                $resumeInfo['bytes_transferred'] = $resumeInfo['total_size'];
                $this->cleanupResumeData($transferId);
            } else {
                $resumeInfo['status'] = self::STATUS_FAILED;
                $resumeInfo['retry_count']++;
            }
            
            $this->saveResumeData();
            return $result;
            
        } catch (Exception $e) {
            $resumeInfo['status'] = self::STATUS_FAILED;
            $resumeInfo['retry_count']++;
            $this->saveResumeData();
            throw $e;
        }
    }
    
    /**
     * Check if a transfer can be resumed
     */
    public function canResumeTransfer($remotePath, $localPath) {
        $transferId = $this->generateTransferId($remotePath, $localPath);
        
        if (!isset($this->resumeData[$transferId])) {
            return false;
        }
        
        $resumeInfo = $this->resumeData[$transferId];
        
        // Check if partial file exists
        if (!file_exists($resumeInfo['local_path'])) {
            return false;
        }
        
        // Check if the partial file size matches our records
        $currentSize = filesize($resumeInfo['local_path']);
        if ($currentSize !== $resumeInfo['bytes_transferred']) {
            return false;
        }
        
        // Check if it's worth resuming (at least 1MB transferred)
        return $resumeInfo['bytes_transferred'] >= 1048576;
    }
    
    /**
     * Get resume information for a transfer
     */
    public function getResumeInfo($transferId) {
        return $this->resumeData[$transferId] ?? null;
    }
    
    /**
     * Update transfer progress
     */
    public function updateTransferProgress($transferId, $bytesTransferred, $chunkIndex = null) {
        if (!isset($this->resumeData[$transferId])) {
            return;
        }
        
        $resumeInfo = &$this->resumeData[$transferId];
        $resumeInfo['bytes_transferred'] = $bytesTransferred;
        $resumeInfo['last_updated'] = time();
        
        if ($chunkIndex !== null) {
            $resumeInfo['chunks_completed'][] = $chunkIndex;
        }
        
        // Save progress periodically (every 5 seconds)
        if (time() - ($resumeInfo['last_saved'] ?? 0) >= 5) {
            $this->saveResumeData();
            $resumeInfo['last_saved'] = time();
        }
    }
    
    /**
     * Cache file or directory information
     */
    public function cacheData($key, $data, $type = self::CACHE_TYPE_TRANSFER, $ttl = null) {
        $ttl = $ttl ?: $this->maxCacheAge;
        $cacheFile = $this->getCacheFilePath($key, $type);
        
        $cacheData = [
            'data' => $data,
            'type' => $type,
            'created_at' => time(),
            'expires_at' => time() + $ttl,
            'size' => strlen(serialize($data))
        ];
        
        if (file_put_contents($cacheFile, serialize($cacheData)) !== false) {
            mftpLog(LOG_DEBUG, "TransferResumeManager: Cached data for key: $key");
            $this->cleanupExpiredCache();
        }
    }
    
    /**
     * Retrieve cached data
     */
    public function getCachedData($key, $type = self::CACHE_TYPE_TRANSFER) {
        $cacheFile = $this->getCacheFilePath($key, $type);
        
        if (!file_exists($cacheFile)) {
            return null;
        }
        
        $cacheData = unserialize(file_get_contents($cacheFile));
        
        if (!$cacheData || $cacheData['expires_at'] < time()) {
            unlink($cacheFile);
            return null;
        }
        
        return $cacheData['data'];
    }
    
    /**
     * Cache directory listing for faster subsequent access
     */
    public function cacheDirectoryListing($path, $listing, $connectionType) {
        $cacheKey = $this->generateCacheKey($path, $connectionType, 'directory');
        $this->cacheData($cacheKey, $listing, self::CACHE_TYPE_DIRECTORY);
    }
    
    /**
     * Get cached directory listing
     */
    public function getCachedDirectoryListing($path, $connectionType) {
        $cacheKey = $this->generateCacheKey($path, $connectionType, 'directory');
        return $this->getCachedData($cacheKey, self::CACHE_TYPE_DIRECTORY);
    }
    
    /**
     * Cache file information (size, permissions, etc.)
     */
    public function cacheFileInfo($path, $fileInfo, $connectionType) {
        $cacheKey = $this->generateCacheKey($path, $connectionType, 'file_info');
        $this->cacheData($cacheKey, $fileInfo, self::CACHE_TYPE_FILE_INFO, 3600); // 1 hour TTL for file info
    }
    
    /**
     * Get cached file information
     */
    public function getCachedFileInfo($path, $connectionType) {
        $cacheKey = $this->generateCacheKey($path, $connectionType, 'file_info');
        return $this->getCachedData($cacheKey, self::CACHE_TYPE_FILE_INFO);
    }
    
    /**
     * Perform the actual resume transfer
     */
    private function performResumeTransfer($resumeInfo, $connection, $startOffset) {
        $localPath = $resumeInfo['local_path'];
        $remotePath = $resumeInfo['remote_path'];
        $remainingSize = $resumeInfo['total_size'] - $startOffset;
        
        // Open local file for appending
        $localHandle = fopen($localPath, 'ab');
        if (!$localHandle) {
            throw new RuntimeException("Failed to open local file for resume: $localPath");
        }
        
        try {
            // Get optimal buffer size for resume transfer
            $bufferSize = $this->performanceOptimizer->getOptimalBufferSize($remainingSize, 
                $resumeInfo['connection_type']);
            
            // Resume the transfer based on connection type
            if (method_exists($connection, 'resumeDownload')) {
                // Use native resume if available
                $result = $connection->resumeDownload($remotePath, $localHandle, $startOffset);
            } else {
                // Manual resume implementation
                $result = $this->manualResumeTransfer($connection, $remotePath, $localHandle, 
                    $startOffset, $bufferSize, $resumeInfo['transfer_id']);
            }
            
            return $result;
            
        } finally {
            fclose($localHandle);
        }
    }
    
    /**
     * Manual resume transfer implementation
     */
    private function manualResumeTransfer($connection, $remotePath, $localHandle, $startOffset, $bufferSize, $transferId) {
        // This is a simplified implementation
        // In a real scenario, you'd need to handle FTP REST command or SFTP seek
        
        $bytesTransferred = 0;
        $chunkIndex = intval($startOffset / $bufferSize);
        
        try {
            // For FTP, you would send REST command
            // For SFTP, you would seek to the offset
            
            // Simulate resume transfer
            while ($bytesTransferred < ($this->resumeData[$transferId]['total_size'] - $startOffset)) {
                // Read chunk from remote
                // Write chunk to local file
                // Update progress
                
                $this->updateTransferProgress($transferId, $startOffset + $bytesTransferred, $chunkIndex);
                $chunkIndex++;
                
                // Break condition for demonstration
                break;
            }
            
            return ['success' => true, 'bytes_transferred' => $bytesTransferred];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Generate a unique transfer ID
     */
    private function generateTransferId($remotePath, $localPath) {
        return md5($remotePath . '|' . $localPath . '|' . time());
    }
    
    /**
     * Generate cache key
     */
    private function generateCacheKey($path, $connectionType, $operation) {
        return md5($connectionType . '|' . $operation . '|' . $path);
    }
    
    /**
     * Get cache file path
     */
    private function getCacheFilePath($key, $type) {
        return $this->cacheDirectory . '/' . $type . '_' . $key . '.cache';
    }
    
    /**
     * Load resume data from storage
     */
    private function loadResumeData() {
        $resumeFile = $this->resumeDirectory . '/resume_data.json';
        
        if (file_exists($resumeFile)) {
            $data = json_decode(file_get_contents($resumeFile), true);
            if ($data) {
                $this->resumeData = $data;
                mftpLog(LOG_DEBUG, "TransferResumeManager: Loaded " . count($this->resumeData) . " resume entries");
            }
        }
    }
    
    /**
     * Save resume data to storage
     */
    private function saveResumeData() {
        $resumeFile = $this->resumeDirectory . '/resume_data.json';
        file_put_contents($resumeFile, json_encode($this->resumeData, JSON_PRETTY_PRINT));
    }
    
    /**
     * Clean up completed resume data
     */
    private function cleanupResumeData($transferId) {
        unset($this->resumeData[$transferId]);
        $this->saveResumeData();
    }
    
    /**
     * Clean up expired cache files
     */
    private function cleanupExpiredCache() {
        $currentTime = time();
        $cacheFiles = glob($this->cacheDirectory . '/*.cache');
        $totalSize = 0;
        $fileSizes = [];
        
        foreach ($cacheFiles as $file) {
            $fileSize = filesize($file);
            $totalSize += $fileSize;
            $fileSizes[$file] = $fileSize;
            
            // Check if file is expired
            $fileContents = file_get_contents($file);
            $cacheData = $fileContents !== false ? unserialize($fileContents) : false;
            if (!$cacheData || $cacheData['expires_at'] < $currentTime) {
                unlink($file);
                $totalSize -= $fileSize;
                unset($fileSizes[$file]);
            }
        }
        
        // If cache size exceeds limit, remove oldest files
        if ($totalSize > $this->maxCacheSize) {
            arsort($fileSizes); // Sort by size, largest first
            foreach ($fileSizes as $file => $size) {
                unlink($file);
                $totalSize -= $size;
                if ($totalSize <= $this->maxCacheSize) {
                    break;
                }
            }
        }
    }
    
    /**
     * Create necessary directories
     */
    private function createDirectories() {
        if (!file_exists($this->resumeDirectory)) {
            mkdir($this->resumeDirectory, 0755, true);
        }
        
        if (!file_exists($this->cacheDirectory)) {
            mkdir($this->cacheDirectory, 0755, true);
        }
    }
    
    /**
     * Get transfer statistics
     */
    public function getTransferStats() {
        $stats = [
            'total_resume_entries' => count($this->resumeData),
            'pending_transfers' => 0,
            'active_transfers' => 0,
            'completed_transfers' => 0,
            'failed_transfers' => 0,
            'cache_files' => 0,
            'cache_size' => 0
        ];
        
        // Count resume entries by status
        foreach ($this->resumeData as $resume) {
            switch ($resume['status']) {
                case self::STATUS_PENDING:
                    $stats['pending_transfers']++;
                    break;
                case self::STATUS_ACTIVE:
                    $stats['active_transfers']++;
                    break;
                case self::STATUS_COMPLETED:
                    $stats['completed_transfers']++;
                    break;
                case self::STATUS_FAILED:
                    $stats['failed_transfers']++;
                    break;
            }
        }
        
        // Count cache files
        $cacheFiles = glob($this->cacheDirectory . '/*.cache');
        $stats['cache_files'] = count($cacheFiles);
        
        foreach ($cacheFiles as $file) {
            $stats['cache_size'] += filesize($file);
        }
        
        return $stats;
    }
    
    /**
     * Clear all cache and resume data
     */
    public function clearAll() {
        // Clear resume data
        $this->resumeData = [];
        $this->saveResumeData();
        
        // Clear cache files
        $cacheFiles = glob($this->cacheDirectory . '/*.cache');
        foreach ($cacheFiles as $file) {
            unlink($file);
        }
        
        mftpLog(LOG_INFO, "TransferResumeManager: Cleared all cache and resume data");
    }
} 