<?php

require_once(dirname(__FILE__) . '/../constants.php');
require_once(dirname(__FILE__) . '/PerformanceOptimizer.php');
require_once(dirname(__FILE__) . '/../lib/helpers.php');
require_once(dirname(__FILE__) . '/../lib/logging.php');

/**
 * High-performance streaming file transfer manager
 * Handles efficient upload/download with dynamic buffer optimization
 */
class StreamingTransferManager {
    
    private $performanceOptimizer;
    private $progressCallback;
    private $transferStats;
    
    // Transfer states
    const STATE_IDLE = 'idle';
    const STATE_UPLOADING = 'uploading';
    const STATE_DOWNLOADING = 'downloading';
    const STATE_COMPLETED = 'completed';
    const STATE_ERROR = 'error';
    const STATE_CANCELLED = 'cancelled';
    
    public function __construct($progressCallback = null) {
        $this->performanceOptimizer = PerformanceOptimizer::getInstance();
        $this->progressCallback = $progressCallback;
        $this->resetTransferStats();
    }
    
    /**
     * Reset transfer statistics
     */
    private function resetTransferStats() {
        $this->transferStats = [
            'state' => self::STATE_IDLE,
            'bytes_transferred' => 0,
            'total_bytes' => 0,
            'start_time' => null,
            'last_progress_time' => null,
            'transfer_rate' => 0,
            'eta_seconds' => null,
            'error_message' => null
        ];
    }
    
    /**
     * Stream upload with optimized buffering
     */
    public function streamUpload($localPath, $remoteHandle, $fileSize = null, $connectionType = 'ftp', $isSSL = false) {
        if (!is_readable($localPath)) {
            throw new InvalidArgumentException("Local file is not readable: $localPath");
        }
        
        $actualFileSize = $fileSize ?: filesize($localPath);
        $bufferSize = $this->performanceOptimizer->getOptimalBufferSize($actualFileSize, $connectionType, $isSSL);
        $progressInterval = $this->performanceOptimizer->getProgressCallbackInterval($actualFileSize);
        
        $this->initializeTransfer($actualFileSize, self::STATE_UPLOADING);
        
        $localHandle = fopen($localPath, 'rb');
        if (!$localHandle) {
            throw new RuntimeException("Failed to open local file for reading: $localPath");
        }
        
        try {
            $this->performStreamingTransfer($localHandle, $remoteHandle, $bufferSize, $progressInterval, 'upload');
        } finally {
            fclose($localHandle);
        }
        
        return $this->completeTransfer();
    }
    
    /**
     * Stream download with optimized buffering
     */
    public function streamDownload($remoteHandle, $localPath, $fileSize = null, $connectionType = 'ftp', $isSSL = false) {
        $bufferSize = $this->performanceOptimizer->getOptimalBufferSize($fileSize, $connectionType, $isSSL);
        $progressInterval = $this->performanceOptimizer->getProgressCallbackInterval($fileSize);
        
        $this->initializeTransfer($fileSize, self::STATE_DOWNLOADING);
        
        $localHandle = fopen($localPath, 'wb');
        if (!$localHandle) {
            throw new RuntimeException("Failed to open local file for writing: $localPath");
        }
        
        try {
            $this->performStreamingTransfer($remoteHandle, $localHandle, $bufferSize, $progressInterval, 'download');
        } finally {
            fclose($localHandle);
        }
        
        return $this->completeTransfer();
    }
    
    /**
     * Stream chunked upload with resume capability
     */
    public function streamChunkedUpload($localPath, $uploadCallback, $chunkSize = null, $resumeOffset = 0) {
        if (!is_readable($localPath)) {
            throw new InvalidArgumentException("Local file is not readable: $localPath");
        }
        
        $fileSize = filesize($localPath);
        $actualChunkSize = $chunkSize ?: $this->performanceOptimizer->getOptimalChunkSize($fileSize);
        
        $this->initializeTransfer($fileSize, self::STATE_UPLOADING);
        $this->transferStats['bytes_transferred'] = $resumeOffset;
        
        $handle = fopen($localPath, 'rb');
        if (!$handle) {
            throw new RuntimeException("Failed to open local file for reading: $localPath");
        }
        
        try {
            if ($resumeOffset > 0) {
                fseek($handle, $resumeOffset);
            }
            
            $chunkIndex = 0;
            while (!feof($handle) && $this->transferStats['state'] === self::STATE_UPLOADING) {
                $chunk = fread($handle, $actualChunkSize);
                if ($chunk === false || $chunk === '') {
                    break;
                }
                
                $chunkResult = $uploadCallback($chunk, $chunkIndex, $this->transferStats['bytes_transferred']);
                if (!$chunkResult) {
                    throw new RuntimeException("Chunk upload failed at index $chunkIndex");
                }
                
                $this->transferStats['bytes_transferred'] += strlen($chunk);
                $this->updateProgress();
                $chunkIndex++;
            }
        } finally {
            fclose($handle);
        }
        
        return $this->completeTransfer();
    }
    
    /**
     * Initialize transfer statistics
     */
    private function initializeTransfer($totalBytes, $state) {
        $this->transferStats['state'] = $state;
        $this->transferStats['total_bytes'] = $totalBytes ?: 0;
        $this->transferStats['bytes_transferred'] = 0;
        $this->transferStats['start_time'] = microtime(true);
        $this->transferStats['last_progress_time'] = $this->transferStats['start_time'];
        
        mftpLog(LOG_DEBUG, "StreamingTransferManager: Initialized $state transfer, size: " . 
               ($totalBytes ? number_format($totalBytes) . ' bytes' : 'unknown'));
    }
    
    /**
     * Perform the actual streaming transfer between handles
     */
    private function performStreamingTransfer($sourceHandle, $destHandle, $bufferSize, $progressInterval, $direction) {
        $lastProgressReport = time();
        
        while (!feof($sourceHandle) && $this->transferStats['state'] !== self::STATE_CANCELLED) {
            $buffer = fread($sourceHandle, $bufferSize);
            if ($buffer === false || $buffer === '') {
                break;
            }
            
            $bytesWritten = fwrite($destHandle, $buffer);
            if ($bytesWritten === false) {
                throw new RuntimeException("Failed to write data during $direction");
            }
            
            $this->transferStats['bytes_transferred'] += $bytesWritten;
            
            // Update progress at specified intervals
            $currentTime = time();
            if ($currentTime - $lastProgressReport >= $progressInterval) {
                $this->updateProgress();
                $lastProgressReport = $currentTime;
            }
            
            // Keep connection alive for long transfers
            if ($direction === 'upload') {
                outputStreamKeepAlive();
            }
        }
        
        // Final progress update
        $this->updateProgress();
    }
    
    /**
     * Update transfer progress and statistics
     */
    private function updateProgress() {
        $currentTime = microtime(true);
        $elapsed = $currentTime - $this->transferStats['start_time'];
        
        if ($elapsed > 0) {
            $this->transferStats['transfer_rate'] = $this->transferStats['bytes_transferred'] / $elapsed;
            
            if ($this->transferStats['total_bytes'] > 0 && $this->transferStats['transfer_rate'] > 0) {
                $remaining = $this->transferStats['total_bytes'] - $this->transferStats['bytes_transferred'];
                $this->transferStats['eta_seconds'] = $remaining / $this->transferStats['transfer_rate'];
            }
        }
        
        $this->transferStats['last_progress_time'] = $currentTime;
        
        // Call progress callback if provided
        if ($this->progressCallback && is_callable($this->progressCallback)) {
            // Modern callable syntax
            ($this->progressCallback)($this->transferStats);
        }
    }
    
    /**
     * Complete the transfer and return final statistics
     */
    private function completeTransfer() {
        $this->transferStats['state'] = self::STATE_COMPLETED;
        $this->updateProgress();
        
        $elapsed = $this->transferStats['last_progress_time'] - $this->transferStats['start_time'];
        $avgRate = $elapsed > 0 ? $this->transferStats['bytes_transferred'] / $elapsed : 0;
        
        mftpLog(LOG_DEBUG, sprintf(
            "StreamingTransferManager: Transfer completed. %s bytes in %.2f seconds (avg: %s/s)",
            number_format($this->transferStats['bytes_transferred']),
            $elapsed,
            $this->formatBytes($avgRate)
        ));
        
        return [
            'success' => true,
            'bytes_transferred' => $this->transferStats['bytes_transferred'],
            'elapsed_time' => $elapsed,
            'average_rate' => $avgRate,
            'peak_rate' => $this->transferStats['transfer_rate']
        ];
    }
    
    /**
     * Cancel an ongoing transfer
     */
    public function cancelTransfer() {
        if ($this->transferStats['state'] === self::STATE_UPLOADING || 
            $this->transferStats['state'] === self::STATE_DOWNLOADING) {
            $this->transferStats['state'] = self::STATE_CANCELLED;
            mftpLog(LOG_INFO, "StreamingTransferManager: Transfer cancelled by user");
        }
    }
    
    /**
     * Get current transfer statistics
     */
    public function getTransferStats() {
        return $this->transferStats;
    }
    
    /**
     * Get transfer progress as percentage
     */
    public function getProgressPercentage() {
        if ($this->transferStats['total_bytes'] <= 0) {
            return 0;
        }
        
        return min(100, ($this->transferStats['bytes_transferred'] / $this->transferStats['total_bytes']) * 100);
    }
    
    /**
     * Check if a transfer is currently active
     */
    public function isTransferActive() {
        return in_array($this->transferStats['state'], [self::STATE_UPLOADING, self::STATE_DOWNLOADING]);
    }
    
    /**
     * Format bytes into human readable format
     */
    private function formatBytes($bytes) {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $unitIndex = 0;
        
        while ($bytes >= 1024 && $unitIndex < count($units) - 1) {
            $bytes /= 1024;
            $unitIndex++;
        }
        
        return round($bytes, 2) . ' ' . $units[$unitIndex];
    }
    
    /**
     * Create a progress callback that outputs JSON for AJAX requests
     */
    public static function createJsonProgressCallback() {
        return function($stats) {
            $progress = [
                'state' => $stats['state'],
                'progress' => 0,
                'bytes_transferred' => $stats['bytes_transferred'],
                'total_bytes' => $stats['total_bytes'],
                'transfer_rate' => $stats['transfer_rate'],
                'eta_seconds' => $stats['eta_seconds']
            ];
            
            if ($stats['total_bytes'] > 0) {
                $progress['progress'] = min(100, ($stats['bytes_transferred'] / $stats['total_bytes']) * 100);
            }
            
            // Output progress for AJAX polling
            if (php_sapi_name() !== 'cli') {
                echo json_encode($progress) . "\n";
                flush();
            }
        };
    }
    
    /**
     * Create an optimized file copy using streaming
     */
    public function optimizedFileCopy($sourcePath, $destPath) {
        if (!is_readable($sourcePath)) {
            throw new InvalidArgumentException("Source file is not readable: $sourcePath");
        }
        
        $fileSize = filesize($sourcePath);
        $bufferSize = $this->performanceOptimizer->getOptimalBufferSize($fileSize);
        
        $this->initializeTransfer($fileSize, 'copying');
        
        $sourceHandle = fopen($sourcePath, 'rb');
        $destHandle = fopen($destPath, 'wb');
        
        if (!$sourceHandle || !$destHandle) {
            throw new RuntimeException("Failed to open files for copying");
        }
        
        try {
            $this->performStreamingTransfer($sourceHandle, $destHandle, $bufferSize, 2, 'copy');
        } finally {
            if ($sourceHandle) fclose($sourceHandle);
            if ($destHandle) fclose($destHandle);
        }
        
        return $this->completeTransfer();
    }
} 