<?php

require_once(dirname(__FILE__) . '/../constants.php');
require_once(dirname(__FILE__) . '/../lib/helpers.php');

/**
 * Performance optimization class for file transfer operations
 * Dynamically calculates optimal buffer sizes and manages streaming operations
 */
class PerformanceOptimizer {
    
    // Buffer size constants
    const MIN_BUFFER_SIZE = 8192;          // 8KB minimum
    const MAX_BUFFER_SIZE = 16777216;      // 16MB maximum  
    const DEFAULT_BUFFER_SIZE = 262144;    // 256KB default
    const MEMORY_SAFETY_FACTOR = 0.25;    // Use 25% of available memory max
    
    // File size thresholds for different strategies
    const SMALL_FILE_THRESHOLD = 1048576;     // 1MB
    const LARGE_FILE_THRESHOLD = 104857600;   // 100MB
    const HUGE_FILE_THRESHOLD = 1073741824;   // 1GB
    
    // Connection type multipliers
    const FTP_MULTIPLIER = 1.0;
    const SFTP_MULTIPLIER = 0.8;    // SFTP has more overhead
    const SSL_MULTIPLIER = 0.9;     // SSL adds overhead
    
    private static $instance = null;
    private $systemMemory;
    private $availableMemory;
    private $optimalBufferCache = [];
    
    private function __construct() {
        $this->initializeSystemInfo();
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Initialize system memory information
     */
    private function initializeSystemInfo() {
        // Get PHP memory limit
        $memoryLimit = ini_get('memory_limit');
        if ($memoryLimit == -1) {
            $this->systemMemory = 1073741824; // Default to 1GB if unlimited
        } else {
            $this->systemMemory = $this->parseMemorySize($memoryLimit);
        }
        
        // Calculate available memory (current usage vs limit)
        $currentUsage = memory_get_usage(true);
        $this->availableMemory = max(
            self::MIN_BUFFER_SIZE * 2, 
            ($this->systemMemory - $currentUsage) * self::MEMORY_SAFETY_FACTOR
        );
    }
    
    /**
     * Parse memory size string to bytes
     */
    private function parseMemorySize($size) {
        $size = trim($size);
        $last = strtolower($size[strlen($size)-1]);
        $size = (int) $size;
        
        switch($last) {
            case 'g': $size *= 1024;
            case 'm': $size *= 1024;
            case 'k': $size *= 1024;
        }
        
        return $size;
    }
    
    /**
     * Calculate optimal buffer size for a transfer operation
     */
    public function getOptimalBufferSize($fileSize = null, $connectionType = 'ftp', $isSSL = false) {
        $cacheKey = sprintf('%s_%s_%s_%s', 
            $fileSize ?: 'unknown', 
            $connectionType, 
            $isSSL ? 'ssl' : 'plain',
            $this->availableMemory
        );
        
        if (isset($this->optimalBufferCache[$cacheKey])) {
            return $this->optimalBufferCache[$cacheKey];
        }
        
        $baseBufferSize = $this->calculateBaseBufferSize($fileSize);
        $adjustedSize = $this->applyConnectionMultipliers($baseBufferSize, $connectionType, $isSSL);
        $finalSize = $this->ensureMemoryConstraints($adjustedSize);
        
        $this->optimalBufferCache[$cacheKey] = $finalSize;
        return $finalSize;
    }
    
    /**
     * Calculate base buffer size based on file size
     */
    private function calculateBaseBufferSize($fileSize) {
        if ($fileSize === null) {
            return self::DEFAULT_BUFFER_SIZE;
        }
        
        if ($fileSize <= self::SMALL_FILE_THRESHOLD) {
            // For small files, use smaller buffers to reduce memory overhead
            return min(self::DEFAULT_BUFFER_SIZE, max(self::MIN_BUFFER_SIZE, $fileSize / 4));
        } elseif ($fileSize <= self::LARGE_FILE_THRESHOLD) {
            // For medium files, use progressive scaling
            $scaleFactor = $fileSize / self::LARGE_FILE_THRESHOLD;
            return (int) (self::DEFAULT_BUFFER_SIZE + ($scaleFactor * 524288)); // Up to 768KB
        } elseif ($fileSize <= self::HUGE_FILE_THRESHOLD) {
            // For large files, use larger buffers for efficiency
            return 1048576; // 1MB
        } else {
            // For huge files, use maximum buffer size
            return 2097152; // 2MB
        }
    }
    
    /**
     * Apply connection type specific multipliers
     */
    private function applyConnectionMultipliers($bufferSize, $connectionType, $isSSL) {
        $multiplier = 1.0;
        
        switch (strtolower($connectionType)) {
            case 'sftp':
                $multiplier *= self::SFTP_MULTIPLIER;
                break;
            case 'ftp':
                $multiplier *= self::FTP_MULTIPLIER;
                break;
        }
        
        if ($isSSL) {
            $multiplier *= self::SSL_MULTIPLIER;
        }
        
        return (int) ($bufferSize * $multiplier);
    }
    
    /**
     * Ensure buffer size fits within memory constraints
     */
    private function ensureMemoryConstraints($bufferSize) {
        // Don't exceed available memory
        $maxAllowed = min(self::MAX_BUFFER_SIZE, $this->availableMemory / 4);
        $finalSize = min($bufferSize, $maxAllowed);
        
        // Ensure minimum size
        return max(self::MIN_BUFFER_SIZE, $finalSize);
    }
    
    /**
     * Get optimal chunk size for chunked uploads
     */
    public function getOptimalChunkSize($totalFileSize = null) {
        $baseSize = $this->getOptimalBufferSize($totalFileSize);
        
        // For chunked uploads, use larger chunks to reduce HTTP overhead
        $chunkSize = $baseSize * 8; // 8x buffer size for chunks
        
        // Respect system configuration
        if (defined('MFTP_CHUNK_UPLOAD_SIZE') && MFTP_CHUNK_UPLOAD_SIZE !== 'default') {
            $configuredSize = $this->parseMemorySize(MFTP_CHUNK_UPLOAD_SIZE);
            return min($chunkSize, $configuredSize);
        }
        
        return min($chunkSize, 8388608); // Max 8MB chunks
    }
    
    /**
     * Determine if streaming should be used for a file transfer
     */
    public function shouldUseStreaming($fileSize) {
        if ($fileSize === null) {
            return true; // Default to streaming for unknown sizes
        }
        
        // Use streaming for files larger than 10% of available memory
        return $fileSize > ($this->availableMemory * 0.1);
    }
    
    /**
     * Get progress callback interval based on file size
     */
    public function getProgressCallbackInterval($fileSize) {
        if ($fileSize === null || $fileSize < self::SMALL_FILE_THRESHOLD) {
            return 2; // 2 seconds for small files
        } elseif ($fileSize < self::LARGE_FILE_THRESHOLD) {
            return 5; // 5 seconds for medium files
        } else {
            return 10; // 10 seconds for large files
        }
    }
    
    /**
     * Calculate optimal number of concurrent transfers
     */
    public function getOptimalConcurrency($averageFileSize = null) {
        $baseBufferSize = $this->getOptimalBufferSize($averageFileSize);
        $memoryPerTransfer = $baseBufferSize * 3; // Buffer + overhead
        
        $maxConcurrent = max(1, (int) ($this->availableMemory / $memoryPerTransfer));
        
        // Respect system limits
        if (defined('CHUNK_MAX_SIMULTANEOUS_UPLOADS')) {
            return min($maxConcurrent, CHUNK_MAX_SIMULTANEOUS_UPLOADS);
        }
        
        return min($maxConcurrent, 5); // Reasonable default maximum
    }
    
    /**
     * Get system performance metrics
     */
    public function getPerformanceMetrics() {
        return [
            'system_memory' => $this->systemMemory,
            'available_memory' => $this->availableMemory,
            'memory_usage' => memory_get_usage(true),
            'memory_peak' => memory_get_peak_usage(true),
            'buffer_cache_size' => count($this->optimalBufferCache),
            'recommended_buffer' => $this->getOptimalBufferSize(),
            'recommended_chunk' => $this->getOptimalChunkSize(),
            'max_concurrency' => $this->getOptimalConcurrency()
        ];
    }
    
    /**
     * Clear buffer cache (useful for long-running processes)
     */
    public function clearCache() {
        $this->optimalBufferCache = [];
        $this->initializeSystemInfo(); // Refresh system info
    }
} 