<?php

require_once(dirname(__FILE__) . '/../constants.php');
require_once(dirname(__FILE__) . '/ConnectionPool.php');
require_once(dirname(__FILE__) . '/CacheManager.php');
require_once(dirname(__FILE__) . '/PerformanceOptimizer.php');
require_once(dirname(__FILE__) . '/StreamingTransferManager.php');
require_once(dirname(__FILE__) . '/TransferResumeManager.php');
require_once(dirname(__FILE__) . '/../lib/helpers.php');
require_once(dirname(__FILE__) . '/../lib/logging.php');

/**
 * Central performance management system
 * Orchestrates all performance optimization components
 */
class PerformanceManager {
    
    private static $instance = null;
    private $connectionPool;
    private $cacheManager;
    private $performanceOptimizer;
    private $transferResumeManager;
    private $config;
    private $metrics;
    
    // Performance modes
    const MODE_BALANCED = 'balanced';
    const MODE_SPEED = 'speed';
    const MODE_MEMORY = 'memory';
    const MODE_COMPATIBILITY = 'compatibility';
    
    private function __construct() {
        $this->initializeComponents();
        $this->initializeConfiguration();
        $this->initializeMetrics();
        
        mftpLog(LOG_INFO, "PerformanceManager: Initialized with mode: {$this->config['mode']}");
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Initialize performance components
     */
    private function initializeComponents() {
        $this->connectionPool = ConnectionPool::getInstance();
        $this->cacheManager = CacheManager::getInstance();
        $this->performanceOptimizer = PerformanceOptimizer::getInstance();
        $this->transferResumeManager = TransferResumeManager::getInstance();
    }
    
    /**
     * Initialize performance configuration
     */
    private function initializeConfiguration() {
        $systemMetrics = $this->performanceOptimizer->getPerformanceMetrics();
        
        // Auto-detect optimal mode based on system capabilities
        $autoMode = $this->detectOptimalMode($systemMetrics);
        
        $this->config = [
            'mode' => defined('MFTP_PERFORMANCE_MODE') ? MFTP_PERFORMANCE_MODE : $autoMode,
            'auto_optimization' => true,
            'background_tasks' => true,
            'metrics_collection' => true,
            'cache_preloading' => true,
            'connection_pooling' => true,
            'streaming_transfers' => true,
            'resume_capability' => true,
            'compression' => function_exists('gzcompress'),
            'concurrent_operations' => $systemMetrics['max_concurrency']
        ];
        
        // Apply mode-specific configurations
        $this->applyModeConfiguration();
    }
    
    /**
     * Detect optimal performance mode based on system capabilities
     */
    private function detectOptimalMode($metrics) {
        $memoryMB = $metrics['available_memory'] / 1024 / 1024;
        $concurrency = $metrics['max_concurrency'];
        
        if ($memoryMB > 512 && $concurrency >= 4) {
            return self::MODE_SPEED;
        } elseif ($memoryMB < 128) {
            return self::MODE_MEMORY;
        } else {
            return self::MODE_BALANCED;
        }
    }
    
    /**
     * Apply mode-specific configuration
     */
    private function applyModeConfiguration() {
        switch ($this->config['mode']) {
            case self::MODE_SPEED:
                $this->config['concurrent_operations'] = min(8, $this->config['concurrent_operations']);
                $this->config['cache_preloading'] = true;
                $this->config['streaming_transfers'] = true;
                break;
                
            case self::MODE_MEMORY:
                $this->config['concurrent_operations'] = min(2, $this->config['concurrent_operations']);
                $this->config['cache_preloading'] = false;
                $this->config['compression'] = true;
                break;
                
            case self::MODE_BALANCED:
                $this->config['concurrent_operations'] = min(4, $this->config['concurrent_operations']);
                break;
                
            case self::MODE_COMPATIBILITY:
                $this->config['streaming_transfers'] = false;
                $this->config['concurrent_operations'] = 1;
                $this->config['auto_optimization'] = false;
                break;
        }
    }
    
    /**
     * Initialize performance metrics tracking
     */
    private function initializeMetrics() {
        $this->metrics = [
            'operations_total' => 0,
            'operations_cached' => 0,
            'operations_pooled' => 0,
            'operations_streamed' => 0,
            'operations_resumed' => 0,
            'total_time_saved' => 0,
            'cache_hit_rate' => 0,
            'pool_utilization' => 0,
            'average_operation_time' => 0,
            'last_optimization' => time()
        ];
    }
    
    /**
     * Execute an optimized file operation
     */
    public function executeOperation($operationType, $connectionType, $configuration, $operationData, $callback = null) {
        $startTime = microtime(true);
        $this->metrics['operations_total']++;
        
        try {
            // Pre-operation optimizations
            $this->preOperationOptimization($operationType, $connectionType, $operationData);
            
            // Execute operation using optimal strategy
            $result = $this->executeOptimizedOperation($operationType, $connectionType, $configuration, $operationData, $callback);
            
            // Post-operation optimizations
            $this->postOperationOptimization($operationType, $result, microtime(true) - $startTime);
            
            return $result;
            
        } catch (Exception $e) {
            mftpLog(LOG_ERROR, "PerformanceManager: Operation failed: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Pre-operation optimization checks
     */
    private function preOperationOptimization($operationType, $connectionType, $operationData) {
        // Check cache for read operations
        if (in_array($operationType, ['listDirectory', 'getFileInfo', 'getServerCapabilities'])) {
            $cached = $this->checkCache($operationType, $connectionType, $operationData);
            if ($cached) {
                $this->metrics['operations_cached']++;
                return $cached;
            }
        }
        
        // Optimize connection pool
        if ($this->config['auto_optimization']) {
            $this->connectionPool->optimizePoolConfiguration();
        }
        
        // Preload related data if beneficial
        $this->preloadRelatedData($operationType, $operationData);
    }
    
    /**
     * Execute operation using optimal strategy
     */
    private function executeOptimizedOperation($operationType, $connectionType, $configuration, $operationData, $callback) {
        switch ($operationType) {
            case 'upload':
                return $this->executeOptimizedUpload($connectionType, $configuration, $operationData, $callback);
                
            case 'download':
                return $this->executeOptimizedDownload($connectionType, $configuration, $operationData, $callback);
                
            case 'listDirectory':
                return $this->executeOptimizedDirectoryListing($connectionType, $configuration, $operationData);
                
            case 'multipleOperations':
                return $this->executeOptimizedBulkOperations($connectionType, $configuration, $operationData);
                
            default:
                return $this->executeStandardOperation($operationType, $connectionType, $configuration, $operationData);
        }
    }
    
    /**
     * Execute optimized upload
     */
    private function executeOptimizedUpload($connectionType, $configuration, $operationData, $callback) {
        $localPath = $operationData['localPath'];
        $remotePath = $operationData['remotePath'];
        $fileSize = filesize($localPath);
        
        // Check for resume capability
        if ($this->config['resume_capability'] && $fileSize > 10485760) { // > 10MB
            $resumeInfo = $this->transferResumeManager->getTransferInfo($localPath, $remotePath);
            if ($resumeInfo && $resumeInfo['status'] === TransferResumeManager::STATUS_FAILED) {
                mftpLog(LOG_INFO, "PerformanceManager: Resuming interrupted upload");
                $this->metrics['operations_resumed']++;
                return $this->transferResumeManager->resumeTransfer($resumeInfo['transfer_id']);
            }
        }
        
        // Use connection pool for upload
        return $this->connectionPool->executeWithConnection(
            $connectionType, 
            $configuration, 
            function($connection) use ($operationData, $fileSize, $connectionType, $callback) {
                $this->metrics['operations_pooled']++;
                
                // Use streaming for large files
                if ($this->config['streaming_transfers'] && $fileSize > 1048576) { // > 1MB
                    $streamingManager = new StreamingTransferManager($callback);
                    $this->metrics['operations_streamed']++;
                    
                    // Create transfer operation
                    $transferOp = TransferOperationFactory::getTransferOperation($connectionType, $operationData);
                    return $connection->uploadFile($transferOp);
                } else {
                    // Standard upload for small files
                    $transferOp = TransferOperationFactory::getTransferOperation($connectionType, $operationData);
                    return $connection->uploadFile($transferOp);
                }
            }
        );
    }
    
    /**
     * Execute optimized download
     */
    private function executeOptimizedDownload($connectionType, $configuration, $operationData, $callback) {
        $remotePath = $operationData['remotePath'];
        $localPath = $operationData['localPath'];
        
        // Check cache for file metadata first
        $metadata = $this->cacheManager->getCachedFileMetadata($connectionType, $configuration['host'], $remotePath);
        $fileSize = $metadata['size'] ?? null;
        
        return $this->connectionPool->executeWithConnection(
            $connectionType,
            $configuration,
            function($connection) use ($operationData, $fileSize, $connectionType, $callback) {
                $this->metrics['operations_pooled']++;
                
                // Use streaming for large files
                if ($this->config['streaming_transfers'] && $fileSize && $fileSize > 1048576) {
                    $streamingManager = new StreamingTransferManager($callback);
                    $this->metrics['operations_streamed']++;
                    
                    return $streamingManager->streamDownload(
                        $connection->openRemoteFile($operationData['remotePath']),
                        $operationData['localPath'],
                        $fileSize,
                        $connectionType
                    );
                } else {
                    // Standard download
                    $transferOp = TransferOperationFactory::getTransferOperation($connectionType, $operationData);
                    return $connection->downloadFile($transferOp);
                }
            }
        );
    }
    
    /**
     * Execute optimized directory listing
     */
    private function executeOptimizedDirectoryListing($connectionType, $configuration, $operationData) {
        $path = $operationData['path'];
        $showHidden = $operationData['showHidden'] ?? false;
        $host = $configuration['host'];
        
        // Check cache first
        $cached = $this->cacheManager->getCachedDirectoryListing($connectionType, $host, $path, $showHidden);
        if ($cached) {
            $this->metrics['operations_cached']++;
            return $cached;
        }
        
        // Execute with connection pool
        $result = $this->connectionPool->executeWithConnection(
            $connectionType,
            $configuration,
            function($connection) use ($operationData) {
                $this->metrics['operations_pooled']++;
                return $connection->listDirectory($operationData['path'], $operationData['showHidden'] ?? false);
            }
        );
        
        // Cache the result
        $this->cacheManager->cacheDirectoryListing($connectionType, $host, $path, $result, $showHidden);
        
        // Cache file metadata for each file
        foreach ($result as $fileInfo) {
            if (isset($fileInfo['name']) && !in_array($fileInfo['name'], ['.', '..'])) {
                $filePath = $path . '/' . $fileInfo['name'];
                $this->cacheManager->cacheFileMetadata($connectionType, $host, $filePath, $fileInfo);
            }
        }
        
        return $result;
    }
    
    /**
     * Execute optimized bulk operations
     */
    private function executeOptimizedBulkOperations($connectionType, $configuration, $operationData) {
        $operations = $operationData['operations'];
        
        if (count($operations) > 1 && $this->config['concurrent_operations'] > 1) {
            // Use concurrent execution for multiple operations
            return $this->connectionPool->executeConcurrent($operations);
        } else {
            // Execute sequentially
            $results = [];
            foreach ($operations as $operation) {
                $results[] = $this->executeOperation(
                    $operation['type'],
                    $connectionType,
                    $configuration,
                    $operation['data']
                );
            }
            return $results;
        }
    }
    
    /**
     * Execute standard operation (fallback)
     */
    private function executeStandardOperation($operationType, $connectionType, $configuration, $operationData) {
        return $this->connectionPool->executeWithConnection(
            $connectionType,
            $configuration,
            function($connection) use ($operationType, $operationData) {
                $this->metrics['operations_pooled']++;
                
                switch ($operationType) {
                    case 'deleteFile':
                        return $connection->deleteFile($operationData['remotePath']);
                    case 'deleteDirectory':
                        return $connection->deleteDirectory($operationData['remotePath']);
                    case 'makeDirectory':
                        return $connection->makeDirectory($operationData['remotePath']);
                    case 'rename':
                        return $connection->rename($operationData['source'], $operationData['destination']);
                    case 'changePermissions':
                        return $connection->changePermissions($operationData['remotePath'], $operationData['mode']);
                    default:
                        throw new InvalidArgumentException("Unknown operation type: $operationType");
                }
            }
        );
    }
    
    /**
     * Post-operation optimization
     */
    private function postOperationOptimization($operationType, $result, $executionTime) {
        // Update metrics
        $this->updateMetrics($operationType, $executionTime);
        
        // Invalidate related caches if needed
        if (in_array($operationType, ['upload', 'deleteFile', 'deleteDirectory', 'makeDirectory', 'rename'])) {
            // These operations change directory structure, invalidate caches
            $this->invalidateRelatedCaches($operationType, $result);
        }
        
        // Background optimization tasks
        if ($this->config['background_tasks'] && rand(1, 100) === 1) {
            $this->performBackgroundOptimization();
        }
    }
    
    /**
     * Check cache for operation result
     */
    private function checkCache($operationType, $connectionType, $operationData) {
        switch ($operationType) {
            case 'listDirectory':
                return $this->cacheManager->getCachedDirectoryListing(
                    $connectionType,
                    $operationData['host'],
                    $operationData['path'],
                    $operationData['showHidden'] ?? false
                );
                
            case 'getFileInfo':
                return $this->cacheManager->getCachedFileMetadata(
                    $connectionType,
                    $operationData['host'],
                    $operationData['path']
                );
                
            case 'getServerCapabilities':
                return $this->cacheManager->getCachedServerCapabilities(
                    $connectionType,
                    $operationData['host'],
                    $operationData['port']
                );
        }
        
        return null;
    }
    
    /**
     * Preload related data
     */
    private function preloadRelatedData($operationType, $operationData) {
        if (!$this->config['cache_preloading']) {
            return;
        }
        
        // Example: When listing a directory, preload parent directory info
        if ($operationType === 'listDirectory') {
            $parentPath = dirname($operationData['path']);
            if ($parentPath !== $operationData['path']) {
                // Could preload parent directory listing in background
            }
        }
    }
    
    /**
     * Update performance metrics
     */
    private function updateMetrics($operationType, $executionTime) {
        $this->metrics['average_operation_time'] = 
            ($this->metrics['average_operation_time'] * ($this->metrics['operations_total'] - 1) + $executionTime) 
            / $this->metrics['operations_total'];
            
        // Calculate cache hit rate
        if ($this->metrics['operations_total'] > 0) {
            $this->metrics['cache_hit_rate'] = 
                ($this->metrics['operations_cached'] / $this->metrics['operations_total']) * 100;
        }
        
        // Get pool statistics
        $poolStats = $this->connectionPool->getPoolStatistics();
        $this->metrics['pool_utilization'] = $poolStats['pool_utilization'];
    }
    
    /**
     * Invalidate related caches
     */
    private function invalidateRelatedCaches($operationType, $result) {
        // Implementation depends on operation type and result
        // This is a simplified version
        if (isset($result['directory'])) {
            $this->cacheManager->invalidateDirectoryCache(
                $result['connectionType'] ?? 'ftp',
                $result['host'] ?? 'localhost',
                $result['directory']
            );
        }
    }
    
    /**
     * Perform background optimization tasks
     */
    private function performBackgroundOptimization() {
        // Optimize connection pool
        $this->connectionPool->optimizePoolConfiguration();
        
        // Cleanup caches
        $cacheStats = $this->cacheManager->getStatistics();
        if ($cacheStats['memory_usage'] > 100 * 1024 * 1024) { // > 100MB
            // Trigger cache cleanup
        }
        
        // Update optimization timestamp
        $this->metrics['last_optimization'] = time();
        
        mftpLog(LOG_DEBUG, "PerformanceManager: Performed background optimization");
    }
    
    /**
     * Get comprehensive performance statistics
     */
    public function getPerformanceStatistics() {
        $connectionStats = $this->connectionPool->getPoolStatistics();
        $cacheStats = $this->cacheManager->getStatistics();
        $systemMetrics = $this->performanceOptimizer->getPerformanceMetrics();
        
        return [
            'mode' => $this->config['mode'],
            'operations' => $this->metrics,
            'connections' => $connectionStats,
            'cache' => $cacheStats,
            'system' => $systemMetrics,
            'configuration' => $this->config,
            'uptime' => time() - $this->metrics['last_optimization'],
            'efficiency_score' => $this->calculateEfficiencyScore()
        ];
    }
    
    /**
     * Calculate overall efficiency score
     */
    private function calculateEfficiencyScore() {
        $score = 100;
        
        // Deduct points for poor cache hit rate
        if ($this->metrics['cache_hit_rate'] < 50) {
            $score -= (50 - $this->metrics['cache_hit_rate']) * 0.5;
        }
        
        // Deduct points for poor pool utilization
        if ($this->metrics['pool_utilization'] < 30) {
            $score -= (30 - $this->metrics['pool_utilization']) * 0.3;
        }
        
        // Add points for using optimizations
        $optimizationBonus = 0;
        $optimizationBonus += ($this->metrics['operations_cached'] / max(1, $this->metrics['operations_total'])) * 20;
        $optimizationBonus += ($this->metrics['operations_pooled'] / max(1, $this->metrics['operations_total'])) * 15;
        $optimizationBonus += ($this->metrics['operations_streamed'] / max(1, $this->metrics['operations_total'])) * 10;
        
        return min(100, max(0, $score + $optimizationBonus));
    }
    
    /**
     * Tune performance based on usage patterns
     */
    public function autoTunePerformance() {
        $stats = $this->getPerformanceStatistics();
        
        // Adjust configuration based on performance
        if ($stats['efficiency_score'] < 70) {
            mftpLog(LOG_INFO, "PerformanceManager: Performance below threshold, auto-tuning...");
            
            // Increase cache sizes if memory allows
            if ($stats['system']['available_memory'] > 200 * 1024 * 1024) {
                $this->config['cache_preloading'] = true;
            }
            
            // Adjust concurrent operations based on performance
            if ($stats['operations']['average_operation_time'] > 5.0) {
                $this->config['concurrent_operations'] = max(1, $this->config['concurrent_operations'] - 1);
            } else {
                $this->config['concurrent_operations'] = min(8, $this->config['concurrent_operations'] + 1);
            }
        }
    }
    
    /**
     * Enable/disable specific optimizations
     */
    public function configureOptimization($optimization, $enabled) {
        switch ($optimization) {
            case 'connection_pooling':
                $this->config['connection_pooling'] = $enabled;
                break;
            case 'caching':
                if (!$enabled) {
                    $this->cacheManager->clearAll();
                }
                break;
            case 'streaming':
                $this->config['streaming_transfers'] = $enabled;
                break;
            case 'resume':
                $this->config['resume_capability'] = $enabled;
                break;
            case 'auto_optimization':
                $this->config['auto_optimization'] = $enabled;
                break;
        }
        
        mftpLog(LOG_INFO, "PerformanceManager: Set $optimization to " . ($enabled ? 'enabled' : 'disabled'));
    }
} 