<?php

    class PathOperations {
        
        // Security constants
        const MAX_PATH_LENGTH = 4096;
        const MAX_FILENAME_LENGTH = 255;
        
        /**
         * Validate path for security issues
         * @param string $path The path to validate
         * @param bool $allowAbsolute Whether to allow absolute paths
         * @return bool True if path is safe
         * @throws InvalidArgumentException If path is unsafe
         */
        public static function validatePathSecurity($path, $allowAbsolute = false) {
            if (!is_string($path)) {
                throw new InvalidArgumentException("Path must be a string");
            }
            
            // Check path length
            if (strlen($path) > self::MAX_PATH_LENGTH) {
                throw new InvalidArgumentException("Path exceeds maximum length (" . self::MAX_PATH_LENGTH . " characters)");
            }
            
            // Check for null bytes (path traversal attempt)
            if (strpos($path, "\0") !== false) {
                throw new InvalidArgumentException("Path contains null bytes");
            }
            
            // Check for directory traversal patterns
            if (self::containsDirectoryTraversal($path)) {
                throw new InvalidArgumentException("Path contains directory traversal sequences");
            }
            
            // Check for absolute paths if not allowed
            if (!$allowAbsolute && self::isAbsolutePath($path)) {
                throw new InvalidArgumentException("Absolute paths not allowed");
            }
            
            // Check individual filename lengths
            $pathParts = explode('/', $path);
            foreach ($pathParts as $part) {
                if (strlen($part) > self::MAX_FILENAME_LENGTH) {
                    throw new InvalidArgumentException("Filename too long: " . $part);
                }
            }
            
            return true;
        }
        
        /**
         * Check if path contains directory traversal patterns
         */
        private static function containsDirectoryTraversal($path) {
            $dangerousPatterns = [
                '../', '..\\', 
                '..%2f', '..%2F', '..%5c', '..%5C',
                '%2e%2e%2f', '%2E%2E%2F', '%2e%2e%5c', '%2E%2E%5C',
                '....//....', // Double encoding
                urlencode('../'), urlencode('..\\')
            ];
            
            $lowerPath = strtolower($path);
            foreach ($dangerousPatterns as $pattern) {
                if (strpos($lowerPath, $pattern) !== false) {
                    return true;
                }
            }
            
            return false;
        }
        
        /**
         * Check if path is absolute
         */
        private static function isAbsolutePath($path) {
            return (substr($path, 0, 1) === '/' || 
                    preg_match('/^[a-zA-Z]:[\\\\\/]/', $path) || 
                    substr($path, 0, 2) === '\\\\'); // UNC paths
        }
        
        /**
         * Sanitize and normalize path with security checks
         */
        public static function secureNormalize($path, $allowAbsolute = false) {
            // First validate security
            self::validatePathSecurity($path, $allowAbsolute);
            
            // Then normalize
            return self::normalize($path);
        }
        public static function join() {
            $pathComponents = array();

            foreach (func_get_args() as $pathComponent) {
                if ($pathComponent !== '') {
                    if (substr($pathComponent, 0, 1) == '/')
                        $pathComponents = array();  // if we're back at the root then reset the array
                    $pathComponents[] = $pathComponent;
                }

            }

            return preg_replace('#/+#', '/', join('/', $pathComponents));
        }

        public static function normalize($path) {
            $pathComponents = array();
            $realPathComponentFound = false;  // ..s should be resolved only if they aren't leading the path
            $pathPrefix = substr($path, 0, 1) == '/' ? '/' : '';

            foreach (explode("/", $path) as $pathComponent) {
                if (strlen($pathComponent) == 0 || $pathComponent == '.')
                    continue;

                if ($pathComponent == '..' && $realPathComponentFound) {
                    unset($pathComponents[count($pathComponents) - 1]);
                    continue;
                }

                $pathComponents[] = $pathComponent;
                $realPathComponentFound = true;
            }

            return $pathPrefix . join("/", $pathComponents);
        }

        public static function directoriesMatch($dir1, $dir2) {
            return PathOperations::normalize($dir1) == PathOperations::normalize($dir2);
        }

        public static function remoteDirname($path) {
            // on windows machines $dirName will be \ for root files, we want it to be / for remote paths
            $dirName = dirname($path);
            return ($dirName == "\\") ? "/" : $dirName;
        }

        public static function directoriesInPath($directoryPath) {
            $directories = array();
            while ($directoryPath != "/" && $directoryPath != null && $directoryPath != "") {
                $directories[] = $directoryPath;
                $directoryPath = self::remoteDirname($directoryPath);
            }

            return $directories;
        }

        public static function ensureTrailingSlash($path) {
            if (strlen($path) == 0)
                return "/";

            if (substr($path, strlen($path) - 1, 1) != "/")
                $path .= "/";

            return $path;
        }

        public static function isParentPath($parent, $child) {
            $normalizedChild = self::ensureTrailingSlash(self::normalize($child));
            $normalizedParent = self::ensureTrailingSlash(self::normalize($parent));

            return substr($normalizedChild, 0, strlen($normalizedParent)) == $normalizedParent;
        }

        public static function getFirstPathComponent($path) {
            $pathWithoutLeadingSlashes = preg_replace("|^(/+)|", "", $path);

            $pathComponents = explode("/", $pathWithoutLeadingSlashes);
            return (count($pathComponents) == 0) ? "" : $pathComponents[0];
        }

        public static function stripTrailingSlash($path) {
            return substr($path, -1) === "/" ? substr($path, 0, -1) : $path;
        }

        public static function recursiveDelete($path) {
            if (is_dir($path) === true) {
                $files = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($path), RecursiveIteratorIterator::CHILD_FIRST);

                foreach ($files as $file) {
                    if (in_array($file->getBasename(), array('.', '..')) !== TRUE) {
                        $removeSuccess = true;

                        if ($file->isDir() === TRUE) {
                            $removeSuccess = rmdir($file->getPathName());
                        } else if (($file->isFile() === TRUE) || ($file->isLink() === TRUE)) {
                            $removeSuccess = unlink($file->getPathname());
                        }

                        if(!$removeSuccess)
                            return false;
                    }
                }

                return rmdir($path);
            } else if ((is_file($path) === TRUE) || (is_link($path) === TRUE)) {
                return unlink($path);
            }

            return false;
        }

        public static function pathDepthCompare($path1, $path2) {
            // naive function for ordering paths based on their depth; more slashes == deeper
            $slashCount1 = substr_count($path1, "/");
            $slashCount2 = substr_count($path2, "/");

            if ($slashCount1 == $slashCount2)
                return strcasecmp($path1, $path2); // if same depth, just alphabetical sort. doesn't really matter

            return ($slashCount1 > $slashCount2) ? -1 : 1;
        }
    }

    function pathDepthCompare($path1, $path2) {
        // Wrap the static method above, as PHP doesn't understand passing a static method to usort
        return PathOperations::pathDepthCompare($path1, $path2);
    }