<?php

require_once(dirname(__FILE__) . '/../constants.php');
require_once(dirname(__FILE__) . '/PerformanceOptimizer.php');
require_once(dirname(__FILE__) . '/StreamingTransferManager.php');
require_once(dirname(__FILE__) . '/../lib/helpers.php');
require_once(dirname(__FILE__) . '/../lib/logging.php');
require_once(dirname(__FILE__) . '/connection/ArchiveExtractor.php');

use wapmorgan\UnifiedArchive\UnifiedArchive;

/**
 * Memory-efficient archive processor for large file archives
 * Provides streaming extraction and creation with optimized memory usage
 */
class OptimizedArchiveProcessor {
    
    private $performanceOptimizer;
    private $transferManager;
    private $tempDirectory;
    private $maxMemoryUsage;
    private $currentMemoryUsage;
    
    // Archive processing modes
    const MODE_MEMORY_EFFICIENT = 'memory_efficient';
    const MODE_SPEED_OPTIMIZED = 'speed_optimized';
    const MODE_BALANCED = 'balanced';
    
    // Extraction strategies
    const STRATEGY_STREAM = 'stream';
    const STRATEGY_BATCH = 'batch';
    const STRATEGY_SINGLE = 'single';
    
    public function __construct($tempDirectory = null) {
        $this->performanceOptimizer = PerformanceOptimizer::getInstance();
        $this->transferManager = new StreamingTransferManager();
        $this->tempDirectory = $tempDirectory ?: sys_get_temp_dir();
        
        $metrics = $this->performanceOptimizer->getPerformanceMetrics();
        $this->maxMemoryUsage = $metrics['available_memory'] * 0.6; // Use 60% of available memory
        $this->currentMemoryUsage = 0;
    }
    
    /**
     * Extract archive with memory-efficient streaming
     */
    public function extractArchiveStreaming($archivePath, $extractPath, $connection = null, $progressCallback = null) {
        if (!file_exists($archivePath)) {
            throw new InvalidArgumentException("Archive file does not exist: $archivePath");
        }
        
        $archive = UnifiedArchive::open($archivePath);
        if (!$archive) {
            throw new RuntimeException("Failed to open archive: $archivePath");
        }
        
        $fileList = $archive->getFileNames();
        $totalFiles = count($fileList);
        $extractedFiles = 0;
        $strategy = $this->determineExtractionStrategy($archive, $totalFiles);
        
        mftpLog(LOG_INFO, "OptimizedArchiveProcessor: Extracting $totalFiles files using $strategy strategy");
        
        try {
            switch ($strategy) {
                case self::STRATEGY_STREAM:
                    $result = $this->extractUsingStreamStrategy($archive, $fileList, $extractPath, $connection, $progressCallback);
                    break;
                    
                case self::STRATEGY_BATCH:
                    $result = $this->extractUsingBatchStrategy($archive, $fileList, $extractPath, $connection, $progressCallback);
                    break;
                    
                case self::STRATEGY_SINGLE:
                    $result = $this->extractUsingSingleStrategy($archive, $fileList, $extractPath, $connection, $progressCallback);
                    break;
                    
                default:
                    throw new RuntimeException("Unknown extraction strategy: $strategy");
            }
            
            return $result;
            
        } finally {
            $archive->close();
            $this->cleanupTempFiles();
        }
    }
    
    /**
     * Create archive with memory-efficient streaming
     */
    public function createArchiveStreaming($files, $archivePath, $compressionLevel = null, $progressCallback = null) {
        if (empty($files)) {
            throw new InvalidArgumentException("No files provided for archive creation");
        }
        
        $totalSize = $this->calculateTotalSize($files);
        $strategy = $this->determineCreationStrategy($totalSize, count($files));
        
        mftpLog(LOG_INFO, "OptimizedArchiveProcessor: Creating archive with " . count($files) . " files using $strategy strategy");
        
        switch ($strategy) {
            case self::STRATEGY_STREAM:
                return $this->createUsingStreamStrategy($files, $archivePath, $compressionLevel, $progressCallback);
                
            case self::STRATEGY_BATCH:
                return $this->createUsingBatchStrategy($files, $archivePath, $compressionLevel, $progressCallback);
                
            case self::STRATEGY_SINGLE:
                return $this->createUsingSingleStrategy($files, $archivePath, $compressionLevel, $progressCallback);
                
            default:
                throw new RuntimeException("Unknown creation strategy: $strategy");
        }
    }
    
    /**
     * Determine optimal extraction strategy based on archive characteristics
     */
    private function determineExtractionStrategy($archive, $fileCount) {
        $metrics = $this->performanceOptimizer->getPerformanceMetrics();
        
        // Check if archive supports streaming
        $supportsStreaming = method_exists($archive, 'getFileStream');
        
        if ($fileCount > 1000 || $metrics['available_memory'] < 67108864) { // < 64MB
            return $supportsStreaming ? self::STRATEGY_STREAM : self::STRATEGY_BATCH;
        } elseif ($fileCount > 100) {
            return self::STRATEGY_BATCH;
        } else {
            return self::STRATEGY_SINGLE;
        }
    }
    
    /**
     * Determine optimal creation strategy based on file characteristics
     */
    private function determineCreationStrategy($totalSize, $fileCount) {
        $metrics = $this->performanceOptimizer->getPerformanceMetrics();
        
        if ($totalSize > $metrics['available_memory'] || $fileCount > 1000) {
            return self::STRATEGY_STREAM;
        } elseif ($totalSize > $metrics['available_memory'] * 0.5 || $fileCount > 100) {
            return self::STRATEGY_BATCH;
        } else {
            return self::STRATEGY_SINGLE;
        }
    }
    
    /**
     * Extract using streaming strategy (one file at a time)
     */
    private function extractUsingStreamStrategy($archive, $fileList, $extractPath, $connection, $progressCallback) {
        $extractedFiles = 0;
        $totalFiles = count($fileList);
        $errors = [];
        
        foreach ($fileList as $fileName) {
            try {
                $this->checkMemoryUsage();
                
                if ($this->extractSingleFileStreaming($archive, $fileName, $extractPath, $connection)) {
                    $extractedFiles++;
                }
                
                if ($progressCallback) {
                    $progressCallback($extractedFiles, $totalFiles, $fileName);
                }
                
                // Periodic cleanup to prevent memory leaks
                if ($extractedFiles % 50 === 0) {
                    $this->performMaintenanceCleanup();
                }
                
            } catch (Exception $e) {
                $errors[] = "Failed to extract $fileName: " . $e->getMessage();
                mftpLog(LOG_WARNING, "OptimizedArchiveProcessor: " . end($errors));
            }
        }
        
        return [
            'extracted_files' => $extractedFiles,
            'total_files' => $totalFiles,
            'errors' => $errors,
            'success_rate' => $totalFiles > 0 ? ($extractedFiles / $totalFiles) * 100 : 0
        ];
    }
    
    /**
     * Extract using batch strategy (multiple files at once)
     */
    private function extractUsingBatchStrategy($archive, $fileList, $extractPath, $connection, $progressCallback) {
        $batchSize = $this->calculateOptimalBatchSize(count($fileList));
        $extractedFiles = 0;
        $totalFiles = count($fileList);
        $errors = [];
        
        $batches = array_chunk($fileList, $batchSize);
        
        foreach ($batches as $batchIndex => $batch) {
            try {
                $this->checkMemoryUsage();
                
                $batchResult = $this->extractBatch($archive, $batch, $extractPath, $connection);
                $extractedFiles += $batchResult['extracted'];
                $errors = array_merge($errors, $batchResult['errors']);
                
                if ($progressCallback) {
                    $progressCallback($extractedFiles, $totalFiles, "Batch " . ($batchIndex + 1));
                }
                
                // Cleanup after each batch
                $this->performMaintenanceCleanup();
                
            } catch (Exception $e) {
                $batchError = "Failed to extract batch " . ($batchIndex + 1) . ": " . $e->getMessage();
                $errors[] = $batchError;
                mftpLog(LOG_ERROR, "OptimizedArchiveProcessor: $batchError");
            }
        }
        
        return [
            'extracted_files' => $extractedFiles,
            'total_files' => $totalFiles,
            'errors' => $errors,
            'success_rate' => $totalFiles > 0 ? ($extractedFiles / $totalFiles) * 100 : 0
        ];
    }
    
    /**
     * Extract using single strategy (all files at once)
     */
    private function extractUsingSingleStrategy($archive, $fileList, $extractPath, $connection, $progressCallback) {
        try {
            if ($connection) {
                // Extract to temp directory first, then upload
                $tempExtractPath = $this->createTempDirectory();
                $archive->extractFiles($tempExtractPath);
                
                $uploadedFiles = $this->uploadExtractedFiles($tempExtractPath, $extractPath, $connection, $progressCallback);
                
                return [
                    'extracted_files' => $uploadedFiles,
                    'total_files' => count($fileList),
                    'errors' => [],
                    'success_rate' => 100
                ];
            } else {
                // Direct extraction to local path
                $archive->extractFiles($extractPath);
                
                return [
                    'extracted_files' => count($fileList),
                    'total_files' => count($fileList),
                    'errors' => [],
                    'success_rate' => 100
                ];
            }
        } catch (Exception $e) {
            throw new RuntimeException("Single strategy extraction failed: " . $e->getMessage());
        }
    }
    
    /**
     * Extract a single file using streaming
     */
    private function extractSingleFileStreaming($archive, $fileName, $extractPath, $connection) {
        $fileData = $archive->getFileData($fileName);
        if (!$fileData) {
            return false;
        }
        
        $isDirectory = substr($fileName, -1) === '/';
        $fullPath = rtrim($extractPath, '/') . '/' . ltrim($fileName, '/');
        
        if ($isDirectory) {
            if ($connection) {
                $connection->makeDirectoryWithIntermediates($fullPath);
            } else {
                wp_mkdir_p(dirname($fullPath));
            }
            return true;
        }
        
        if ($connection) {
            // Stream to remote connection
            $tempFile = tempnam($this->tempDirectory, 'extract_');
            file_put_contents($tempFile, $fileData->getContent());
            
            try {
                // Create directory if needed
                $connection->makeDirectoryWithIntermediates(dirname($fullPath));
                
                // Upload using streaming manager
                $this->transferManager->streamUpload($tempFile, $fullPath, filesize($tempFile), 
                    strtolower($connection->getProtocolName()));
                
                return true;
            } finally {
                unlink($tempFile);
            }
        } else {
            // Stream to local file
            wp_mkdir_p(dirname($fullPath));
            return file_put_contents($fullPath, $fileData->getContent()) !== false;
        }
    }
    
    /**
     * Calculate optimal batch size based on available memory
     */
    private function calculateOptimalBatchSize($totalFiles) {
        $metrics = $this->performanceOptimizer->getPerformanceMetrics();
        $availableMemory = $metrics['available_memory'];
        
        // Estimate memory per file (conservative estimate)
        $memoryPerFile = 1048576; // 1MB per file average
        $maxFilesInMemory = max(1, intval($availableMemory / $memoryPerFile / 4)); // Use 25% of available memory
        
        return min($maxFilesInMemory, max(10, $totalFiles / 10)); // Between 10 and totalFiles/10
    }
    
    /**
     * Check current memory usage and cleanup if necessary
     */
    private function checkMemoryUsage() {
        $currentUsage = memory_get_usage(true);
        
        if ($currentUsage > $this->maxMemoryUsage) {
            mftpLog(LOG_WARNING, "OptimizedArchiveProcessor: Memory usage exceeded threshold, performing cleanup");
            $this->performMaintenanceCleanup();
            
            // Force garbage collection
            if (function_exists('gc_collect_cycles')) {
                gc_collect_cycles();
            }
        }
    }
    
    /**
     * Perform maintenance cleanup
     */
    private function performMaintenanceCleanup() {
        // Clear any cached data
        if (method_exists($this->performanceOptimizer, 'clearCache')) {
            $this->performanceOptimizer->clearCache();
        }
        
        // Clear stat cache
        clearstatcache();
        
        // Force garbage collection if available
        if (function_exists('gc_collect_cycles')) {
            gc_collect_cycles();
        }
    }
    
    /**
     * Calculate total size of files to be archived
     */
    private function calculateTotalSize($files) {
        $totalSize = 0;
        foreach ($files as $file) {
            if (is_string($file) && file_exists($file)) {
                $totalSize += filesize($file);
            } elseif (is_array($file) && isset($file['path']) && file_exists($file['path'])) {
                $totalSize += filesize($file['path']);
            }
        }
        return $totalSize;
    }
    
    /**
     * Create temporary directory for processing
     */
    private function createTempDirectory() {
        $tempDir = $this->tempDirectory . '/mftp_extract_' . uniqid();
        if (!mkdir($tempDir, 0755, true)) {
            throw new RuntimeException("Failed to create temporary directory: $tempDir");
        }
        return $tempDir;
    }
    
    /**
     * Clean up temporary files
     */
    private function cleanupTempFiles() {
        // Implementation would recursively clean temp directories
        // This is a placeholder for the cleanup logic
    }
    
    /**
     * Get archive processing statistics
     */
    public function getProcessingStats() {
        return [
            'memory_usage' => memory_get_usage(true),
            'memory_peak' => memory_get_peak_usage(true),
            'memory_limit' => $this->maxMemoryUsage,
            'temp_directory' => $this->tempDirectory,
            'performance_metrics' => $this->performanceOptimizer->getPerformanceMetrics()
        ];
    }
} 