<?php

require_once(dirname(__FILE__) . '/../constants.php');
require_once(dirname(__FILE__) . '/PerformanceOptimizer.php');
require_once(dirname(__FILE__) . '/connection/ConnectionFactory.php');
require_once(dirname(__FILE__) . '/configuration/ConfigurationFactory.php');
require_once(dirname(__FILE__) . '/../lib/helpers.php');
require_once(dirname(__FILE__) . '/../lib/logging.php');

/**
 * Connection pool manager for efficient handling of multiple FTP/SFTP connections
 * Provides connection reuse, load balancing, and automatic cleanup
 */
class ConnectionPool {
    
    private static $instance = null;
    private $pools = [];
    private $activeConnections = [];
    private $connectionStats = [];
    private $maxPoolSize;
    private $maxIdleTime;
    private $performanceOptimizer;
    
    // Connection states
    const STATE_IDLE = 'idle';
    const STATE_ACTIVE = 'active';
    const STATE_ERROR = 'error';
    const STATE_CLOSING = 'closing';
    
    // Pool configuration
    const DEFAULT_MAX_POOL_SIZE = 5;
    const DEFAULT_MAX_IDLE_TIME = 300; // 5 minutes
    const CONNECTION_TIMEOUT = 30;
    const MAX_RETRIES = 3;
    
    private function __construct() {
        $this->performanceOptimizer = PerformanceOptimizer::getInstance();
        $this->maxPoolSize = self::DEFAULT_MAX_POOL_SIZE;
        $this->maxIdleTime = self::DEFAULT_MAX_IDLE_TIME;
        
        // Adjust pool size based on system capabilities
        $maxConcurrency = $this->performanceOptimizer->getOptimalConcurrency();
        $this->maxPoolSize = min(self::DEFAULT_MAX_POOL_SIZE, $maxConcurrency);
        
        mftpLog(LOG_DEBUG, "ConnectionPool: Initialized with max pool size: {$this->maxPoolSize}");
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Get a connection from the pool or create a new one
     */
    public function getConnection($connectionType, $configuration) {
        $poolKey = $this->generatePoolKey($connectionType, $configuration);
        
        // Clean up expired connections first
        $this->cleanupExpiredConnections($poolKey);
        
        // Try to get an idle connection from the pool
        $connection = $this->getIdleConnection($poolKey);
        
        if ($connection) {
            $this->markConnectionActive($connection, $poolKey);
            mftpLog(LOG_DEBUG, "ConnectionPool: Reusing pooled connection for $connectionType");
            return $connection;
        }
        
        // Create new connection if pool has space
        if ($this->canCreateNewConnection($poolKey)) {
            return $this->createNewConnection($connectionType, $configuration, $poolKey);
        }
        
        // Wait for available connection or create one if timeout
        return $this->waitForAvailableConnection($connectionType, $configuration, $poolKey);
    }
    
    /**
     * Return a connection to the pool
     */
    public function returnConnection($connection, $connectionType, $configuration) {
        $poolKey = $this->generatePoolKey($connectionType, $configuration);
        
        if (!$this->isConnectionValid($connection)) {
            $this->removeConnection($connection, $poolKey);
            return;
        }
        
        $this->markConnectionIdle($connection, $poolKey);
        mftpLog(LOG_DEBUG, "ConnectionPool: Returned connection to pool for $connectionType");
    }
    
    /**
     * Execute a transfer operation using a pooled connection
     */
    public function executeWithConnection($connectionType, $configuration, $operation) {
        $connection = null;
        $startTime = microtime(true);
        
        try {
            $connection = $this->getConnection($connectionType, $configuration);
            
            $result = $operation($connection);
            
            $this->updateConnectionStats($connection, true, microtime(true) - $startTime);
            
            return $result;
            
        } catch (Exception $e) {
            if ($connection) {
                $this->updateConnectionStats($connection, false, microtime(true) - $startTime);
            }
            throw $e;
            
        } finally {
            if ($connection) {
                $this->returnConnection($connection, $connectionType, $configuration);
            }
        }
    }
    
    /**
     * Execute multiple operations concurrently with enhanced load balancing
     */
    public function executeConcurrent($operations) {
        $results = [];
        $errors = [];
        $maxConcurrency = min(count($operations), $this->maxPoolSize);
        
        mftpLog(LOG_INFO, "ConnectionPool: Executing " . count($operations) . " operations with max concurrency: $maxConcurrency");
        
        // Enhanced load balancing - sort operations by complexity
        $sortedOperations = $this->sortOperationsByComplexity($operations);
        $chunks = array_chunk($sortedOperations, $maxConcurrency);
        
        foreach ($chunks as $chunk) {
            $chunkResults = $this->executeConcurrentChunk($chunk);
            $results = array_merge($results, $chunkResults['results']);
            $errors = array_merge($errors, $chunkResults['errors']);
        }
        
        return [
            'results' => $results,
            'errors' => $errors,
            'success_count' => count($results),
            'error_count' => count($errors),
            'pool_statistics' => $this->getPoolStatistics()
        ];
    }

    /**
     * Sort operations by complexity for better load balancing
     */
    private function sortOperationsByComplexity($operations) {
        // Simple complexity estimation based on operation type
        $complexityMap = [
            'upload' => 3,
            'download' => 3,
            'list' => 1,
            'delete' => 1,
            'rename' => 1,
            'mkdir' => 1,
            'archive' => 5
        ];
        
        usort($operations, function($a, $b) use ($complexityMap) {
            $aComplexity = $complexityMap[$a['type'] ?? 'unknown'] ?? 2;
            $bComplexity = $complexityMap[$b['type'] ?? 'unknown'] ?? 2;
            return $bComplexity - $aComplexity; // Sort descending (most complex first)
        });
        
        return $operations;
    }

    /**
     * Enhanced connection health monitoring
     */
    public function performHealthCheck($poolKey = null) {
        $healthResults = [];
        
        $poolsToCheck = $poolKey ? [$poolKey => $this->pools[$poolKey]] : $this->pools;
        
        foreach ($poolsToCheck as $key => $pool) {
            $healthResults[$key] = [
                'total_connections' => count($pool),
                'active_connections' => 0,
                'idle_connections' => 0,
                'error_connections' => 0,
                'average_response_time' => 0,
                'success_rate' => 100
            ];
            
            $totalResponseTime = 0;
            $totalOperations = 0;
            
            foreach ($pool as $connectionId => $connectionData) {
                switch ($connectionData['state']) {
                    case self::STATE_ACTIVE:
                        $healthResults[$key]['active_connections']++;
                        break;
                    case self::STATE_IDLE:
                        $healthResults[$key]['idle_connections']++;
                        break;
                    case self::STATE_ERROR:
                        $healthResults[$key]['error_connections']++;
                        break;
                }
                
                // Calculate performance metrics
                if (isset($this->connectionStats[$connectionId])) {
                    $stats = $this->connectionStats[$connectionId];
                    $totalResponseTime += $stats['total_response_time'] ?? 0;
                    $totalOperations += $stats['total_operations'] ?? 0;
                    
                    if ($totalOperations > 0) {
                        $successRate = (($stats['successful_operations'] ?? 0) / $totalOperations) * 100;
                        $healthResults[$key]['success_rate'] = min($healthResults[$key]['success_rate'], $successRate);
                    }
                }
            }
            
            if ($totalOperations > 0) {
                $healthResults[$key]['average_response_time'] = $totalResponseTime / $totalOperations;
            }
        }
        
        return $healthResults;
    }

    /**
     * Get comprehensive pool statistics
     */
    public function getPoolStatistics() {
        $stats = [
            'total_pools' => count($this->pools),
            'total_connections' => 0,
            'active_connections' => 0,
            'idle_connections' => 0,
            'error_connections' => 0,
            'pool_utilization' => 0,
            'memory_usage' => memory_get_usage(true),
            'performance_metrics' => $this->performanceOptimizer->getPerformanceMetrics()
        ];
        
        foreach ($this->pools as $pool) {
            $stats['total_connections'] += count($pool);
            
            foreach ($pool as $connectionData) {
                switch ($connectionData['state']) {
                    case self::STATE_ACTIVE:
                        $stats['active_connections']++;
                        break;
                    case self::STATE_IDLE:
                        $stats['idle_connections']++;
                        break;
                    case self::STATE_ERROR:
                        $stats['error_connections']++;
                        break;
                }
            }
        }
        
        if ($stats['total_connections'] > 0) {
            $stats['pool_utilization'] = ($stats['active_connections'] / $stats['total_connections']) * 100;
        }
        
        return $stats;
    }

    /**
     * Optimize pool configuration based on usage patterns
     */
    public function optimizePoolConfiguration() {
        $stats = $this->getPoolStatistics();
        $metrics = $this->performanceOptimizer->getPerformanceMetrics();
        
        // Dynamic pool size adjustment
        if ($stats['pool_utilization'] > 90) {
            $newMaxSize = min($this->maxPoolSize + 1, $metrics['max_concurrency']);
            if ($newMaxSize > $this->maxPoolSize) {
                $this->maxPoolSize = $newMaxSize;
                mftpLog(LOG_INFO, "ConnectionPool: Increased pool size to {$this->maxPoolSize} due to high utilization");
            }
        } elseif ($stats['pool_utilization'] < 30 && $this->maxPoolSize > 2) {
            $this->maxPoolSize = max(2, $this->maxPoolSize - 1);
            mftpLog(LOG_INFO, "ConnectionPool: Decreased pool size to {$this->maxPoolSize} due to low utilization");
        }
        
        // Adjust idle timeout based on connection reuse patterns
        $avgResponseTime = $this->calculateAverageResponseTime();
        if ($avgResponseTime > 0) {
            // Longer timeouts for slower connections
            $newIdleTime = min(900, max(60, $avgResponseTime * 10));
            if (abs($newIdleTime - $this->maxIdleTime) > 30) {
                $this->maxIdleTime = $newIdleTime;
                mftpLog(LOG_DEBUG, "ConnectionPool: Adjusted idle timeout to {$this->maxIdleTime}s");
            }
        }
    }

    /**
     * Calculate average response time across all connections
     */
    private function calculateAverageResponseTime() {
        $totalTime = 0;
        $totalOperations = 0;
        
        foreach ($this->connectionStats as $stats) {
            $totalTime += $stats['total_response_time'] ?? 0;
            $totalOperations += $stats['total_operations'] ?? 0;
        }
        
        return $totalOperations > 0 ? ($totalTime / $totalOperations) : 0;
    }
    
    /**
     * Execute a chunk of operations concurrently
     */
    private function executeConcurrentChunk($operations) {
        $results = [];
        $errors = [];
        $connections = [];
        
        try {
            // Get connections for all operations
            foreach ($operations as $index => $operation) {
                try {
                    $connection = $this->getConnection(
                        $operation['connectionType'], 
                        $operation['configuration']
                    );
                    $connections[$index] = $connection;
                } catch (Exception $e) {
                    $errors[] = [
                        'index' => $index,
                        'error' => $e->getMessage(),
                        'operation' => $operation
                    ];
                }
            }
            
            // Execute operations
            foreach ($connections as $index => $connection) {
                try {
                    $operation = $operations[$index];
                    $result = $operation['callback']($connection);
                    $results[] = [
                        'index' => $index,
                        'result' => $result,
                        'operation' => $operation
                    ];
                } catch (Exception $e) {
                    $errors[] = [
                        'index' => $index,
                        'error' => $e->getMessage(),
                        'operation' => $operations[$index]
                    ];
                }
            }
            
        } finally {
            // Return all connections to pool
            foreach ($connections as $index => $connection) {
                $operation = $operations[$index];
                $this->returnConnection($connection, $operation['connectionType'], $operation['configuration']);
            }
        }
        
        return [
            'results' => $results,
            'errors' => $errors
        ];
    }
    
    /**
     * Generate a unique key for the connection pool
     */
    private function generatePoolKey($connectionType, $configuration) {
        // Create a hash based on connection parameters
        $keyData = [
            'type' => $connectionType,
            'host' => $configuration['host'] ?? '',
            'port' => $configuration['port'] ?? '',
            'username' => $configuration['username'] ?? '',
            'ssl' => $configuration['ssl'] ?? false
        ];
        
        return md5(serialize($keyData));
    }
    
    /**
     * Get an idle connection from the pool
     */
    private function getIdleConnection($poolKey) {
        if (!isset($this->pools[$poolKey])) {
            return null;
        }
        
        foreach ($this->pools[$poolKey] as $connectionId => $connectionInfo) {
            if ($connectionInfo['state'] === self::STATE_IDLE) {
                return $connectionInfo['connection'];
            }
        }
        
        return null;
    }
    
    /**
     * Check if a new connection can be created
     */
    private function canCreateNewConnection($poolKey) {
        $currentSize = isset($this->pools[$poolKey]) ? count($this->pools[$poolKey]) : 0;
        return $currentSize < $this->maxPoolSize;
    }
    
    /**
     * Create a new connection and add it to the pool
     */
    private function createNewConnection($connectionType, $configuration, $poolKey) {
        $connectionFactory = new ConnectionFactory();
        $configurationFactory = new ConfigurationFactory();
        
        $config = $configurationFactory->getConfiguration($connectionType, $configuration);
        $connection = $connectionFactory->getConnection($connectionType, $config);
        
        try {
            $connection->connect();
            $connection->authenticate();
            
            $connectionId = uniqid('conn_', true);
            
            if (!isset($this->pools[$poolKey])) {
                $this->pools[$poolKey] = [];
            }
            
            $this->pools[$poolKey][$connectionId] = [
                'connection' => $connection,
                'state' => self::STATE_ACTIVE,
                'created_at' => time(),
                'last_used' => time(),
                'use_count' => 1,
                'connection_id' => $connectionId
            ];
            
            mftpLog(LOG_DEBUG, "ConnectionPool: Created new connection ($connectionId) for $connectionType");
            
            return $connection;
            
        } catch (Exception $e) {
            if ($connection) {
                try {
                    $connection->disconnect();
                } catch (Exception $disconnectException) {
                    // Ignore disconnect errors
                }
            }
            throw $e;
        }
    }
    
    /**
     * Wait for an available connection
     */
    private function waitForAvailableConnection($connectionType, $configuration, $poolKey) {
        $maxWaitTime = self::CONNECTION_TIMEOUT;
        $waitInterval = 0.5; // 500ms
        $waited = 0;
        
        while ($waited < $maxWaitTime) {
            $connection = $this->getIdleConnection($poolKey);
            if ($connection) {
                $this->markConnectionActive($connection, $poolKey);
                return $connection;
            }
            
            usleep($waitInterval * 1000000); // Convert to microseconds
            $waited += $waitInterval;
        }
        
        throw new RuntimeException("Connection pool timeout after {$maxWaitTime} seconds");
    }
    
    /**
     * Mark connection as active
     */
    private function markConnectionActive($connection, $poolKey) {
        foreach ($this->pools[$poolKey] as &$connectionInfo) {
            if ($connectionInfo['connection'] === $connection) {
                $connectionInfo['state'] = self::STATE_ACTIVE;
                $connectionInfo['last_used'] = time();
                $connectionInfo['use_count']++;
                break;
            }
        }
    }
    
    /**
     * Mark connection as idle
     */
    private function markConnectionIdle($connection, $poolKey) {
        foreach ($this->pools[$poolKey] as &$connectionInfo) {
            if ($connectionInfo['connection'] === $connection) {
                $connectionInfo['state'] = self::STATE_IDLE;
                $connectionInfo['last_used'] = time();
                break;
            }
        }
    }
    
    /**
     * Check if connection is still valid
     */
    private function isConnectionValid($connection) {
        try {
            return $connection->isConnected() && $connection->isAuthenticated();
        } catch (Exception $e) {
            return false;
        }
    }
    
    /**
     * Remove a connection from the pool
     */
    private function removeConnection($connection, $poolKey) {
        if (!isset($this->pools[$poolKey])) {
            return;
        }
        
        foreach ($this->pools[$poolKey] as $connectionId => $connectionInfo) {
            if ($connectionInfo['connection'] === $connection) {
                try {
                    $connection->disconnect();
                } catch (Exception $e) {
                    // Ignore disconnect errors
                }
                
                unset($this->pools[$poolKey][$connectionId]);
                mftpLog(LOG_DEBUG, "ConnectionPool: Removed connection ($connectionId) from pool");
                break;
            }
        }
    }
    
    /**
     * Clean up expired connections
     */
    private function cleanupExpiredConnections($poolKey = null) {
        $pools = $poolKey ? [$poolKey => $this->pools[$poolKey] ?? []] : $this->pools;
        
        foreach ($pools as $key => $pool) {
            foreach ($pool as $connectionId => $connectionInfo) {
                $idleTime = time() - $connectionInfo['last_used'];
                
                if ($connectionInfo['state'] === self::STATE_IDLE && $idleTime > $this->maxIdleTime) {
                    $this->removeConnection($connectionInfo['connection'], $key);
                }
            }
        }
    }
    
    /**
     * Update connection statistics
     */
    private function updateConnectionStats($connection, $success, $duration) {
        $connectionHash = spl_object_hash($connection);
        
        if (!isset($this->connectionStats[$connectionHash])) {
            $this->connectionStats[$connectionHash] = [
                'total_operations' => 0,
                'successful_operations' => 0,
                'failed_operations' => 0,
                'total_duration' => 0,
                'average_duration' => 0
            ];
        }
        
        $stats = &$this->connectionStats[$connectionHash];
        $stats['total_operations']++;
        $stats['total_duration'] += $duration;
        $stats['average_duration'] = $stats['total_duration'] / $stats['total_operations'];
        
        if ($success) {
            $stats['successful_operations']++;
        } else {
            $stats['failed_operations']++;
        }
    }
    
    /**
     * Get pool statistics
     */
    public function getPoolStats() {
        $stats = [
            'total_pools' => count($this->pools),
            'total_connections' => 0,
            'active_connections' => 0,
            'idle_connections' => 0,
            'pools' => []
        ];
        
        foreach ($this->pools as $poolKey => $pool) {
            $poolStats = [
                'total_connections' => count($pool),
                'active_connections' => 0,
                'idle_connections' => 0,
                'connections' => []
            ];
            
            foreach ($pool as $connectionId => $connectionInfo) {
                $stats['total_connections']++;
                $poolStats['total_connections']++;
                
                if ($connectionInfo['state'] === self::STATE_ACTIVE) {
                    $stats['active_connections']++;
                    $poolStats['active_connections']++;
                } elseif ($connectionInfo['state'] === self::STATE_IDLE) {
                    $stats['idle_connections']++;
                    $poolStats['idle_connections']++;
                }
                
                $poolStats['connections'][$connectionId] = [
                    'state' => $connectionInfo['state'],
                    'created_at' => $connectionInfo['created_at'],
                    'last_used' => $connectionInfo['last_used'],
                    'use_count' => $connectionInfo['use_count'],
                    'idle_time' => time() - $connectionInfo['last_used']
                ];
            }
            
            $stats['pools'][$poolKey] = $poolStats;
        }
        
        return $stats;
    }
    
    /**
     * Close all connections and clear pools
     */
    public function closeAllConnections() {
        foreach ($this->pools as $poolKey => $pool) {
            foreach ($pool as $connectionInfo) {
                try {
                    $connectionInfo['connection']->disconnect();
                } catch (Exception $e) {
                    // Ignore disconnect errors
                }
            }
        }
        
        $this->pools = [];
        $this->connectionStats = [];
        
        mftpLog(LOG_INFO, "ConnectionPool: All connections closed and pools cleared");
    }
    
    /**
     * Destructor to ensure connections are closed
     */
    public function __destruct() {
        $this->closeAllConnections();
    }
} 