<?php

require_once(dirname(__FILE__) . '/../constants.php');
require_once(dirname(__FILE__) . '/PerformanceOptimizer.php');
require_once(dirname(__FILE__) . '/../lib/helpers.php');
require_once(dirname(__FILE__) . '/../lib/logging.php');

/**
 * Comprehensive cache manager for performance optimization
 * Implements multiple caching strategies for different data types
 */
class CacheManager {
    
    private static $instance = null;
    private $cacheDirectory;
    private $memoryCaches = [];
    private $performanceOptimizer;
    private $config;
    
    // Cache types
    const CACHE_DIRECTORY_LISTINGS = 'directory_listings';
    const CACHE_FILE_METADATA = 'file_metadata';
    const CACHE_SERVER_CAPABILITIES = 'server_capabilities';
    const CACHE_CONNECTION_INFO = 'connection_info';
    const CACHE_TRANSFER_RESUME = 'transfer_resume';
    const CACHE_ARCHIVE_CONTENTS = 'archive_contents';
    const CACHE_USER_PREFERENCES = 'user_preferences';
    
    // Cache storage types
    const STORAGE_MEMORY = 'memory';
    const STORAGE_FILE = 'file';
    const STORAGE_HYBRID = 'hybrid'; // Memory + file fallback
    
    // Cache strategies
    const STRATEGY_LRU = 'lru';        // Least Recently Used
    const STRATEGY_LFU = 'lfu';        // Least Frequently Used
    const STRATEGY_TTL = 'ttl';        // Time To Live
    const STRATEGY_ADAPTIVE = 'adaptive'; // Adaptive based on usage
    
    private function __construct() {
        $this->performanceOptimizer = PerformanceOptimizer::getInstance();
        $this->initializeConfiguration();
        $this->initializeCacheDirectory();
        $this->initializeMemoryCaches();
        
        mftpLog(LOG_DEBUG, "CacheManager: Initialized with directory: {$this->cacheDirectory}");
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Initialize cache configuration
     */
    private function initializeConfiguration() {
        $metrics = $this->performanceOptimizer->getPerformanceMetrics();
        
        $this->config = [
            // Memory cache limits (percentage of available memory)
            'memory_cache_size' => min(104857600, $metrics['available_memory'] * 0.1), // 10% of available memory, max 100MB
            'max_cache_entries' => 10000,
            'default_ttl' => 3600, // 1 hour
            
            // File cache limits
            'file_cache_size' => 524288000, // 500MB
            'cleanup_threshold' => 0.8, // Clean when 80% full
            'compression_enabled' => function_exists('gzcompress'),
            
            // Cache strategies per type
            'cache_strategies' => [
                self::CACHE_DIRECTORY_LISTINGS => ['storage' => self::STORAGE_HYBRID, 'strategy' => self::STRATEGY_LRU, 'ttl' => 1800],
                self::CACHE_FILE_METADATA => ['storage' => self::STORAGE_MEMORY, 'strategy' => self::STRATEGY_LFU, 'ttl' => 3600],
                self::CACHE_SERVER_CAPABILITIES => ['storage' => self::STORAGE_FILE, 'strategy' => self::STRATEGY_TTL, 'ttl' => 86400],
                self::CACHE_CONNECTION_INFO => ['storage' => self::STORAGE_MEMORY, 'strategy' => self::STRATEGY_TTL, 'ttl' => 900],
                self::CACHE_TRANSFER_RESUME => ['storage' => self::STORAGE_FILE, 'strategy' => self::STRATEGY_TTL, 'ttl' => 604800],
                self::CACHE_ARCHIVE_CONTENTS => ['storage' => self::STORAGE_HYBRID, 'strategy' => self::STRATEGY_LRU, 'ttl' => 7200],
                self::CACHE_USER_PREFERENCES => ['storage' => self::STORAGE_FILE, 'strategy' => self::STRATEGY_TTL, 'ttl' => 2592000]
            ],
            
            // Performance settings
            'background_cleanup' => true,
            'preload_common_data' => true,
            'cache_compression_threshold' => 1024, // Compress data larger than 1KB
        ];
    }
    
    /**
     * Initialize cache directory structure
     */
    private function initializeCacheDirectory() {
        $this->cacheDirectory = sys_get_temp_dir() . '/mftp_cache';
        
        if (!is_dir($this->cacheDirectory)) {
            mkdir($this->cacheDirectory, 0755, true);
        }
        
        // Create subdirectories for different cache types
        foreach (array_keys($this->config['cache_strategies']) as $cacheType) {
            $subDir = $this->cacheDirectory . '/' . $cacheType;
            if (!is_dir($subDir)) {
                mkdir($subDir, 0755, true);
            }
        }
    }
    
    /**
     * Initialize in-memory caches
     */
    private function initializeMemoryCaches() {
        foreach (array_keys($this->config['cache_strategies']) as $cacheType) {
            $this->memoryCaches[$cacheType] = [
                'data' => [],
                'access_times' => [],
                'access_counts' => [],
                'size' => 0
            ];
        }
    }
    
    /**
     * Store data in cache with automatic strategy selection
     */
    public function set($cacheType, $key, $data, $ttl = null) {
        $strategy = $this->config['cache_strategies'][$cacheType] ?? null;
        if (!$strategy) {
            throw new InvalidArgumentException("Unknown cache type: $cacheType");
        }
        
        $ttl = $ttl ?? $strategy['ttl'];
        $cacheKey = $this->generateCacheKey($cacheType, $key);
        
        // Serialize and possibly compress data
        $serializedData = $this->serializeData($data);
        $compressedData = $this->compressData($serializedData);
        
        $cacheEntry = [
            'data' => $compressedData,
            'original_size' => strlen($serializedData),
            'compressed_size' => strlen($compressedData),
            'created_at' => time(),
            'expires_at' => time() + $ttl,
            'access_count' => 0,
            'last_access' => time(),
            'compressed' => $compressedData !== $serializedData
        ];
        
        // Store based on strategy
        switch ($strategy['storage']) {
            case self::STORAGE_MEMORY:
                $this->setMemoryCache($cacheType, $cacheKey, $cacheEntry);
                break;
            case self::STORAGE_FILE:
                $this->setFileCache($cacheType, $cacheKey, $cacheEntry);
                break;
            case self::STORAGE_HYBRID:
                $this->setMemoryCache($cacheType, $cacheKey, $cacheEntry);
                $this->setFileCache($cacheType, $cacheKey, $cacheEntry);
                break;
        }
        
        mftpLog(LOG_DEBUG, "CacheManager: Stored $cacheType cache entry '$key' (size: {$cacheEntry['compressed_size']} bytes)");
        
        // Trigger cleanup if needed
        $this->conditionalCleanup($cacheType);
    }
    
    /**
     * Retrieve data from cache
     */
    public function get($cacheType, $key) {
        $strategy = $this->config['cache_strategies'][$cacheType] ?? null;
        if (!$strategy) {
            return null;
        }
        
        $cacheKey = $this->generateCacheKey($cacheType, $key);
        $cacheEntry = null;
        
        // Try to get from appropriate storage
        switch ($strategy['storage']) {
            case self::STORAGE_MEMORY:
                $cacheEntry = $this->getMemoryCache($cacheType, $cacheKey);
                break;
            case self::STORAGE_FILE:
                $cacheEntry = $this->getFileCache($cacheType, $cacheKey);
                break;
            case self::STORAGE_HYBRID:
                // Try memory first, fallback to file
                $cacheEntry = $this->getMemoryCache($cacheType, $cacheKey);
                if (!$cacheEntry) {
                    $cacheEntry = $this->getFileCache($cacheType, $cacheKey);
                    // If found in file, promote to memory
                    if ($cacheEntry) {
                        $this->setMemoryCache($cacheType, $cacheKey, $cacheEntry);
                    }
                }
                break;
        }
        
        if (!$cacheEntry) {
            return null;
        }
        
        // Check expiration
        if ($cacheEntry['expires_at'] < time()) {
            $this->delete($cacheType, $key);
            return null;
        }
        
        // Update access statistics
        $cacheEntry['access_count']++;
        $cacheEntry['last_access'] = time();
        $this->updateCacheEntry($cacheType, $cacheKey, $cacheEntry, $strategy);
        
        // Decompress and deserialize data
        $decompressedData = $this->decompressData($cacheEntry['data'], $cacheEntry['compressed']);
        $data = $this->deserializeData($decompressedData);
        
        mftpLog(LOG_DEBUG, "CacheManager: Retrieved $cacheType cache entry '$key' (access count: {$cacheEntry['access_count']})");
        
        return $data;
    }
    
    /**
     * Delete cache entry
     */
    public function delete($cacheType, $key) {
        $cacheKey = $this->generateCacheKey($cacheType, $key);
        
        // Remove from memory cache
        if (isset($this->memoryCaches[$cacheType]['data'][$cacheKey])) {
            unset($this->memoryCaches[$cacheType]['data'][$cacheKey]);
            unset($this->memoryCaches[$cacheType]['access_times'][$cacheKey]);
            unset($this->memoryCaches[$cacheType]['access_counts'][$cacheKey]);
        }
        
        // Remove from file cache
        $filePath = $this->getFileCachePath($cacheType, $cacheKey);
        if (file_exists($filePath)) {
            unlink($filePath);
        }
    }
    
    /**
     * Cache directory listing with metadata
     */
    public function cacheDirectoryListing($connectionType, $host, $path, $listing, $showHidden = false) {
        $key = sprintf('%s:%s:%s:%s', $connectionType, $host, $path, $showHidden ? 'hidden' : 'visible');
        
        // Add caching metadata
        $cachedListing = [
            'listing' => $listing,
            'cached_at' => time(),
            'connection_type' => $connectionType,
            'host' => $host,
            'path' => $path,
            'show_hidden' => $showHidden,
            'file_count' => count($listing)
        ];
        
        $this->set(self::CACHE_DIRECTORY_LISTINGS, $key, $cachedListing);
    }
    
    /**
     * Get cached directory listing
     */
    public function getCachedDirectoryListing($connectionType, $host, $path, $showHidden = false) {
        $key = sprintf('%s:%s:%s:%s', $connectionType, $host, $path, $showHidden ? 'hidden' : 'visible');
        $cached = $this->get(self::CACHE_DIRECTORY_LISTINGS, $key);
        
        return $cached ? $cached['listing'] : null;
    }
    
    /**
     * Cache file metadata
     */
    public function cacheFileMetadata($connectionType, $host, $filePath, $metadata) {
        $key = sprintf('%s:%s:%s', $connectionType, $host, $filePath);
        
        $cachedMetadata = array_merge($metadata, [
            'cached_at' => time(),
            'connection_type' => $connectionType,
            'host' => $host,
            'file_path' => $filePath
        ]);
        
        $this->set(self::CACHE_FILE_METADATA, $key, $cachedMetadata);
    }
    
    /**
     * Get cached file metadata
     */
    public function getCachedFileMetadata($connectionType, $host, $filePath) {
        $key = sprintf('%s:%s:%s', $connectionType, $host, $filePath);
        return $this->get(self::CACHE_FILE_METADATA, $key);
    }
    
    /**
     * Cache server capabilities
     */
    public function cacheServerCapabilities($connectionType, $host, $port, $capabilities) {
        $key = sprintf('%s:%s:%d', $connectionType, $host, $port);
        
        $cachedCapabilities = array_merge($capabilities, [
            'cached_at' => time(),
            'connection_type' => $connectionType,
            'host' => $host,
            'port' => $port
        ]);
        
        $this->set(self::CACHE_SERVER_CAPABILITIES, $key, $cachedCapabilities);
    }
    
    /**
     * Get cached server capabilities
     */
    public function getCachedServerCapabilities($connectionType, $host, $port) {
        $key = sprintf('%s:%s:%d', $connectionType, $host, $port);
        return $this->get(self::CACHE_SERVER_CAPABILITIES, $key);
    }
    
    /**
     * Invalidate related caches when directory structure changes
     */
    public function invalidateDirectoryCache($connectionType, $host, $path) {
        // Get all cache keys for this connection and path
        $pattern = sprintf('%s:%s:%s', $connectionType, $host, $path);
        
        // Invalidate directory listings
        foreach ($this->memoryCaches[self::CACHE_DIRECTORY_LISTINGS]['data'] as $cacheKey => $entry) {
            if (strpos($cacheKey, $pattern) === 0) {
                $this->delete(self::CACHE_DIRECTORY_LISTINGS, substr($cacheKey, strlen($this->generateCacheKey(self::CACHE_DIRECTORY_LISTINGS, ''))));
            }
        }
        
        // Invalidate file metadata for files in this directory
        foreach ($this->memoryCaches[self::CACHE_FILE_METADATA]['data'] as $cacheKey => $entry) {
            if (strpos($cacheKey, $pattern) === 0) {
                $this->delete(self::CACHE_FILE_METADATA, substr($cacheKey, strlen($this->generateCacheKey(self::CACHE_FILE_METADATA, ''))));
            }
        }
        
        mftpLog(LOG_DEBUG, "CacheManager: Invalidated directory cache for $connectionType://$host$path");
    }
    
    /**
     * Store data in memory cache
     */
    private function setMemoryCache($cacheType, $cacheKey, $cacheEntry) {
        // Check memory limits
        if ($this->memoryCaches[$cacheType]['size'] > $this->config['memory_cache_size']) {
            $this->cleanupMemoryCache($cacheType);
        }
        
        $this->memoryCaches[$cacheType]['data'][$cacheKey] = $cacheEntry;
        $this->memoryCaches[$cacheType]['access_times'][$cacheKey] = time();
        $this->memoryCaches[$cacheType]['access_counts'][$cacheKey] = 0;
        $this->memoryCaches[$cacheType]['size'] += $cacheEntry['compressed_size'];
    }
    
    /**
     * Get data from memory cache
     */
    private function getMemoryCache($cacheType, $cacheKey) {
        if (!isset($this->memoryCaches[$cacheType]['data'][$cacheKey])) {
            return null;
        }
        
        // Update access tracking
        $this->memoryCaches[$cacheType]['access_times'][$cacheKey] = time();
        $this->memoryCaches[$cacheType]['access_counts'][$cacheKey]++;
        
        return $this->memoryCaches[$cacheType]['data'][$cacheKey];
    }
    
    /**
     * Store data in file cache
     */
    private function setFileCache($cacheType, $cacheKey, $cacheEntry) {
        $filePath = $this->getFileCachePath($cacheType, $cacheKey);
        $result = file_put_contents($filePath, serialize($cacheEntry), LOCK_EX);
        
        if ($result === false) {
            mftpLog(LOG_WARNING, "CacheManager: Failed to write file cache: $filePath");
        }
    }
    
    /**
     * Get data from file cache
     */
    private function getFileCache($cacheType, $cacheKey) {
        $filePath = $this->getFileCachePath($cacheType, $cacheKey);
        
        if (!file_exists($filePath)) {
            return null;
        }
        
        $data = file_get_contents($filePath);
        if ($data === false) {
            return null;
        }
        
        $cacheEntry = unserialize($data);
        return $cacheEntry !== false ? $cacheEntry : null;
    }
    
    /**
     * Update cache entry access statistics
     */
    private function updateCacheEntry($cacheType, $cacheKey, $cacheEntry, $strategy) {
        switch ($strategy['storage']) {
            case self::STORAGE_MEMORY:
                $this->memoryCaches[$cacheType]['data'][$cacheKey] = $cacheEntry;
                break;
            case self::STORAGE_FILE:
                $this->setFileCache($cacheType, $cacheKey, $cacheEntry);
                break;
            case self::STORAGE_HYBRID:
                $this->memoryCaches[$cacheType]['data'][$cacheKey] = $cacheEntry;
                break; // Don't update file cache on every access
        }
    }
    
    /**
     * Generate cache key
     */
    private function generateCacheKey($cacheType, $key) {
        return $cacheType . '_' . md5($key);
    }
    
    /**
     * Get file cache path
     */
    private function getFileCachePath($cacheType, $cacheKey) {
        return $this->cacheDirectory . '/' . $cacheType . '/' . $cacheKey . '.cache';
    }
    
    /**
     * Serialize data for storage
     */
    private function serializeData($data) {
        return serialize($data);
    }
    
    /**
     * Deserialize data from storage
     */
    private function deserializeData($data) {
        return unserialize($data);
    }
    
    /**
     * Compress data if beneficial
     */
    private function compressData($data) {
        if (!$this->config['compression_enabled'] || 
            strlen($data) < $this->config['cache_compression_threshold']) {
            return $data;
        }
        
        $compressed = gzcompress($data, 6);
        return $compressed !== false && strlen($compressed) < strlen($data) ? $compressed : $data;
    }
    
    /**
     * Decompress data if needed
     */
    private function decompressData($data, $isCompressed) {
        if (!$isCompressed || !$this->config['compression_enabled']) {
            return $data;
        }
        
        $decompressed = gzuncompress($data);
        return $decompressed !== false ? $decompressed : $data;
    }
    
    /**
     * Cleanup memory cache based on strategy
     */
    private function cleanupMemoryCache($cacheType) {
        $strategy = $this->config['cache_strategies'][$cacheType];
        $cache = &$this->memoryCaches[$cacheType];
        
        switch ($strategy['strategy']) {
            case self::STRATEGY_LRU:
                $this->cleanupLRU($cache);
                break;
            case self::STRATEGY_LFU:
                $this->cleanupLFU($cache);
                break;
            case self::STRATEGY_TTL:
                $this->cleanupTTL($cache);
                break;
            case self::STRATEGY_ADAPTIVE:
                $this->cleanupAdaptive($cache);
                break;
        }
        
        // Recalculate size
        $cache['size'] = 0;
        foreach ($cache['data'] as $entry) {
            $cache['size'] += $entry['compressed_size'];
        }
    }
    
    /**
     * LRU cleanup - remove least recently used entries
     */
    private function cleanupLRU(&$cache) {
        asort($cache['access_times']);
        $toRemove = array_slice(array_keys($cache['access_times']), 0, max(1, count($cache['access_times']) * 0.25));
        
        foreach ($toRemove as $key) {
            unset($cache['data'][$key]);
            unset($cache['access_times'][$key]);
            unset($cache['access_counts'][$key]);
        }
    }
    
    /**
     * LFU cleanup - remove least frequently used entries
     */
    private function cleanupLFU(&$cache) {
        asort($cache['access_counts']);
        $toRemove = array_slice(array_keys($cache['access_counts']), 0, max(1, count($cache['access_counts']) * 0.25));
        
        foreach ($toRemove as $key) {
            unset($cache['data'][$key]);
            unset($cache['access_times'][$key]);
            unset($cache['access_counts'][$key]);
        }
    }
    
    /**
     * TTL cleanup - remove expired entries
     */
    private function cleanupTTL(&$cache) {
        $currentTime = time();
        $toRemove = [];
        
        foreach ($cache['data'] as $key => $entry) {
            if ($entry['expires_at'] < $currentTime) {
                $toRemove[] = $key;
            }
        }
        
        foreach ($toRemove as $key) {
            unset($cache['data'][$key]);
            unset($cache['access_times'][$key]);
            unset($cache['access_counts'][$key]);
        }
    }
    
    /**
     * Adaptive cleanup - combines multiple strategies
     */
    private function cleanupAdaptive(&$cache) {
        // First remove expired entries
        $this->cleanupTTL($cache);
        
        // If still over limit, use LRU
        if ($cache['size'] > $this->config['memory_cache_size']) {
            $this->cleanupLRU($cache);
        }
    }
    
    /**
     * Conditional cleanup based on usage
     */
    private function conditionalCleanup($cacheType) {
        $cache = $this->memoryCaches[$cacheType];
        
        if ($cache['size'] > $this->config['memory_cache_size'] || 
            count($cache['data']) > $this->config['max_cache_entries']) {
            $this->cleanupMemoryCache($cacheType);
        }
        
        // Background cleanup for file cache
        if ($this->config['background_cleanup'] && rand(1, 100) === 1) {
            $this->performBackgroundCleanup();
        }
    }
    
    /**
     * Background cleanup of file caches
     */
    private function performBackgroundCleanup() {
        foreach (array_keys($this->config['cache_strategies']) as $cacheType) {
            $cacheDir = $this->cacheDirectory . '/' . $cacheType;
            $files = glob($cacheDir . '/*.cache');
            
            foreach ($files as $file) {
                $cacheEntry = unserialize(file_get_contents($file));
                if ($cacheEntry && $cacheEntry['expires_at'] < time()) {
                    unlink($file);
                }
            }
        }
    }
    
    /**
     * Get comprehensive cache statistics
     */
    public function getStatistics() {
        $stats = [
            'memory_usage' => 0,
            'memory_entries' => 0,
            'file_entries' => 0,
            'total_size' => 0,
            'hit_rate' => 0,
            'cache_types' => []
        ];
        
        foreach ($this->memoryCaches as $cacheType => $cache) {
            $stats['memory_usage'] += $cache['size'];
            $stats['memory_entries'] += count($cache['data']);
            
            // Count file cache entries
            $cacheDir = $this->cacheDirectory . '/' . $cacheType;
            $fileCount = count(glob($cacheDir . '/*.cache'));
            $stats['file_entries'] += $fileCount;
            
            $stats['cache_types'][$cacheType] = [
                'memory_entries' => count($cache['data']),
                'file_entries' => $fileCount,
                'memory_size' => $cache['size'],
                'strategy' => $this->config['cache_strategies'][$cacheType]
            ];
        }
        
        $stats['total_size'] = $stats['memory_usage'];
        
        return $stats;
    }
    
    /**
     * Clear all caches
     */
    public function clearAll() {
        // Clear memory caches
        foreach ($this->memoryCaches as $cacheType => &$cache) {
            $cache['data'] = [];
            $cache['access_times'] = [];
            $cache['access_counts'] = [];
            $cache['size'] = 0;
        }
        
        // Clear file caches
        foreach (array_keys($this->config['cache_strategies']) as $cacheType) {
            $cacheDir = $this->cacheDirectory . '/' . $cacheType;
            $files = glob($cacheDir . '/*.cache');
            foreach ($files as $file) {
                unlink($file);
            }
        }
        
        mftpLog(LOG_INFO, "CacheManager: Cleared all caches");
    }
    
    /**
     * Preload common data for better performance
     */
    public function preloadCommonData() {
        if (!$this->config['preload_common_data']) {
            return;
        }
        
        // This would be called with common directory listings, server capabilities, etc.
        // Implementation depends on specific use case
        mftpLog(LOG_DEBUG, "CacheManager: Preloading common data");
    }
} 