<?php

declare(strict_types=1);

/**
 * Log Priority Enum for PHP 8.1+ Modernization
 * 
 * Modern enum replacement for traditional LOG_* constants
 * Provides better type safety and functionality
 */
enum LogPriority: int
{
    case EMERGENCY = LOG_EMERG;
    case ALERT = LOG_ALERT;
    case CRITICAL = LOG_CRIT;
    case ERROR = LOG_ERR;
    case WARNING = LOG_WARNING;
    case NOTICE = LOG_NOTICE;
    case INFO = LOG_INFO;
    case DEBUG = LOG_DEBUG;
    
    /**
     * Get text representation of log priority
     */
    public function getText(): string
    {
        return match ($this) {
            self::EMERGENCY => 'EMERG',
            self::ALERT => 'ALERT',
            self::CRITICAL => 'CRIT',
            self::ERROR => 'ERR',
            self::WARNING => 'WARNING',
            self::NOTICE => 'NOTICE',
            self::INFO => 'INFO',
            self::DEBUG => 'DEBUG',
        };
    }
    
    /**
     * Get colored text representation for console output
     */
    public function getColoredText(): string
    {
        return match ($this) {
            self::EMERGENCY => "\033[41m\033[37m EMERG \033[0m",  // White on red
            self::ALERT => "\033[41m ALERT \033[0m",              // Red background
            self::CRITICAL => "\033[31m CRIT \033[0m",            // Red text
            self::ERROR => "\033[31m ERR \033[0m",                // Red text
            self::WARNING => "\033[33m WARNING \033[0m",          // Yellow text
            self::NOTICE => "\033[36m NOTICE \033[0m",            // Cyan text
            self::INFO => "\033[32m INFO \033[0m",                // Green text
            self::DEBUG => "\033[37m DEBUG \033[0m",              // Gray text
        };
    }
    
    /**
     * Check if this priority should be logged based on threshold
     */
    public function shouldLog(self $threshold): bool
    {
        return $this->value <= $threshold->value;
    }
    
    /**
     * Get severity level (0 = most severe, 7 = least severe)
     */
    public function getSeverityLevel(): int
    {
        return $this->value;
    }
    
    /**
     * Check if this is a critical priority that requires immediate attention
     */
    public function isCritical(): bool
    {
        return match ($this) {
            self::EMERGENCY, self::ALERT, self::CRITICAL => true,
            default => false,
        };
    }
    
    /**
     * Create from legacy integer value with validation
     */
    public static function fromLegacyValue(int $value): self
    {
        return self::tryFrom($value) ?? throw new InvalidArgumentException(
            "Invalid log priority value: $value"
        );
    }
    
    /**
     * Get default threshold for production environments
     */
    public static function getProductionThreshold(): self
    {
        return self::WARNING;
    }
    
    /**
     * Get default threshold for development environments
     */
    public static function getDevelopmentThreshold(): self
    {
        return self::DEBUG;
    }
} 