<?php

declare(strict_types=1);

/**
 * File Type Enum for PHP 8.1+ Modernization
 * 
 * Modern enum for file type classification with enhanced functionality
 */
enum FileType: string
{
    case FILE = 'file';
    case DIRECTORY = 'directory';
    case SYMLINK = 'symlink';
    case BLOCK_DEVICE = 'block_device';
    case CHARACTER_DEVICE = 'character_device';
    case FIFO = 'fifo';
    case SOCKET = 'socket';
    case UNKNOWN = 'unknown';
    
    /**
     * Get human-readable description
     */
    public function getDescription(): string
    {
        return match ($this) {
            self::FILE => 'Regular file',
            self::DIRECTORY => 'Directory',
            self::SYMLINK => 'Symbolic link',
            self::BLOCK_DEVICE => 'Block device',
            self::CHARACTER_DEVICE => 'Character device',
            self::FIFO => 'Named pipe (FIFO)',
            self::SOCKET => 'Unix socket',
            self::UNKNOWN => 'Unknown file type',
        };
    }
    
    /**
     * Get icon representation for UI
     */
    public function getIcon(): string
    {
        return match ($this) {
            self::FILE => '📄',
            self::DIRECTORY => '📁',
            self::SYMLINK => '🔗',
            self::BLOCK_DEVICE => '🔲',
            self::CHARACTER_DEVICE => '⚡',
            self::FIFO => '🔀',
            self::SOCKET => '🔌',
            self::UNKNOWN => '❓',
        };
    }
    
    /**
     * Check if this file type can be downloaded
     */
    public function isDownloadable(): bool
    {
        return match ($this) {
            self::FILE, self::SYMLINK => true,
            default => false,
        };
    }
    
    /**
     * Check if this file type can be uploaded to (for directories)
     */
    public function isUploadTarget(): bool
    {
        return $this === self::DIRECTORY;
    }
    
    /**
     * Check if this file type can be edited
     */
    public function isEditable(): bool
    {
        return $this === self::FILE;
    }
    
    /**
     * Check if this file type can have permissions changed
     */
    public function canChangePermissions(): bool
    {
        return match ($this) {
            self::FILE, self::DIRECTORY, self::SYMLINK => true,
            default => false,
        };
    }
    
    /**
     * Create from legacy string with better error handling
     */
    public static function fromLegacyValue(string $value): self
    {
        return self::tryFrom(strtolower($value)) ?? self::UNKNOWN;
    }
    
    /**
     * Detect file type from file path using modern PHP functions
     */
    public static function detectFromPath(string $path): self
    {
        if (!file_exists($path)) {
            return self::UNKNOWN;
        }
        
        return match (true) {
            is_file($path) => self::FILE,
            is_dir($path) => self::DIRECTORY,
            is_link($path) => self::SYMLINK,
            default => self::UNKNOWN,
        };
    }
    
    /**
     * Get CSS class for styling
     */
    public function getCssClass(): string
    {
        return match ($this) {
            self::FILE => 'file-item',
            self::DIRECTORY => 'directory-item',
            self::SYMLINK => 'symlink-item',
            default => 'unknown-item',
        };
    }
} 