<?php

declare(strict_types=1);

/**
 * Connection State Enum for PHP 8.1+ Modernization
 * 
 * Replaces traditional class constants with modern enum syntax
 * Provides type safety and better IDE support
 */
enum ConnectionState: string
{
    case IDLE = 'idle';
    case ACTIVE = 'active';
    case CONNECTING = 'connecting';
    case AUTHENTICATING = 'authenticating';
    case ERROR = 'error';
    case CLOSING = 'closing';
    case DISCONNECTED = 'disconnected';
    
    /**
     * Get human-readable description of the connection state
     */
    public function getDescription(): string
    {
        return match ($this) {
            self::IDLE => 'Connection is idle and ready for operations',
            self::ACTIVE => 'Connection is actively performing operations',
            self::CONNECTING => 'Attempting to establish connection',
            self::AUTHENTICATING => 'Authenticating with remote server',
            self::ERROR => 'Connection is in error state',
            self::CLOSING => 'Connection is being closed',
            self::DISCONNECTED => 'Connection has been terminated',
        };
    }
    
    /**
     * Check if the connection state allows new operations
     */
    public function allowsOperations(): bool
    {
        return match ($this) {
            self::IDLE, self::ACTIVE => true,
            default => false,
        };
    }
    
    /**
     * Check if the connection state is considered healthy
     */
    public function isHealthy(): bool
    {
        return match ($this) {
            self::IDLE, self::ACTIVE, self::CONNECTING, self::AUTHENTICATING => true,
            self::ERROR, self::CLOSING, self::DISCONNECTED => false,
        };
    }
    
    /**
     * Get all available connection states
     */
    public static function getAllStates(): array
    {
        return array_map(fn($case) => $case->value, self::cases());
    }
    
    /**
     * Create state from legacy string value with validation
     */
    public static function fromLegacyValue(string $value): self
    {
        return self::tryFrom($value) ?? throw new InvalidArgumentException(
            "Invalid connection state: '$value'. Valid states: " . implode(', ', self::getAllStates())
        );
    }
} 