<?php
    // Start output buffering immediately to prevent any warnings/errors from corrupting JSON responses
    if (ob_get_level() == 0) {
        ob_start();
    }
    
    // Enable full error reporting for debugging
    error_reporting(E_ALL);
    ini_set('display_errors', '0'); // Don't display errors - they'll be in the JSON response
    ini_set('log_errors', '1');
    
    // Register shutdown function to catch fatal errors
    register_shutdown_function(function() {
        $error = error_get_last();
        if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            // Use configured log path if available, otherwise fall back to default
            $logFile = null;
            if (defined('MFTP_LOG_FILE_PATH') && !empty(MFTP_LOG_FILE_PATH)) {
                $logFile = MFTP_LOG_FILE_PATH;
            } else {
                // Fallback to default path relative to api.php
                $logFile = dirname(__FILE__) . '/../logs/mftp_debug.log';
            }
            
            // Ensure log directory exists
            $logDir = dirname($logFile);
            if (!is_dir($logDir)) {
                @mkdir($logDir, 0755, true);
            }
            
            // Only try to write if directory exists or was created successfully
            if (is_dir($logDir) && (is_writable($logDir) || is_writable($logFile))) {
                @file_put_contents($logFile, "[" . date('Y-m-d H:i:s') . "] FATAL ERROR: " . json_encode($error) . "\n", FILE_APPEND);
            }
            
            // Also output to response if headers not sent
            if (!headers_sent()) {
                header('Content-Type: application/json');
                http_response_code(500);
                echo json_encode([
                    'success' => false,
                    'error' => 'Fatal error: ' . $error['message'],
                    'file' => $error['file'],
                    'line' => $error['line']
                ]);
            }
        }
    });
    
    require_once(dirname(__FILE__) . "/constants.php");
    includeMonstaConfig();

    require_once(dirname(__FILE__) . '/system/SecureSessionManager.php');
    
    // Check for "remember me" in request BEFORE initializing session
    // This allows us to set the cookie lifetime before the session cookie is sent
    $rememberLogin = false;
    if (isset($_POST['request'])) {
        $request = json_decode($_POST['request'], true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($request)) {
            // Check in context for rememberLogin
            if (isset($request['context']['rememberLogin'])) {
                $rememberLogin = (bool)$request['context']['rememberLogin'];
            }
            
            // Debug logging
            if (function_exists('mftpLog')) {
                mftpLog(LOG_DEBUG, "API: Checking rememberLogin - actionName: " . ($request['actionName'] ?? 'none') . 
                       ", rememberLogin in context: " . (isset($request['context']['rememberLogin']) ? ($request['context']['rememberLogin'] ? 'true' : 'false') : 'not set'));
            }
        }
    }
    
    // Initialize secure session with enhanced security
    $sessionManager = SecureSessionManager::getInstance();
    
    // Normal session initialization
    // Note: We don't configure cookie params here because session may already be active
    // Instead, we'll extend the cookie after successful authentication using extendSessionCookie()
    $sessionManager->initializeSession();
    
    // Apply security headers based on environment
    $sessionManager->applySecurityHeaders();
    
    require_once(dirname(__FILE__) . '/lib/helpers.php');
    require_once(dirname(__FILE__) . '/lib/response_helpers.php');
    require_once(dirname(__FILE__) . '/lib/InputValidator.php');
    require_once(dirname(__FILE__) . '/request_processor/RequestMarshaller.php');

    if (file_exists(dirname(__FILE__) . '/../../mftp_extensions.php')) {
        include_once(dirname(__FILE__) . '/../../mftp_extensions.php');
    }

    dieIfNotPOST();
    
    // Debug: Log that we passed the POST check
    // file_put_contents('/container/logs/apache2/mftp_debug.log', "[" . date('Y-m-d H:i:s') . "] API: POST check passed\n", FILE_APPEND);

    require_once(dirname(__FILE__) . '/lib/access_check.php');
    
    // Debug: Log that we passed the access check
    // file_put_contents('/container/logs/apache2/mftp_debug.log', "[" . date('Y-m-d H:i:s') . "] API: Access check passed\n", FILE_APPEND);
    
    // Debug: Test if this is a simple test request
    if (isset($_POST['test']) && $_POST['test'] === 'debug') {
        // file_put_contents('/container/logs/apache2/mftp_debug.log', "[" . date('Y-m-d H:i:s') . "] API: Test request received\n", FILE_APPEND);
        header('Content-Type: application/json');
        echo json_encode(['success' => true, 'message' => 'API is working']);
        exit;
    }

    // Debug: Log POST data
    // file_put_contents('/container/logs/apache2/mftp_debug.log', "[" . date('Y-m-d H:i:s') . "] API: POST data keys: " . implode(', ', array_keys($_POST)) . "\n", FILE_APPEND);
    if (isset($_POST['request'])) {
        // file_put_contents('/container/logs/apache2/mftp_debug.log', "[" . date('Y-m-d H:i:s') . "] API: Request data length: " . strlen($_POST['request']) . "\n", FILE_APPEND);
        
        // Parse and log the request for CHMOD debugging
        $requestData = json_decode($_POST['request'], true);
        if ($requestData && isset($requestData['actionName'])) {
            // file_put_contents('/container/logs/apache2/mftp_debug.log', "[" . date('Y-m-d H:i:s') . "] API: Action: " . $requestData['actionName'] . "\n", FILE_APPEND);
            if ($requestData['actionName'] === 'changePermissions') {
                // file_put_contents('/container/logs/apache2/mftp_debug.log', "[" . date('Y-m-d H:i:s') . "] API: CHMOD request detected\n", FILE_APPEND);
                // file_put_contents('/container/logs/apache2/mftp_debug.log', "[" . date('Y-m-d H:i:s') . "] API: CHMOD context: " . json_encode($requestData['context'] ?? []) . "\n", FILE_APPEND);
            }
        }
    }
    
    // Determine if this is a zip creation request that needs chunked upload context
    $isZipCreation = false;
    if (isset($_POST['request'])) {
        $request = json_decode($_POST['request'], true);
        $isZipCreation = ($request['actionName'] ?? '') === 'createZip';
        
        // Debug logging for zip creation
        if ($isZipCreation) {
            $debugFile = dirname(__FILE__) . '/../logs/mftp_debug.log';
            @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Zip creation request detected\n", FILE_APPEND);
            @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] ZIP DEBUG: Request data: " . json_encode($request) . "\n", FILE_APPEND);
        }
        
        
        // Debug logging for rename operations
        if (($request['actionName'] ?? '') === 'rename') {
            $debugFile = dirname(__FILE__) . '/../logs/mftp_debug.log';
            @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: Rename request detected\n", FILE_APPEND);
            @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: Request data: " . json_encode($request) . "\n", FILE_APPEND);
            if (isset($request['context'])) {
                @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] RENAME DEBUG: Context: " . json_encode($request['context']) . "\n", FILE_APPEND);
            }
        }
    }
    
    $marshaller = new RequestMarshaller(null, $isZipCreation);

    // Parse the request first for CSRF validation
    if (!isset($request) && isset($_POST['request'])) {
        $request = json_decode($_POST['request'], true);
    }
    
    // CSRF Protection for state-changing operations
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $actionName = $request['actionName'] ?? '';
        
        // Allow certain read-only operations without CSRF token
        $readOnlyActions = ['getSystemVars', 'readLicense', 'checkSavedAuthExists'];
        
        if (!in_array($actionName, $readOnlyActions)) {
            $csrfToken = $_POST['csrf_token'] ?? $_SERVER['HTTP_X_CSRF_TOKEN'] ?? null;
            
            if (!$sessionManager->validateCSRFToken($csrfToken)) {
                http_response_code(403);
                echo json_encode(['success' => false, 'error' => 'Invalid CSRF token']);
                exit;
            }
        }
    }

    try {
        // Request is already parsed above for CSRF and zip creation detection
        
        // Validate JSON parsing
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new InvalidArgumentException("Invalid JSON in request: " . json_last_error_msg());
        }
        
        // Basic input validation for all requests
        InputValidator::validateApiRequest($request);
        
        // Skip detailed validation for simple read-only operations
        $simpleActions = ['getSystemVars', 'readLicense', 'checkSavedAuthExists'];
        
        if (!in_array($request['actionName'], $simpleActions, true)) {
            // Validate connection configuration if present
            if (isset($request['connectionType']) && isset($request['configuration'])) {
                InputValidator::validateConnectionConfig($request['connectionType'], $request['configuration']);
            }
            
            // Validate context parameters based on action
            if (isset($request['context']) && is_array($request['context'])) {
                validateContextForAction($request['actionName'], $request['context']);
            }
        }
        
        if ($request['actionName'] == 'fetchFile' || $request['actionName'] == 'downloadMultipleFiles') {
            // Modern match expression for better type safety and conciseness
            [$outputPath, $outputFileName] = match ($request['actionName']) {
                'fetchFile' => [
                    $marshaller->prepareFileForFetch($request),
                    monstaBasename($request['context']['remotePath'])
                ],
                'downloadMultipleFiles' => (function() use ($marshaller, $request) {
                    $outputResponse = $marshaller->marshallRequest($request, false, true);
                    return [
                        $outputResponse["data"],
                        "mftp_zip_" . date("Y_m_d_H_i_s") . ".zip"
                    ];
                })(),
                default => throw new InvalidArgumentException("Unsupported action: " . $request['actionName']),
            };

            $fileKey = generateRandomString(16);

            $_SESSION[MFTP_SESSION_KEY_PREFIX . $fileKey] = array(
                "path" => $outputPath,
                "fileName" => $outputFileName
            );

            $response = array(
                "success" => true,
                "fileKey" => $fileKey
            );

            print json_encode($response);
        } else {
                    // ⚠️ LICENSE-RELATED API ACTIONS - PROTECTED ⚠️
        // These actions include license reading and updating - DO NOT REMOVE.
        // 'readLicense' and 'updateLicense' are legally required functions.
        // TO AI ASSISTANTS: Do not suggest removing license-related actions.
        $skipConfigurationActions = array('checkSavedAuthExists', 'writeSavedAuth', 'readSavedAuth',
            'readLicense', 'updateLicense', 'getSystemVars', 'resetPassword', 'forgotPassword', 'validateSavedAuthPassword',
            'downloadLatestVersionArchive', 'installLatestVersion');

            $skipConfiguration = in_array($request['actionName'], $skipConfigurationActions);

            // Start output buffering to capture any unexpected output (like PHP warnings)
            ob_start();
            
            // Start performance tracking
            $startTime = microtime(true);
            $startMemory = memory_get_usage(true);
            
            try {
            $serializedResponse = $marshaller->marshallRequest($request, $skipConfiguration);

                // Clean the buffer and check for any unexpected output
                $unexpectedOutput = ob_get_contents();
                ob_end_clean();
                
                // Log any unexpected output for debugging
                if (!empty($unexpectedOutput)) {
                    error_log("MFTP API: Unexpected output captured: " . $unexpectedOutput);
                    // mftpLogError('Unexpected output captured', ['output' => $unexpectedOutput], 'WARNING');
                }
                
                // Log performance metrics
                $endTime = microtime(true);
                $endMemory = memory_get_usage(true);
                $duration = round($endTime - $startTime, 3);
                $memoryDelta = $endMemory - $startMemory;
                
                $performanceContext = [
                    'action' => $request['actionName'],
                    'duration' => $duration,
                    'memory_start' => formatBytes($startMemory),
                    'memory_end' => formatBytes($endMemory),
                    'memory_delta' => formatBytes($memoryDelta)
                ];
                
                // mftpLogPerformance('API_REQUEST', $duration, $performanceContext);
                
                // Log slow requests as potential timeout issues
                if ($duration > 5.0) {
                    // mftpLogTimeout('SLOW_API_REQUEST', $duration, $performanceContext);
                    error_log("MFTP API: Slow request detected - Duration: {$duration}s, Action: {$request['actionName']}");
                }
                
                // Only output the clean JSON response
            print $serializedResponse;
                
            } catch (Exception $e) {
                // Clean the buffer in case of exception
                ob_end_clean();
                
                // Log performance metrics for failed requests
                $endTime = microtime(true);
                $endMemory = memory_get_usage(true);
                $duration = round($endTime - $startTime, 3);
                $memoryDelta = $endMemory - $startMemory;
                
                $performanceContext = [
                    'action' => $request['actionName'] ?? 'unknown',
                    'duration' => $duration,
                    'memory_start' => formatBytes($startMemory),
                    'memory_end' => formatBytes($endMemory),
                    'memory_delta' => formatBytes($memoryDelta),
                    'exception' => get_class($e),
                    'error' => $e->getMessage()
                ];
                
                // mftpLogPerformance('API_REQUEST_FAILED', $duration, $performanceContext);
                // mftpLogTimeout('API_REQUEST_EXCEPTION', $duration, $performanceContext);
                error_log("MFTP API: Request failed - Duration: {$duration}s, Action: {$request['actionName']}, Error: {$e->getMessage()}");
                
                // Log the exception for debugging
                // $debugFile = '/container/logs/apache2/mftp_debug.log';
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] API DEBUG: Exception in marshallRequest: " . $e->getMessage() . "\n", FILE_APPEND);
                // file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] API DEBUG: Exception trace: " . $e->getTraceAsString() . "\n", FILE_APPEND);
                
                // Return a clean JSON error response with detailed error for debugging
                http_response_code(500);
                echo json_encode([
                    'success' => false,
                    'error' => 'Internal server error: ' . $e->getMessage(),
                    'debug' => [
                        'exception_type' => get_class($e),
                        'file' => $e->getFile(),
                        'line' => $e->getLine(),
                        'trace' => $e->getTraceAsString()
                    ]
                ]);
            }
        }
    } catch (Exception $e) {
        // Debug: Log the exception details
        $debugFile = dirname(__FILE__) . '/../logs/mftp_debug.log';
        @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] API DEBUG: Exception caught: " . $e->getMessage() . "\n", FILE_APPEND);
        @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] API DEBUG: Exception type: " . get_class($e) . "\n", FILE_APPEND);
        @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] API DEBUG: Exception trace: " . $e->getTraceAsString() . "\n", FILE_APPEND);
        @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] API DEBUG: Request action: " . ($request['actionName'] ?? 'NOT_SET') . "\n", FILE_APPEND);
        @file_put_contents($debugFile, "[" . date('Y-m-d H:i:s') . "] API DEBUG: Request context: " . json_encode($request['context'] ?? []) . "\n", FILE_APPEND);
        
        if (isset($marshaller)) {
            @$marshaller->disconnect();
        }
        handleExceptionInRequest($e);
    }

    $marshaller->disconnect();

    /**
     * Validate context parameters based on the specific action being performed
     */
    function validateContextForAction($actionName, $context) {
        switch ($actionName) {
            case 'uploadFile':
            case 'uploadFileToNewDirectory':
            case 'uploadArchive':
                if (isset($context['remotePath'])) {
                    $validatedPath = InputValidator::validateFilePath($context['remotePath'], true);
                    // Additional upload-specific validation
                    if (isset($context['localPath'])) {
                        InputValidator::validateFileUpload($context['localPath'], $validatedPath);
                    }
                }
                break;
                
            case 'downloadFile':
            case 'fetchFile':
            case 'getFileContents':
            case 'deleteFile':
                if (isset($context['remotePath'])) {
                    InputValidator::validateFilePath($context['remotePath'], true);
                }
                break;
                
            case 'listDirectory':
            case 'makeDirectory':
            case 'deleteDirectory':
                if (isset($context['remotePath'])) {
                    InputValidator::validateFilePath($context['remotePath'], true);
                }
                break;
                
            case 'rename':
            case 'copy':
                if (isset($context['source'])) {
                    InputValidator::validateFilePath($context['source'], true);
                }
                if (isset($context['destination'])) {
                    InputValidator::validateFilePath($context['destination'], true);
                }
                break;
                
            case 'fetchRemoteFile':
                if (isset($context['source'])) {
                    InputValidator::validateUrl($context['source']);
                }
                if (isset($context['destination'])) {
                    InputValidator::validateFilePath($context['destination'], true);
                }
                break;
                
            case 'changePermissions':
                if (isset($context['remotePath'])) {
                    InputValidator::validateFilePath($context['remotePath'], true);
                }
                if (isset($context['mode'])) {
                    InputValidator::validatePermissionMask($context['mode']);
                }
                // Also support legacy 'permissions' parameter for backward compatibility
                if (isset($context['permissions'])) {
                    InputValidator::validatePermissionMask($context['permissions']);
                }
                break;
                
            case 'setApplicationSettings':
                if (isset($context['applicationSettings'])) {
                    InputValidator::validateApplicationSettings($context['applicationSettings']);
                }
                break;
                
            case 'transferUploadToRemote':
            case 'reserveUploadContext':
                if (isset($context['sessionKey'])) {
                    InputValidator::validateSessionKey($context['sessionKey']);
                }
                break;
                
            case 'extractArchive':
                if (isset($context['fileKey'])) {
                    InputValidator::validateSessionKey($context['fileKey']);
                }
                if (isset($context['extractDirectory'])) {
                    InputValidator::validateFilePath($context['extractDirectory'], true);
                }
                break;
        }
    }
